use std::io::prelude::*;
use std::net::{TcpStream, ToSocketAddrs};
use std::str;
/// A finger client library
/// See https://datatracker.ietf.org/doc/html/rfc1288
/// Be mature.
///
/// Inspired by Elif Batuman's "The Idiot"
/// https://datatracker.ietf.org/doc/html/draft-ietf-uri-url-finger-02

#[derive(Debug)]
pub enum Error {
    InvalidFingerURL,
    HostError,
    IoError(std::io::Error),
    Utf8Error(str::Utf8Error),
}

impl From<str::Utf8Error> for Error {
    fn from(err: str::Utf8Error) -> Error {
        Error::Utf8Error(err)
    }
}

impl From<std::io::Error> for Error {
    fn from(err: std::io::Error) -> Error {
        Error::IoError(err)
    }
}

pub struct Request {
    query: String,
    host: String,
    // TODO
    // timeout: Option<u64>, // seconds
    // max_response_len: Option<u64>,
}

/// From the draft spec:
/// https://datatracker.ietf.org/doc/html/draft-ietf-uri-url-finger-02
pub fn from_url(url: &str) -> Result<Request, Error> {
    if url.starts_with("finger://") {
        let i = url[9..].find("/").unwrap_or(url.len() - 9);
        let host = &url[9..9 + i];
        let query = &url[9 + i..].trim_matches('/');
        return Ok(finger(query, host));
    }
    return Err(Error::InvalidFingerURL);
}

// TODO better host resolving
/// rfc1288 2.2 is ignored, queries are UTF-8 strings
/// Uses sensible defaults for timeout and response len
/// Host can include port. Defaults to 79 if not included
pub fn finger(query: &str, host: &str) -> Request {
    let mut q = String::new();
    q.push_str(query);
    if !q.ends_with("\r\n") {
        q.push_str("\r\n")
    }

    Request {
        query: q.to_owned(),
        host: host.to_owned(),
        // timeout: Some(3),
        // max_response_len: Some(1024 * 10), // 10KB
    }
}
impl Request {
    /// rfc1288 2.2 is ignored, we will allow UTF-8
    pub fn send(self) -> Result<String, Error> {
        let dest: String;
        if !self.host.contains(":") {
            dest = format!("{}:79", self.host);
        } else {
            dest = self.host;
        }
        let server = dest.to_socket_addrs()?.next().ok_or(Error::HostError)?;
        let mut stream = TcpStream::connect(&server)?;

        stream.write(self.query.as_bytes())?;
        let mut line: Vec<u8> = vec![];
        stream.read_to_end(&mut line)?;

        return Ok(str::from_utf8(&line)?.to_owned());
    }
}
