/*! Tools to execute (process) a job */

/*
    Copyright (C) 2022 John Goerzen <jgoerzen@complete.org>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

use anyhow;
use std::ffi::{OsStr, OsString};
use std::io::{copy, Read};
use std::os::unix::ffi::OsStrExt;
use std::process::{Command, ExitStatus, Stdio};
use tracing::*;

/** Low-level code to execute a job.  See [`crate::cmd::cmd_exec`] for
higher-level interfaces. */
pub fn exec_job<T: Read>(
    command: &OsString,
    params: &[OsString],
    env: Vec<(Vec<u8>, Vec<u8>)>,
    payload: Option<T>,
    stdoutput: Stdio,
    stderr: Stdio,
) -> Result<ExitStatus, anyhow::Error> {
    debug!("Preparing to run {:?} with params {:?}", command, params);
    let mut child = Command::new(command)
        .args(params)
        .envs(
            env.iter()
                .map(|(k, v)| (OsStr::from_bytes(k), OsStr::from_bytes(v))),
        )
        .stdin(Stdio::piped())
        .stdout(stdoutput)
        .stderr(stderr)
        .spawn()?;
    debug!("Command PID {} started successfully", child.id());
    let mut payloadbytes = 0;

    if let Some(mut payload) = payload {
        // We have to manually copy because reading from stdin for the header uses
        // an 8K buffer.  Bleh.
        payloadbytes = copy(
            &mut payload,
            &mut child.stdin.take().expect("Missing child stdin fd"),
        )?;
    };

    debug!(
        "Wrote {} payload bytes to child; now waiting for it to exit",
        payloadbytes
    );

    let exitstatus = child.wait()?;
    if exitstatus.success() {
        debug!("Command exited successfully with status {:?}", exitstatus);
    } else {
        error!("Command exited abnormally with status {:?}", exitstatus);
    }

    Ok(exitstatus)
}
