/*! Tools executing jobs */

/*
    Copyright (C) 2022 John Goerzen <jgoerzen@complete.org>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

use crate::cmd::cmd_queue::QueueOptsWithDecoder;
use crate::exec;
use crate::header::CommandTypeV1;
use crate::jobfile;
use crate::jobqueue;
use crate::seqfile;
use anyhow::{anyhow, bail};
use clap::Args;
use std::ffi::OsStr;
use std::ffi::OsString;
use std::fmt;
use std::fmt::Debug;
use std::fs::{remove_file, File};
use std::io::Read;
use std::os::unix::ffi::OsStrExt;
use std::process::{ExitStatus, Stdio};
use std::str::FromStr;
use tracing::*;

/// Option for what to do if an error is encountered while processing a queue
#[derive(Debug)]
pub enum OnError {
    /// Delete the file and mark the job as done; do not return an error code.  If --never-delete
    /// is given, this becomes the same as Leave.
    Delete,

    /// Leave the file and mark the job as done; do not return an error code.
    Leave,

    /// Abort processing with an error code and do not mark the job as done.  It can be retried by a subsequent queue processing command.
    Retry,
}

// consider enum-utils fromstr derivation or others at
// https://crates.io/search?q=enum_utils
impl FromStr for OnError {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Self, anyhow::Error> {
        match s.trim().to_lowercase().as_str() {
            "delete" => Ok(OnError::Delete),
            "leave" => Ok(OnError::Leave),
            "retry" => Ok(OnError::Retry),
            _ => Err(anyhow!(
                "on-error must be 'ignore-and-delete', 'ignore-and-leave', or 'retry'"
            )),
        }
    }
}

impl fmt::Display for OnError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(self, f)
    }
}

/// Option for what to do with output
#[derive(Debug)]
pub enum OutputTo {
    /// Don't capture; use the calling environment's stdout/stderr
    PassBoth,

    /// Write both stdout and stderr to a file
    SaveBoth,
}

// consider enum-utils fromstr derivation or others at
// https://crates.io/search?q=enum_utils
impl FromStr for OutputTo {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Self, anyhow::Error> {
        match s.trim().to_lowercase().as_str() {
            "passboth" => Ok(OutputTo::PassBoth),
            "saveboth" => Ok(OutputTo::SaveBoth),
            _ => Err(anyhow!("output-to must be 'passboth' or 'saveboth'")),
        }
    }
}

impl fmt::Display for OutputTo {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(self, f)
    }
}

#[derive(Debug, Args)]
pub struct ProcessOpts {
    /// Allow the parameters from the job to be appended to the command's
    /// command line (not doing so by default)
    #[clap(long)]
    pub allow_job_params: bool,

    /// Ignore the payload; do not pass it to the command.
    #[clap(long)]
    pub ignore_payload: bool,

    /// The command to run
    #[clap(value_name = "COMMAND")]
    pub command: OsString,

    /// Optional parameters to pass to COMMAND, before those contained
    /// in the request file.
    #[clap(last = true)]
    pub params: Vec<OsString>,
}

#[derive(Debug, Args)]
pub struct QueueProcessOpts {
    #[clap(flatten)]
    pub qdopts: QueueOptsWithDecoder,

    /// What to do with an error.  Can be Delete, Leave, or Retry.
    #[clap(long, value_name="ONERROR", default_value_t = OnError::Retry)]
    pub on_error: OnError,

    /// Never delete processed job files in any circumstance
    #[clap(long)]
    pub never_delete: bool,

    /// Maximum number of jobs to process
    #[clap(long, value_name = "JOBS", short = 'n')]
    pub maxjobs: Option<u64>,

    /// What to do with stdout and stderr of the process.  Can be PassBoth or SaveBoth.
    #[clap(long, value_name="DEST", default_value_t = OutputTo::PassBoth)]
    pub output_to: OutputTo,

    #[clap(flatten)]
    pub processopts: ProcessOpts,
}

/// The result type from running a command.
#[derive(Debug, Eq, PartialEq)]
pub enum CmdResult {
    Success,
    Failure,
    ExitResult(ExitStatus),
}

impl CmdResult {
    fn success(&self) -> bool {
        match self {
            CmdResult::Success => true,
            CmdResult::Failure => false,
            CmdResult::ExitResult(es) => es.success(),
        }
    }
}

/// Process some input to execute, optionally writing the output to a given File.
pub fn cmd_exec_generic<R: Read>(
    opts: &ProcessOpts,
    mut input: R,
    inputfilename: Option<&OsString>,
    queuedir: Option<&OsString>,
    output: Option<File>,
) -> Result<CmdResult, anyhow::Error> {
    let meta = jobfile::read_jobfile_header(&mut input)?;
    let input = if opts.ignore_payload {
        // Just to be really clear, we will not be processing input
        // in this case!
        std::mem::drop(input);
        None
    } else {
        Some(input)
    };

    match meta.cmd_type {
        CommandTypeV1::NOP => Ok(CmdResult::Success),
        CommandTypeV1::Fail => Ok(CmdResult::Failure),
        CommandTypeV1::Command(ref copts) => {
            let mut params = opts.params.clone();
            if opts.allow_job_params {
                let jobparams: Vec<OsString> = copts
                    .params
                    .iter()
                    .map(|x| OsStr::from_bytes(x).to_os_string())
                    .collect();

                params.extend(jobparams);
            } else if !copts.params.is_empty() {
                warn!("Parameters given in job ignored because --allow-job-params not given");
            }

            debug!(
                "Proparing to execute job {} from {:?}",
                meta.seq, &inputfilename
            );
            let env = meta.get_envvars(inputfilename, queuedir);

            if let Some(outfile) = output {
                Ok(CmdResult::ExitResult(exec::exec_job(
                    &opts.command,
                    &params,
                    env,
                    input,
                    Stdio::from(outfile.try_clone()?),
                    Stdio::from(outfile),
                )?))
            } else {
                Ok(CmdResult::ExitResult(exec::exec_job(
                    &opts.command,
                    &params,
                    env,
                    input,
                    Stdio::inherit(),
                    Stdio::inherit(),
                )?))
            }
        }
    }
}

/// Execute a job using a job file piped in on stdin.  This is just a very
/// thin wrapper around [`cmd_exec_generic`].
pub fn cmd_execstdin(opts: &ProcessOpts) -> Result<(), anyhow::Error> {
    let res = cmd_exec_generic(opts, std::io::stdin(), None, None, None)?;

    if res.success() {
        Ok(())
    } else {
        bail!("Command exited with non-success status {:?}", res);
    }
}

/// The main worker of the program; it executes a queue.
pub fn cmd_execqueue(opts: &QueueProcessOpts) -> Result<(), anyhow::Error> {
    let seqfn = jobqueue::get_seqfile(&opts.qdopts.qopts.queuedir);
    let mut lock = seqfile::prepare_seqfile_lock(&seqfn)?;

    let mut seqf = seqfile::SeqFile::open(&seqfn, &mut lock)?;

    let mut qmap = jobqueue::scanqueue_map(&opts.qdopts.qopts.queuedir, &opts.qdopts.decoder)?;
    let mut jobsprocessed = 0;
    let maxjobs = opts.maxjobs.unwrap_or(u64::MAX);
    while let Some((filename, _meta)) = qmap.remove(&seqf.get_next()) {
        if jobsprocessed >= maxjobs {
            debug!(
                "Stopping processing as requested maximum jobs processed {} has been hit",
                maxjobs
            );
            return Ok(());
        }
        jobsprocessed += 1;

        let mut inputinfo =
            jobqueue::queue_openjob(&opts.qdopts.qopts.queuedir, &filename, &opts.qdopts.decoder)?;
        let inhandle = inputinfo.reader.take().unwrap();
        let fullfilename = jobqueue::queue_genfilename(&opts.qdopts.qopts.queuedir, &filename);
        let res = match opts.output_to {
            OutputTo::PassBoth => cmd_exec_generic(
                &opts.processopts,
                inhandle,
                Some(&filename.clone()),
                Some(&opts.qdopts.qopts.queuedir),
                None,
            )?,
            OutputTo::SaveBoth => {
                let mut savefilename = OsString::from(&fullfilename);
                savefilename.push(".out");
                trace!(
                    "Writing stdout and stderr to {:?} per request",
                    savefilename
                );
                let savefile = File::create(savefilename)?;
                cmd_exec_generic(
                    &opts.processopts,
                    inhandle,
                    Some(&filename.clone()),
                    Some(&opts.qdopts.qopts.queuedir),
                    Some(savefile),
                )?
            }
        };
        match (res.success(), &opts.on_error) {
            (true, _) | (false, OnError::Delete) => {
                seqf.increment()?;
                if !opts.never_delete {
                    debug!("Deleting {:?}", fullfilename);
                    remove_file(fullfilename)?;
                }
            }
            (false, OnError::Retry) => bail!(
                "Aborting processing due to exit status {:?} from command",
                res
            ),
            (false, OnError::Leave) => {
                seqf.increment()?;
            }
        }
    }
    debug!(
        "Stopping processing as there is no job {} to process next",
        seqf.get_next()
    );

    Ok(())
}
