use std::{iter::Iterator, str::FromStr};
use serde_derive::{Deserialize, Serialize};
use anyhow::Result;
use lazy_static::lazy_static;
use std::sync::Arc;
use serde_json::json;
use tide::Request;
use http_types::headers::HeaderValue;
use tide::security::{CorsMiddleware, Origin};
use tide_acme::{AcmeConfig, TideRustlsExt};
use s3compat::{Bucket, BucketConfiguration};
use s3compat::creds::Credentials;
use async_std::io::Cursor;
extern crate biscuit_auth as biscuit;
use biscuit::{crypto::PublicKey, token::Biscuit};
use surf::http::{Url, Method, Mime};

lazy_static! {
    static ref DB : Arc<rocksdb::DB> = {

        let prefix_extractor = rocksdb::SliceTransform::create_fixed_prefix(5);

        let mut opts = rocksdb::Options::default();
        opts.create_if_missing(true);
        opts.set_prefix_extractor(prefix_extractor);

        let configure = env_var_config();
        let db = rocksdb::DB::open(&opts, configure.db).unwrap();
        Arc::new(db)
    };

    static ref JWTS : Arc<lockfree::map::Map<String, (i64, String)>> = {
        Arc::new(lockfree::map::Map::new())
    };

    static ref PROVIDERS : Arc<lockfree::map::Map<String, Vec<String>>> = {
        let col : lockfree::map::Map<String, Vec<String>> = lockfree::map::Map::new();
        col.insert("aws".to_string(), vec![
            "us-east-1".to_string(), 
            "us-east-2".to_string(), 
            "us-west-1".to_string(), 
            "us-west-2".to_string(), 
            "ca-central-1".to_string(), 
            "ap-south-1".to_string(), 
            "ap-northeast-1".to_string(), 
            "ap-northeast-2".to_string(), 
            "ap-northeast-3".to_string(), 
            "cn-north-1".to_string(), 
            "cn-northwest-1".to_string(), 
            "eu-north-1".to_string(),
            "eu-central-1".to_string(),
            "eu-west-1".to_string(),
            "eu-west-2".to_string(),
            "eu-west-3".to_string(),
            "me-south-1".to_string(),
            "sa-east-1".to_string(),
        ]);
        col.insert("do".to_string(), vec![
            "nyc3".to_string(),
            "ams3".to_string(),
            "spg1".to_string(),
            "fra1".to_string(),
        ]);
        col.insert("yandex".to_string(), vec!["ru-central1".to_string()]);
        col.insert("wasabi".to_string(), vec![
            "wa-us-east-1".to_string(), 
            "wa-us-east-2".to_string(), 
            "wa-us-west-1".to_string(), 
            "wa-eu-central-1".to_string()
        ]);
        col.insert("vultr".to_string(), vec![
            "ewr1".to_string(),
        ]);
        Arc::new(col)
    };
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct Object {
    pub last_modified: String,
    pub e_tag: String,
    pub storage_class: String,
    pub key: String,
    pub size: u64,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct VerifyResponse {
    key: Vec<u8>,
    token: Vec<u8>,
    expiry: i64,
    username: String,
    scopes: Vec<String>,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct ObjectForm {
    pub region: String,
    pub bucket: String,
    pub object_path: String,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct BucketForm {
    pub region: String,
    pub bucket: String,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct ListForm {
    pub region: String,
    pub bucket: String,
    pub prefix: String,
    pub delimiter: Option<String>,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct Provider {
    pub id: uuid::Uuid,
    pub name: String,
    pub secret_key: String,
    pub access_key: String,
    pub username: String,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct ProviderForm {
    pub name: String,
    pub secret_key: String,
    pub access_key: String,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct UploadForm {
    pub object_path: String,
    pub bucket: String,
    pub region: String,
    pub file: String,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct DeleteForm {
    pub object_path: String,
    pub bucket: String,
    pub region: String,
}

#[derive(Deserialize, Debug, Clone)]
pub struct EnvVarConfig {
  pub port: u16,
  pub origin: String,
  pub db: String,
  pub secure: bool,
  pub certs: String,
  pub domain: String,
  pub broker: String,
  pub auto_cert: bool,
  pub key_path: String,
  pub cert_path: String,
}

fn env_var_config() -> EnvVarConfig {
 
    let mut port : u16 = 9999;
    let mut secure = false;
    let mut auto_cert = true;
    let mut origin = "*".to_string();
    let mut db: String = "db".to_string();
    let mut certs = "certs".to_string();
    let mut domain = "localhost".to_string();
    let mut broker = "http://localhost:8080".to_string();
    let mut key_path = "certs/private_key.pem".to_string();
    let mut cert_path = "certs/chain.pem".to_string();

    let _ : Vec<String> = go_flag::parse(|flags| {
        flags.add_flag("port", &mut port);
        flags.add_flag("origin", &mut origin);
        flags.add_flag("secure", &mut secure);
        flags.add_flag("db", &mut db);
        flags.add_flag("domain", &mut domain);
        flags.add_flag("certs", &mut certs);
        flags.add_flag("broker", &mut broker);
        flags.add_flag("auto_cert", &mut auto_cert);
        flags.add_flag("key_path", &mut key_path);
        flags.add_flag("cert_path", &mut cert_path);
    });

    EnvVarConfig{port, origin, secure, domain, certs, db, broker, auto_cert, cert_path, key_path}
}

async fn jwt_verify(token: String, operation: String) -> Result<(bool, String)> {

    let configure = env_var_config();

    let mut parts = token.split(" ");
    let auth_type = parts.next().unwrap();
    if auth_type == "Bearer" {
        let token = parts.next().unwrap();
        let time = nippy::get_unix_ntp_time().await?;

        match JWTS.get(&token.to_string()) {
            Some(entry) => {
                let (expiry, username) = entry.val();
                if expiry > &time {
                    return Ok((true, username.to_string()));
                } else {
                    return Ok((false, "".to_string()));
                }
            },
            None => {}
        }

        let broker_url = format!("{}/verify", configure.broker);
        let auth = format!("Bearer {}", token);
        let url = Url::parse(&broker_url)?;
        let mime = Mime::from_str("application/json").unwrap();
        let request = surf::Request::builder(Method::Get, url.clone())
        .header("authorization", &auth)
        .content_type(mime)
        .build();

        let mut res = surf::client().send(request).await.unwrap();
        if res.status() == 200 {
            let j: VerifyResponse = res.body_json().await.unwrap();
            let bis = Biscuit::from(&j.token)?;

            let public_key = PublicKey::from_bytes(&j.key).unwrap();
            let mut v1 = bis.verify(public_key)?;
            let f = format!("allow if right(#authority, \"files\", #{})", "full");
            let r = f.as_ref();
            v1.add_policy(r)?;
            let f = format!("allow if right(#authority, \"files\", #{})", operation);
            let r = f.as_ref();
            v1.add_policy(r)?;
            v1.deny()?;
            v1.verify()?;

            JWTS.insert(token.to_string(), (j.expiry, j.username.clone()));
            
            Ok((true, j.username.clone()))
        } else {
            Ok((false, "".to_string()))
        }
    } else {
        Ok((false, "".to_string()))
    }
}

fn replace(key: String, value: Vec<u8>) -> Result<()> {
    DB.put(key.clone(), value.clone())?;
    Ok(())
}

fn get_providers() -> Result<Vec<Provider>> {
    let prefix = "providers".as_bytes();
    let i = DB.prefix_iterator(prefix);
    let res : Vec<Provider> = i.map(|(_, v)| {
        let data: Provider = rmp_serde::from_read_ref(&v).unwrap();
        data
    }).collect();
    Ok(res)
}

fn get_provider_by_name(provider_name: String, username: String) -> Result<Option<Provider>> {
    let providers = get_providers()?;
    Ok(providers.into_iter().filter(|provider| provider.name == provider_name && provider.username == username).last())
}

fn puts_provider(provider: Provider) -> Result<()> {
    let key = format!("providers_{}", provider.id);
    let value = rmp_serde::to_vec_named(&provider)?;
    replace(key, value)?;
    Ok(())
}

fn is_provider_valid(provider_name: String) -> Result<bool> {
    for rg in PROVIDERS.iter() {
        let provider = rg.key().to_string();
        if provider == provider_name {
            return Ok(true);
        }
    }
    Ok(false)
}

fn provider_put(provider_form: ProviderForm, username: String) -> Result<bool> {
    let check = is_provider_valid(provider_form.clone().name)?;
    if !check {
        return Ok(false);
    }

    match get_provider_by_name(provider_form.clone().name, username.clone())? {
        Some(mut provider) => {
            provider.access_key = provider_form.access_key;
            provider.secret_key = provider_form.secret_key;
            puts_provider(provider)?;
            Ok(true)
        },
        None => {
            let id = uuid::Uuid::new_v4();
            let provider = Provider{
                id,
                name: provider_form.name,
                access_key: provider_form.access_key,
                secret_key: provider_form.secret_key,
                username,
            };
            puts_provider(provider)?;
            Ok(true)
        }
    }
}

async fn health(_: Request<()>) -> tide::Result {
    Ok(tide::Response::builder(200).header("content-type", "application/json").build())
}

async fn provider(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "provider".to_string()).await?;
            if check {

                let r =  req.body_string().await?;
                let provider_form : ProviderForm = serde_json::from_str(&r)?;
                let check = provider_put(provider_form, username)?;

                if check {
                    Ok(tide::Response::builder(200).header("content-type", "application/json").build())
                } else {
                    let j = json!({"error": "provider is invalid"});
                    println!("{}", j);
                    Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build())
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

async fn get_url_object(req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "get_object".to_string()).await?;
            if check {

                let region = req.param("region")?.to_string();
                let bucket = req.param("bucket")?.to_string();
                let object_path = req.param("object_path")?.to_string();

                let mut provider_maybe_name: Option<String> = None;
                for rg in PROVIDERS.iter() {
                    if rg.val().contains(&region) {
                        provider_maybe_name = Some(rg.key().to_string());
                    }
                }

                match provider_maybe_name {
                    Some(provider_name) => {
                        let provider_value = get_provider_by_name(provider_name, username)?;

                        match provider_value {
                            Some(provider) => {
                                let bucket_name = bucket;
                                let region = region.parse()?;
                            
                                let access_key = provider.access_key;
                                let secret_key = provider.secret_key;
                                let credentials = Credentials::new(Some(&access_key), Some(&secret_key), None, None, None)?;
                            
                                let bucket = Bucket::new(&bucket_name, region, credentials)?;

                                let (data, code) = bucket.get_object(object_path.clone()).await?;

                                let content_type;
                                match infer::get(&data) {
                                    Some(kind) => {
                                        content_type = kind.mime_type();
                                    },
                                    None => {
                                        content_type = "application/octet-stream";
                                    }
                                }

                                return Ok(tide::Response::builder(code).body(data)
                                    .header("content-type", content_type)
                                    .build());
                            },
                            None => {
                                let j = json!({"error": "region not valid for provider"});
                                println!("{}", j);
                                return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                            }
                        }
                    },
                    None => {
                        let j = json!({"error": "region not valid for provider"});
                        println!("{}", j);
                        return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                    }
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

async fn get_object(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "get_object".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let object_form : ObjectForm = serde_json::from_str(&r)?;

                let mut provider_maybe_name: Option<String> = None;
                for rg in PROVIDERS.iter() {
                    if rg.val().contains(&object_form.region) {
                        provider_maybe_name = Some(rg.key().to_string());
                    }
                }

                match provider_maybe_name {
                    Some(provider_name) => {
                        let provider_value = get_provider_by_name(provider_name, username)?;

                        match provider_value {
                            Some(provider) => {
                                let bucket_name = object_form.bucket;
                                let region = object_form.region.parse()?;
                            
                                let access_key = provider.access_key;
                                let secret_key = provider.secret_key;
                                let credentials = Credentials::new(Some(&access_key), Some(&secret_key), None, None, None)?;
                            
                                let bucket = Bucket::new(&bucket_name, region, credentials)?;

                                let (data, code) = bucket.get_object(object_form.object_path.clone()).await?;

                                let content_type;
                                match infer::get(&data) {
                                    Some(kind) => {
                                        content_type = kind.mime_type();
                                    },
                                    None => {
                                        content_type = "application/octet-stream";
                                    }
                                }

                                return Ok(tide::Response::builder(code).body(data)
                                    .header("content-type", content_type)
                                    .build());
                            },
                            None => {
                                let j = json!({"error": "region not valid for provider"});
                                println!("{}", j);
                                return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                            }
                        }
                    },
                    None => {
                        let j = json!({"error": "region not valid for provider"});
                        println!("{}", j);
                        return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                    }
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

async fn put_object(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "put_object".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let upload_form : UploadForm = serde_json::from_str(&r)?;

                let mut provider_maybe_name: Option<String> = None;
                for rg in PROVIDERS.iter() {
                    if rg.val().contains(&upload_form.region) {
                        provider_maybe_name = Some(rg.key().to_string());
                    }
                }

                match provider_maybe_name {
                    Some(provider_name) => {
                        let provider_value = get_provider_by_name(provider_name, username)?;

                        match provider_value {
                            Some(provider) => {
                                let bucket_name = upload_form.bucket;
                                let region = upload_form.region.parse()?;
                            
                                let access_key = provider.access_key;
                                let secret_key = provider.secret_key;
                                let credentials = Credentials::new(Some(&access_key), Some(&secret_key), None, None, None)?;
                            
                                let bucket = Bucket::new(&bucket_name, region, credentials)?;

                                let mut file = Cursor::new(base64::decode(upload_form.file)?);

                                let code = bucket.put_object_stream(&mut file, upload_form.object_path).await?;

                                return Ok(tide::Response::builder(code).header("content-type", "application/json").build());
                            },
                            None => {
                                let j = json!({"error": "region not valid for provider"});
                                println!("{}", j);
                                return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                            }
                        }
                    },
                    None => {
                        let j = json!({"error": "region not valid for provider"});
                        println!("{}", j);
                        return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                    }
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

async fn delete_object(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "delete_object".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let delete_form : DeleteForm = serde_json::from_str(&r)?;

                let mut provider_maybe_name: Option<String> = None;
                for rg in PROVIDERS.iter() {
                    if rg.val().contains(&delete_form.region) {
                        provider_maybe_name = Some(rg.key().to_string());
                    }
                }

                match provider_maybe_name {
                    Some(provider_name) => {
                        let provider_value  = get_provider_by_name(provider_name, username)?;

                        match provider_value {
                            Some(provider) => {
                                let bucket_name = delete_form.bucket;
                                let region = delete_form.region.parse()?;
                            
                                let access_key = provider.access_key;
                                let secret_key = provider.secret_key;
                                let credentials = Credentials::new(Some(&access_key), Some(&secret_key), None, None, None)?;
                            
                                let bucket = Bucket::new(&bucket_name, region, credentials)?;

                                let (_, code) = bucket.delete_object(delete_form.object_path).await?;

                                return Ok(tide::Response::builder(code).header("content-type", "application/json").build());
                            },
                            None => {
                                let j = json!({"error": "region not valid for provider"});
                                println!("{}", j);
                                return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                            }
                        }
                    },
                    None => {
                        let j = json!({"error": "region not valid for provider"});
                        println!("{}", j);
                        return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                    }
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

async fn create_bucket(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "create_bucket".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let bucket_form : BucketForm = serde_json::from_str(&r)?;

                let mut provider_maybe_name: Option<String> = None;
                for rg in PROVIDERS.iter() {
                    if rg.val().contains(&bucket_form.region) {
                        provider_maybe_name = Some(rg.key().to_string());
                    }
                }

                match provider_maybe_name {
                    Some(provider_name) => {
                        let provider_value  = get_provider_by_name(provider_name, username)?;

                        match provider_value {
                            Some(provider) => {
                                let bucket_name = bucket_form.bucket;
                                let region = bucket_form.region.parse()?;
                            
                                let access_key = provider.access_key;
                                let secret_key = provider.secret_key;
                                let credentials = Credentials::new(Some(&access_key), Some(&secret_key), None, None, None)?;
                                
                                let config = BucketConfiguration::default();
                                let bucket_response = Bucket::create(&bucket_name, region, credentials, config).await?;

                                if bucket_response.response_code == 200 {
                                    return Ok(tide::Response::builder(200).header("content-type", "application/json").build());
                                } else {
                                    let j = json!({"error": bucket_response.response_text}).to_string();
                                    println!("{}", j);
                                    return Ok(tide::Response::builder(bucket_response.response_code).body(j).header("content-type", "application/json").build());
                                }
                            },
                            None => {
                                let j = json!({"error": "region not valid for provider"});
                                println!("{}", j);
                                return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                            }
                        }
                    },
                    None => {
                        let j = json!({"error": "region not valid for provider"});
                        println!("{}", j);
                        return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                    }
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

async fn delete_bucket(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "delete_bucket".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let bucket_form : BucketForm = serde_json::from_str(&r)?;

                let mut provider_maybe_name: Option<String> = None;
                for rg in PROVIDERS.iter() {
                    if rg.val().contains(&bucket_form.region) {
                        provider_maybe_name = Some(rg.key().to_string());
                    }
                }

                match provider_maybe_name {
                    Some(provider_name) => {
                        let provider_value  = get_provider_by_name(provider_name, username)?;

                        match provider_value {
                            Some(provider) => {
                                let bucket_name = bucket_form.bucket;
                                let region = bucket_form.region.parse()?;
                            
                                let access_key = provider.access_key;
                                let secret_key = provider.secret_key;
                                let credentials = Credentials::new(Some(&access_key), Some(&secret_key), None, None, None)?;
                                
                                let bucket = Bucket::new(&bucket_name, region, credentials)?;

                                let code = bucket.delete().await?;

                                return Ok(tide::Response::builder(code).header("content-type", "application/json").build());
                            },
                            None => {
                                let j = json!({"error": "region not valid for provider"});
                                println!("{}", j);
                                return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                            }
                        }
                    },
                    None => {
                        let j = json!({"error": "region not valid for provider"});
                        println!("{}", j);
                        return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                    }
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

async fn list_bucket(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "list_bucket".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let list_form : ListForm = serde_json::from_str(&r)?;

                let mut provider_maybe_name: Option<String> = None;
                for rg in PROVIDERS.iter() {
                    if rg.val().contains(&list_form.region) {
                        provider_maybe_name = Some(rg.key().to_string());
                    }
                }

                match provider_maybe_name {
                    Some(provider_name) => {
                        let provider_value  = get_provider_by_name(provider_name, username)?;

                        match provider_value {
                            Some(provider) => {
                                let bucket_name = list_form.bucket;
                                let region = list_form.region.parse()?;
                            
                                let access_key = provider.access_key;
                                let secret_key = provider.secret_key;
                                let credentials = Credentials::new(Some(&access_key), Some(&secret_key), None, None, None)?;
                                
                                let bucket = Bucket::new(&bucket_name, region, credentials)?;

                                let mut objects: Vec<Object> = Vec::new();
                                
                                let results = bucket.list(list_form.prefix, list_form.delimiter).await?;

                                for result in results {
                                    for object in result.contents {
                                        let obj = Object {
                                            last_modified: object.last_modified,
                                            e_tag: object.e_tag,
                                            storage_class: object.storage_class,
                                            key: object.key,
                                            size: object.size,
                                        };
                                        objects.push(obj);
                                    }
                                }

                                let j = json!({"objects": objects}).to_string();
                                println!("{}", j);
                                return Ok(tide::Response::builder(200).body(j).header("content-type", "application/json").build());
                            },
                            None => {
                                let j = json!({"error": "region not valid for provider"});
                                println!("{}", j);
                                return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                            }
                        }
                    },
                    None => {
                        let j = json!({"error": "region not valid for provider"});
                        println!("{}", j);
                        return Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build());
                    }
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => {
            Ok(tide::Response::builder(401).header("content-type", "application/json").build())
        }
    }
}

#[async_std::main]
async fn main() -> tide::Result<()> {

    let configure = env_var_config();

    let cors = CorsMiddleware::new()
    .allow_methods("GET, POST, OPTIONS".parse::<HeaderValue>().unwrap())
    .allow_headers("authorization".parse::<HeaderValue>().unwrap())
    .allow_origin(Origin::from(configure.origin))
    .allow_credentials(false);
    
    let mut app = tide::new();
    app.with(driftwood::DevLogger);
    app.with(cors);
    app.at("/").get(health);
    app.at("/").head(health);
    app.at("/provider").post(provider);
    app.at("/get").post(get_object);
    app.at("/put").post(put_object);
    app.at("/del").post(delete_object);
    app.at("/create_bucket").post(create_bucket);
    app.at("/delete_bucket").post(delete_bucket);
    app.at("/list_bucket").post(list_bucket);
    app.at("/files/:region/:bucket/:object_path").get(get_url_object);

    let ip = format!("0.0.0.0:{}", configure.port);

    if configure.secure && configure.auto_cert {
        app.listen(
            tide_rustls::TlsListener::build().addrs("0.0.0.0:443").acme(
                AcmeConfig::new()
                    .domains(vec![configure.domain])
                    .cache_dir(configure.certs)
                    .production(),
            ),
        )
        .await?;
    } else if configure.secure && !configure.auto_cert {
        app.listen(
            tide_rustls::TlsListener::build()
            .addrs("0.0.0.0:443")
            .cert(configure.cert_path)
            .key(configure.key_path)
        )
        .await?;
    } else {
        app.listen(ip).await?;
    }

    Ok(())
}