//! A [BlockDevice] that is an Abstraction Over Any Given [BlockDevice].
//! To Add A New [BlockDevice], 

use alloc::{format, string, vec};
use vec::Vec;
use string::*;
use crate::{FsResult, block::{self, BLOCK_SIZE, BlockAddr, DISK_SIZE}};
use lazy_static::lazy_static;
use spin::Mutex;

lazy_static! {
    static ref DEVICE_0: Mutex<BlockDevice> = Mutex::new(
        BlockDevice::MemBlockDevice(MemDevice::new(DISK_SIZE)
    ));
    static ref DEVICE_1: Mutex<BlockDevice> = Mutex::new(
        BlockDevice::MemBlockDevice(MemDevice::new(DISK_SIZE)
    ));
    static ref DEVICE_2: Mutex<BlockDevice> = Mutex::new(
        BlockDevice::MemBlockDevice(MemDevice::new(DISK_SIZE)
    ));
    static ref DEVICE_3: Mutex<BlockDevice> = Mutex::new(
        BlockDevice::MemBlockDevice(MemDevice::new(DISK_SIZE)
    ));
}

/// Set BlockDevice 0
pub fn set_dev0(dev: BlockDevice) {
    *DEVICE_0.lock() = dev
}

/// Get the Mutex Holding BlockDevice 0
pub fn dev0() -> &'static Mutex<BlockDevice> {
    &DEVICE_0
}
/// Set BlockDevice 1
pub fn set_dev1(dev: BlockDevice) {
    *DEVICE_1.lock() = dev
}

/// Get the Mutex Holding BlockDevice 1
pub fn dev1() -> &'static Mutex<BlockDevice> {
    &DEVICE_1
}

/// Set BlockDevice 2
pub fn set_dev2(dev: BlockDevice) {
    *DEVICE_2.lock() = dev
}

/// Get the Mutex Holding BlockDevice 2
pub fn dev2() -> &'static Mutex<BlockDevice> {
    &DEVICE_2
}

/// Set BlockDevice 3
pub fn set_dev3(dev: BlockDevice) {
    *DEVICE_3.lock() = dev
}
/// Get the Mutex Holding BlockDevice 3
pub fn dev3() -> &'static Mutex<BlockDevice> {
    &DEVICE_3
}

//TODO(George): Add the 'indentify' function, to get Innformation about the underlying Device.
// ie, Disk Size, Name, Mount point.
/// Represents An Arbitrary [BlockDevice]'s Operations
pub trait BlockDeviceIO {
    /// Read A Single Block From A [BlockDevice]
    fn read(&mut self, addr: BlockAddr, buf: &mut [u8]) -> FsResult<()>;
    /// Write A Single Block Into A [BlockDevice]
    fn write(&mut self, addr: BlockAddr, buf: &[u8]) -> FsResult<()>;
}


/// Used To Define Any [BlockDeviceIO] implementation
pub struct DeviceVTable {
    read: fn(addr: BlockAddr, buf: &mut [u8]) -> FsResult<()>,
    write: fn(addr: BlockAddr, buf: &[u8]) -> FsResult<()>,
}

/// Wraps A Single BlockDevice.
pub enum BlockDevice {
    /// A In-Memory [BlockDevice]
    MemBlockDevice(MemDevice),

    /// A Generic [BlockDevice]
    GenericBlockDevice(DeviceVTable)
}
/// A In-Memory [BlockDevice]
pub struct MemDevice {
    disk: Vec<[u8; block::BLOCK_SIZE]>,
}

impl BlockDeviceIO for BlockDevice {
    fn read(&mut self, addr: BlockAddr, buf: &mut [u8]) -> FsResult<()> {
        match self {
            BlockDevice::MemBlockDevice(dev) => dev.read(addr, buf),
            BlockDevice::GenericBlockDevice(table) => (table.read)(addr, buf),
        }
    }

    fn write(&mut self, addr: BlockAddr, buf: &[u8]) -> FsResult<()> {
        match self {
            BlockDevice::MemBlockDevice(dev) => dev.write(addr, buf),
            BlockDevice::GenericBlockDevice(table) => (table.write)(addr, buf),
        }
    }
}

impl BlockDeviceIO for MemDevice {
    fn read(&mut self, addr: BlockAddr, buf: &mut [u8]) -> FsResult<()> {
        if addr as usize >= self.disk.len() {return Err(format!("Block {} is out of range. (max: {})", addr, self.disk.len()))};
        if buf.len() != block::BLOCK_SIZE {
            return Err(format!("Bad Block Size, Expected {}, Got {}.", block::BLOCK_SIZE, buf.len()));
        };
        for index in 0..BLOCK_SIZE {
            buf[index] = self.disk[addr as usize][index];
        };
        Ok(())
    }

    fn write(&mut self, addr: BlockAddr, buf: &[u8]) -> FsResult<()> {
        if addr as usize >= self.disk.len() {return Err("Block Out Of Range.".to_string())};
        if buf.len() != block::BLOCK_SIZE {
            return Err(format!("Bad Block Size, Expected {}, Got {}.", block::BLOCK_SIZE, buf.len()));
        };
        for index in 0..BLOCK_SIZE {
            self.disk[addr as usize][index] = buf[index];
        };

        Ok(())
    }
}

impl MemDevice {
    /// Create A New Memory Device With The Given Capacity.
    pub fn new(capacity: usize) -> Self {
        Self {
            disk: vec![[0; BLOCK_SIZE]; capacity],
        }
    }
}
