//! Represents A Single Block on a block Device
//! Note: At the moment, uses Device 0.

use core::fmt::Debug;

use crate::{bitmap::{Bitmap}, block_device::{self, BlockDeviceIO}};
use alloc::vec::Vec;
use bytes::{Buf, BufMut, Bytes, BytesMut};

/// Size Of One Block, In Bytes.
pub const BLOCK_SIZE: usize = 512;

/// The Address Of A Single Block.
pub type BlockAddr = u32;


//TODO(George): Get Disk Size From the block Device
/// How Large Is The Disk (128MB)
pub const DISK_SIZE: usize = (1024 * 1024 * 128) / BLOCK_SIZE;
/// Where in the block does the data start?
pub const DATA_START: usize = core::mem::size_of::<BlockAddr>();
/// How Large is the usable data area in a block?
pub const DATA_SIZE: usize = BLOCK_SIZE - DATA_START;
/// How Many Blocks Are Writable on a Disk.
pub const MAX_BLOCKS: usize = DISK_SIZE as usize;
#[derive(Clone, Copy)]
/// Represents A Block On A Disk. Holds A [BlockAddr] and a [u8; 512]
pub struct Block {
    addr: BlockAddr,
    data: [u8; BLOCK_SIZE],
}

impl Block {
    /// Collect all linked blocks into the given vector.
    pub fn collect_into_vec(&self, buffer: &mut Vec<Block>) {
        buffer.push(*self);
        if let Some(next) = self.next() {
            next.collect_into_vec(buffer)
        }
    }

    /// The Address Of The Block
    pub fn addr(&self) -> BlockAddr {
        self.addr
    }

    fn new(addr: BlockAddr, data: [u8; BLOCK_SIZE]) -> Self {
        Self {
            addr,
            data
        }
    }

    /// Load A Block from Device 0.
    pub fn read(addr: BlockAddr) -> Self {
        // #[cfg(all(feature = "std", feature = "debug"))]
        // println!("Reading Block #{}...", addr);
        let mut buf = [0; BLOCK_SIZE];
        let mut dev0= block_device::dev0().lock();
        dev0.read(addr, &mut buf).expect("Device Read Failed...");
        Block::new(addr, buf)
    }

    /// Write the block out to the disk
    pub fn write(&self) {
        let mut dev0= block_device::dev0().lock();
        //#[cfg(all(feature = "std", feature = "debug"))]
        //println!("Writing Block #{}...", self.addr);
        dev0.write(self.addr, &self.data).expect("Device Read Failed...");
    }

    /// Attempt To Allocate A New Block. Returns [None] on failure.
    pub fn alloc() -> Option<Block> {
        if let Some(addr) = Bitmap::alloc_next_free() {
            #[cfg(feature = "std")]
            println!("Allocated Block 0x{:06x}", addr);
            Some(Block::read(addr))
        } else {
            None
        }
    }

    /// Free the current block.
    pub fn free(&self) {
        Bitmap::free(self.addr())
    }

    /// Get an Immutable Reference To the Underlying Block Data
    pub fn data_ref(&self) -> &[u8] {
        &self.data
    }

    /// Get an Mutable Reference To the Underlying Block Data
    pub fn data_mut(&mut self) -> &mut [u8] {
        &mut self.data
    }

    /// Get The Next Block in the List. Returns [None] if there is no Next block.
    pub fn next(&self) -> Option<Block> {
        let mut buffer = Bytes::copy_from_slice(self.data_ref());
        let next = buffer.get_u32_le();
        if next == 0 {return None;};
        return Some(Block::read(next))
    }

    /// Set The Current block's next block to the given [Block]
    pub fn set_next(&mut self, block: Block) {
        let mut buffer = BytesMut::from(self.data_ref());
        buffer.put_u32_le(block.addr());
        buffer.copy_to_slice(self.data_mut());
        self.write();
    }

    /// Get The Raw [BlockAddr] of the next Block. Returns 0 if there is no next Block.
    pub fn next_addr(&self) -> BlockAddr {
        let mut buffer = Bytes::copy_from_slice(self.data_ref());
        let next = buffer.get_u32();
        return next;
    }


}

impl Debug for Block {
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        writeln!(f, "Block @ 0x{:08x}: ", self.addr)?;
        writeln!(f, "Data: ")?;
        for (index, byte) in self.data.iter().enumerate() {
            write!(f,"{:0X}, ", byte)?;
            if index > 0 && index % 16 == 0 {
                writeln!(f)?;
            }
        }
        Ok(())

    }
}



