//! Tracks Allocated & Free [Block]s on a [BlockDevice]
use alloc::string::String;
use bit_field::BitField;
use core::fmt::Write;
#[allow(unused)]
use crate::block_device::BlockDevice;


use crate::block::*;

/// The First Bitmap On Disk.
pub const BITMAP_ADDR: BlockAddr =  8;
/// The First Data Block On The Disk.
pub const DATA_ADDR: BlockAddr = BITMAP_ADDR + BITMAP_COUNT as u32;
/// The Size Of One Bitmap.
pub const BITMAP_SIZE: usize = BLOCK_SIZE;
/// The Amount Of Blocks in a single bitmap.
pub const BLOCKS_PER_BITMAP: usize = BLOCK_SIZE * 8; 
/// How Many Bitmaps Are There.
pub const BITMAP_COUNT: usize = MAX_BLOCKS / BLOCKS_PER_BITMAP;

/// A Single Bitmap Block.
pub struct Bitmap {_private: ()}


impl Bitmap {
    /// Which Bitmap is needed? block = bitmap_addr + ((addr - data_addr) / bitmap_size / 8) 
    pub fn block_addr(addr: BlockAddr) -> BlockAddr {
        let size = BITMAP_SIZE as BlockAddr ;
        let i = addr - DATA_ADDR;
        return BITMAP_ADDR + (i / size );
    }

    /// What Is The Buffer Offset? offset = (i - DATA_ADDR) % BITMAP_SIZE
    pub fn buffer_offset(addr: BlockAddr) -> usize {
        let i = (addr - DATA_ADDR) as usize;
        return i % BITMAP_SIZE;
    }

    /// Get Block State from a given &[u8]
    fn get_block_state(buffer: &[u8], addr: BlockAddr) -> bool {
        let i = Bitmap::buffer_offset(addr);
        return buffer[i / 8].get_bit(i % 8)
    }

    /// Get Block State from a given &[u8]
    fn set_block_state(buffer: &mut [u8], addr: BlockAddr, state: bool) {
        let i = Bitmap::buffer_offset(addr);
        buffer[i / 8].set_bit(i % 8, state);
    }

    /// Is The Current Data Block allocated?, Panics if [BlockAddr] < [DATA_ADDR]
    pub fn is_alloc(addr: BlockAddr) -> bool {
        let block = Block::read(Bitmap::block_addr(addr));
        Self::get_block_state(block.data_ref(), addr)
    }


    /// Allocate The Given Block, Panics if [BlockAddr] < [DATA_ADDR]
    pub fn alloc(addr: BlockAddr) {
        #[cfg(feature = "std")]
            println!("Allocated Block 0x{:06x}", addr);
        let mut block = Block::read(Bitmap::block_addr(addr));
        Self::set_block_state(block.data_mut(), addr, true);
        block.write();
    }

    /// Free The Given Block, , Panics if [BlockAddr] < [DATA_ADDR]
    pub fn free(addr: BlockAddr) {
        #[cfg(feature = "std")]
            println!("Freeing Block 0x{:06x}", addr);
        let mut block = Block::read(Bitmap::block_addr(addr));
        Self::set_block_state(block.data_mut(), addr, false);
        block.write();
    }

    /// (DEBUG FN): Fill the given String with debug information
    /// Regarding the Current state of the Bitmap. 
    /// A: Allocated,
    /// F: Free
    pub fn blocks_state(buffer: &mut String) {
        for addr in DATA_ADDR..DATA_ADDR + ((BITMAP_SIZE) as BlockAddr) {
            write!(buffer, "{}",  if Bitmap::is_alloc(addr as BlockAddr) { 'A' } else { 'F' }).expect("");
            if addr > DATA_ADDR && addr % 8 == 0 {writeln!(buffer).expect("");} 
        }
    }

    /// Attempts to Find The Next Free Block. Returns [None] if Failed.
    pub fn next_free() -> Option<BlockAddr> {
        for addr in DATA_ADDR..(DATA_ADDR + MAX_BLOCKS as u32) {
            #[cfg(feature = "std")]
            println!("Checking Block 0x{:06x}", addr);
            if !Bitmap::is_alloc(addr) {return Some(addr);}
        }
        None
    }
    /// Attempts to Find & Allocate The Next Free Block. Returns [None] if Failed.
    pub fn alloc_next_free() -> Option<BlockAddr> {
        #[cfg(feature = "std")]
            println!("Attempting To Allocate Block.");
        if let Some(addr) = Bitmap::next_free() {
            Bitmap::alloc(addr);
            Some(addr)
        } else {
            None
        }
    }

    /// Returns Disk Usage Stats, in Blocks. (Total, Used, Free)
    pub fn stats() -> (usize, usize, usize) {
        let total = MAX_BLOCKS;
        let max: BlockAddr = MAX_BLOCKS as BlockAddr;
        let mut free: usize = 0;
        let mut used: usize = 0;

        for addr in DATA_ADDR..max {
            if Bitmap::is_alloc(addr) {
                used += 1;
            } else {
                free += 1;
            }

        }

        return (total, used, free );
    }
}
