//! Tests for the file-with-meta library.
/*
 * Copyright (c) 2021  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

use std::error;
use std::fs;
use std::path;

use expect_exit::ExpectedResult;

#[test]
fn test_simple() -> Result<(), Box<dyn error::Error>> {
    let tempd_obj = tempfile::tempdir()?;
    let tempd: &path::Path = tempd_obj.as_ref();
    println!("Using tempd {}", tempd.to_string_lossy());

    let path = tempd.join("triv-data.dat");
    let path_meta = tempd.join("triv-data.dat.metatriv");

    println!("match_meta() should return None if there are no files");
    assert!(super::match_meta(&path, &path_meta).is_none());

    println!("Creating the data file...");
    fs::write(&path, "Hello!\n")?;

    println!("match_meta() should return None if there is no meta file");
    assert!(super::match_meta(&path, &path_meta).is_none());

    println!("from_file() should return a valid metadata object");
    let meta = super::FileHttpMetadata::from_file(&path)?;

    println!("from_file() returned {:?}", meta);

    println!("match_meta() should still return None");
    assert!(super::match_meta(&path, &path_meta).is_none());

    println!("Creating the metadata file");
    fs::write(&path_meta, serde_json::to_string(&meta)?)?;

    println!("match_meta() should now return something real");
    let meta_read = super::match_meta(&path, &path_meta)
        .expect_result_("match_meta() should return something")?;

    println!("The two metadata objects should be exactly the same");
    assert_eq!(meta, meta_read);

    println!("Writing modified metadata");
    let meta_mod = super::FileHttpMetadata {
        file_size: meta.file_size + 1,
        ..meta_read
    };
    fs::write(&path_meta, serde_json::to_string(&meta_mod)?)?;

    println!("match_meta() should now return None again");
    assert!(super::match_meta(&path, &path_meta).is_none());

    println!("Writing the original metadata back");
    fs::write(&path_meta, serde_json::to_string(&meta)?)?;

    println!("match_meta() should, once again, return something useful");
    assert_eq!(
        meta,
        super::match_meta(&path, &path_meta)
            .expect_result_("match_meta() should return something")?
    );

    println!("Modifying a header field should not cause match_meta() to return None");
    let meta_mod = super::FileHttpMetadata {
        file_size: meta_mod.file_size - 1,
        hdr_last_modified: Some("hello there".to_string()),
        ..meta_mod
    };
    fs::write(&path_meta, serde_json::to_string(&meta_mod)?)?;
    let meta_read = super::match_meta(&path, &path_meta)
        .expect_result_("match_meta() should return something")?;
    assert_ne!(meta, meta_read);
    let meta_read = super::FileHttpMetadata {
        hdr_last_modified: None,
        ..meta_read
    };
    assert_eq!(meta, meta_read);

    println!("Removing the data file");
    fs::remove_file(&path)?;

    println!("match_meta() should return None again");
    assert!(super::match_meta(&path, &path_meta).is_none());

    println!("Looks like we reached the end of the trivial test sequence");
    Ok(())
}

#[test]
fn test_source() -> Result<(), Box<dyn error::Error>> {
    let tempd_obj = tempfile::tempdir()?;
    let tempd: &path::Path = tempd_obj.as_ref();
    println!("Using tempd {}", tempd.to_string_lossy());

    let path = tempd.join("src-data.dat");
    let path_meta = tempd.join("src-data.dat.metasrc");

    let path_source = tempd.join("src-src.dat");

    println!("match...source() should return empty with no files");
    assert!(super::match_meta_with_source(&path, &path_meta, &path_source).is_none());

    println!("from..source() should error out with no files");
    super::FileHttpMetadata::from_file_with_source(&path, &path_source).unwrap_err();

    println!("match...source() should return empty with no meta file");
    assert!(super::match_meta_with_source(&path, &path_meta, &path_source).is_none());
    fs::write(&path, "Hi there!")?;

    println!("match...source() should return something even with no source file");
    let meta = super::FileHttpMetadata::from_file(&path)?;
    assert!(meta.source_file_size.is_none());
    assert!(meta.source_file_mtime.is_none());
    fs::write(&path_meta, serde_json::to_string(&meta)?)?;

    let meta_read = super::match_meta_with_source(&path, &path_meta, &path_source)
        .expect("match..source() with no source should still succeed");
    assert_eq!(meta, meta_read);

    println!("Now let us see what happens if we create a source file...");
    fs::write(&path_source, "Something something something source file")?;
    let meta_source = super::FileHttpMetadata::from_file(&path_source)?;
    let meta = super::FileHttpMetadata::from_file_with_source(&path, &path_source)?;
    assert_eq!(meta.source_file_size, Some(meta_source.file_size));
    assert_eq!(meta.source_file_mtime, Some(meta_source.file_mtime));
    assert_ne!(meta, meta_read);
    assert_eq!(
        meta,
        super::FileHttpMetadata {
            source_file_size: Some(meta_source.file_size),
            source_file_mtime: Some(meta_source.file_mtime),
            ..meta_read
        }
    );

    println!("Making sure match..with_source() also works");
    fs::write(&path_meta, serde_json::to_string(&meta)?)?;
    let meta_read = super::match_meta_with_source(&path, &path_meta, &path_source)
        .expect("match..source() with no source should still succeed");
    assert_eq!(meta, meta_read);

    println!("Now let us remove the source file and see what happens");
    fs::remove_file(&path_source)?;

    println!("match..source() should return None");
    super::match_meta_with_source(&path, &path_meta, &path_source)
        .ok_or(super::MetadataError {
            msg: "why is there something here?".to_string(),
        })
        .expect_err("match..source() should return None with no source file and recorded data");

    println!("from..source() should error out");
    super::FileHttpMetadata::from_file_with_source(&path, &path_source).unwrap_err();

    println!("from..source_meta() should still succeed");
    let meta_read = super::FileHttpMetadata::from_file_with_source_meta(&path, &meta_source)?;
    assert_eq!(meta, meta_read);

    println!("Replacing the metadata, forgetting about the source part");
    let meta_read = super::FileHttpMetadata {
        source_file_size: None,
        source_file_mtime: None,
        ..meta_read
    };
    fs::write(&path_meta, serde_json::to_string(&meta_read)?)?;
    let meta_read = super::match_meta_with_source(&path, &path_meta, &path_source)
        .expect("match..source() with no source should still succeed");
    assert_ne!(meta, meta_read);
    assert_eq!(
        meta,
        super::FileHttpMetadata {
            source_file_size: Some(meta_source.file_size),
            source_file_mtime: Some(meta_source.file_mtime),
            ..meta_read
        }
    );

    println!("Done with the source test sequence");
    Ok(())
}

#[test]
fn test_default() -> Result<(), Box<dyn error::Error>> {
    let meta = super::FileHttpMetadata::default();
    println!("default meta: {:?}", meta);

    assert_eq!(meta.format.version.major, 0);
    assert_eq!(meta.format.version.minor, 1);
    assert_eq!(meta.file_size, 0);
    assert_eq!(meta.file_mtime, 0);
    assert!(meta.source_file_size.is_none());
    assert!(meta.source_file_mtime.is_none());

    Ok(())
}

#[test]
fn test_send() {
    fn assert_send<T: Send>() {}

    assert_send::<super::FileHttpMetadata>();
    assert_send::<super::MetadataError>();
}

#[test]
fn test_sync() {
    fn assert_sync<T: Sync>() {}

    assert_sync::<super::FileHttpMetadata>();
    assert_sync::<super::MetadataError>();
}
