#![warn(missing_docs)]
//! Store a file's metadata for caching purposes.
//!
//! The [`FileHttpMetadata`] structure may be serialized and
//! stored in a JSON file alongside the real file, e.g. one with
//! ".meta" appended to the file name. Then either the [`match_meta`]
//! function may be used directly, or the [`build_req`] one may be
//! used to modify an HTTP request, adding the necessary headers to
//! make sure that the file is not downloaded if there have been
//! no changes on the remote server.
//!
//! Example for checking whether a file needs to be downloaded:
//! ```rust
//! # use std::error;
//! use std::fs;
//! # use std::io::{self, Read, Write};
//! # use std::path;
//!
//! # fn main() -> Result<(), Box<dyn error::Error>> {
//! # let agent = ureq::agent();
//! # let tempd_obj = tempfile::tempdir()?;
//! # let destdir: &path::Path = tempd_obj.as_ref();
//! let dst = destdir.join("data.json");
//! let dst_meta = destdir.join("data.json.meta");
//! let (req, stored_meta) = file_with_meta::build_req(
//!     agent.get("https://example.com/"),
//!     &dst,
//!     &dst_meta,
//! );
//! let resp = req.call()?;
//! match resp.status() {
//!     304 => println!("Nothing was fetched"),
//!     _ => {
//!         println!("Storing the content");
//!         /* ... */
//! #         let mut reader = resp.into_reader();
//! #         let mut outfile = fs::File::create(&dst)?;
//! #         let mut writer = io::BufWriter::new(&outfile);
//! #         loop {
//! #             let mut buf = [0; 8192];
//! #             let n = reader.read(&mut buf[..])?;
//! #             if n == 0 {
//! #                 break;
//! #             }
//! #             writer.write_all(&buf[..n])?;
//! #         }
//! #         writer.flush()?;
//! #         outfile.sync_all()?;
//!
//!         println!("Updating the file's metadata");
//!         let meta = file_with_meta::FileHttpMetadata::from_file(&dst)?;
//!         fs::write(&dst_meta, serde_json::to_string(&meta).unwrap())?;
//!     }
//! };
//! # Ok(())
//! # }
//! ```
//!
//! Example for checking whether a file has changed since its metadata
//! was last updated:
//! ```rust
//! let dst = "/path/to/file.dat";
//! let dst_meta = "/path/to/file.dat.meta";
//!
//! match file_with_meta::match_meta(&dst, &dst_meta).is_some() {
//!     true => println!("No change"),
//!     false => println!("Somebody touched our file, recreate it?"),
//! };
//! ```
//!
//! The [`match_meta_with_source`] function may be used to additionally
//! make sure that a "source" file has not been modified since this file
//! was last generated from its data.

#[doc(html_root_url = "https://docs.rs/file-with-meta/0.1.0")]
/*
 * Copyright (c) 2021  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
use std::error;
use std::fmt;
use std::fs;
use std::path;
use std::time;

use serde::{Deserialize, Serialize};

#[cfg(test)]
mod tests;

/// An error that occurred during loading the metadata.
#[derive(Debug)]
pub struct MetadataError {
    /// The error message.
    msg: String,
}

impl fmt::Display for MetadataError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.msg)
    }
}

impl error::Error for MetadataError {}

impl MetadataError {
    /// Return a boxed error with the specified message.
    pub fn boxed(msg: String) -> Box<Self> {
        Box::new(Self { msg })
    }
}

/// The version of the format of the serialized metadata.
#[derive(Debug, Serialize, Deserialize, PartialEq)]
pub struct MetadataFormatVersion {
    /// The major version number; bumped when a field is removed or
    /// its type is changed.
    major: u32,
    /// The minor version number; bumped when a new field is added.
    minor: u32,
}

impl Default for MetadataFormatVersion {
    /// The default format version is the most recent one.
    fn default() -> Self {
        Self { major: 0, minor: 1 }
    }
}

/// Information about the format of the JSON-serialized metadata.
#[derive(Debug, Default, Serialize, Deserialize, PartialEq)]
pub struct MetadataFormat {
    /// The version of the metadata format, currently 0.x.
    version: MetadataFormatVersion,
}

#[derive(Debug, Serialize, Deserialize)]
struct MetadataTopLevelFormatOnly {
    format: MetadataFormat,
}

/// Information about a single file's last modification time and,
/// if specified, some relevant HTTP headers returned by the server
/// that the file was fetched from.
#[derive(Debug, Default, Serialize, Deserialize, PartialEq)]
pub struct FileHttpMetadata {
    /// The version of the metadata as stored in a JSON string.
    pub format: MetadataFormat,
    /// The size of the file.
    pub file_size: u64,
    /// The modification time of the file as a Unix timestamp.
    pub file_mtime: u64,
    /// The "Last-Modified" header as returned by an HTTP server.
    pub hdr_last_modified: Option<String>,
    /// The "ETag" header as returned by an HTTP server.
    pub hdr_etag: Option<String>,
    /// The size of the source file if applicable.
    pub source_file_size: Option<u64>,
    /// The modification time of the source file if applicable.
    pub source_file_mtime: Option<u64>,
    /// A hook for external users to store information about whether
    /// the file's contents has been validated.
    pub verified: bool,
}

impl FileHttpMetadata {
    /// Examine an existing file and return a metadata structure
    /// recording its size and last modification time.
    pub fn from_file<P>(path: P) -> Result<Self, Box<dyn error::Error>>
    where
        P: AsRef<path::Path>,
    {
        match fs::metadata(&path) {
            Ok(meta) => Ok(Self {
                file_size: meta.len(),
                file_mtime: mtime_to_unix(&meta),
                ..Self::default()
            }),
            Err(err) => Err(MetadataError::boxed(format!(
                "Could not examine {}: {}",
                path.as_ref().to_string_lossy(),
                err
            ))),
        }
    }

    /// Examine an existing file and return a metadata structure
    /// recording its size and last modification time, as well as
    /// that of the specified "source" file.
    pub fn from_file_with_source<P1, P2>(path: P1, src: P2) -> Result<Self, Box<dyn error::Error>>
    where
        P1: AsRef<path::Path>,
        P2: AsRef<path::Path>,
    {
        let meta = Self::from_file(path)?;
        match fs::metadata(&src) {
            Ok(src_meta) => Ok(Self {
                source_file_size: Some(src_meta.len()),
                source_file_mtime: Some(mtime_to_unix(&src_meta)),
                ..meta
            }),
            Err(err) => Err(MetadataError::boxed(format!(
                "Could not examine {}: {}",
                src.as_ref().to_string_lossy(),
                err
            ))),
        }
    }

    /// Examine an existing file and return a metadata structure
    /// recording its size and last modification time, as well as
    /// the previously-stored one for a "source" file.
    pub fn from_file_with_source_meta<P>(
        path: P,
        src_meta: &FileHttpMetadata,
    ) -> Result<Self, Box<dyn error::Error>>
    where
        P: AsRef<path::Path>,
    {
        let meta = Self::from_file(path)?;
        Ok(Self {
            source_file_size: Some(src_meta.file_size),
            source_file_mtime: Some(src_meta.file_mtime),
            ..meta
        })
    }

    /// Parse a metadata structure from the supplied JSON string.
    /// Verify the version specified in the "format" element, do not
    /// even attempt to parse unknown versions.
    pub fn parse(contents: &str) -> Result<Self, Box<dyn error::Error>> {
        match serde_json::from_str::<MetadataTopLevelFormatOnly>(contents) {
            Ok(header) => match header.format.version.major {
                0 => serde_json::from_str::<Self>(contents)
                    .map_err(|err| -> Box<dyn error::Error> { Box::new(err) }),
                _ => Err(MetadataError::boxed(format!(
                    "Unsupported metadata format version {}",
                    header.format.version.major
                ))),
            },
            Err(err) => Err(Box::new(err)),
        }
    }
}

/// Unwrap a [`fs::Metadata`] object's last modified timestamp,
/// assume it may be converted to a Unix timestamp, and return
/// the number of seconds since the Unix epoch.
pub fn mtime_to_unix(metadata: &fs::Metadata) -> u64 {
    metadata
        .modified()
        .unwrap()
        .duration_since(time::SystemTime::UNIX_EPOCH)
        .unwrap()
        .as_secs()
}

/// Verify that a file has not been changed since the last time
/// the metadata was stored.
pub fn match_meta<P1, P2>(dst: P1, dst_meta: P2) -> Option<FileHttpMetadata>
where
    P1: AsRef<path::Path>,
    P2: AsRef<path::Path>,
{
    match fs::metadata(&dst) {
        Ok(file_meta) => match fs::read_to_string(&dst_meta) {
            Ok(contents) => match FileHttpMetadata::parse(&contents) {
                Ok(meta) => match file_meta.is_file()
                    && file_meta.len() == meta.file_size
                    && mtime_to_unix(&file_meta) == meta.file_mtime
                {
                    true => Some(meta),
                    false => None,
                },
                Err(_) => None,
            },
            Err(_) => None,
        },
        Err(_) => None,
    }
}

/// Verify that a file has not been changed, and additionally verify
/// that its source file, specified by the `src` local path, has
/// also not been changed. Useful when e.g. uncompressing or otherwise
/// processing downloaded files.
pub fn match_meta_with_source<P1, P2, P3>(
    dst: P1,
    dst_meta: P2,
    src: P3,
) -> Option<FileHttpMetadata>
where
    P1: AsRef<path::Path>,
    P2: AsRef<path::Path>,
    P3: AsRef<path::Path>,
{
    match_meta(dst, dst_meta).and_then(|meta| match fs::metadata(src) {
        Ok(src_meta) => {
            let src_len = src_meta.len();
            match meta.source_file_size.or(Some(src_len)).unwrap() == src_len {
                true => {
                    let src_mtime = mtime_to_unix(&src_meta);
                    match meta.source_file_mtime.or(Some(src_mtime)).unwrap() == src_mtime {
                        true => Some(meta),
                        false => None,
                    }
                }
                false => None,
            }
        }
        Err(_) => match meta.source_file_size.is_some() || meta.source_file_mtime.is_some() {
            false => Some(meta),
            true => None,
        },
    })
}

#[cfg(feature = "ureq")]
/// Add the "If-Modified-Since" and/or "If-None-Match" headers to
/// an HTTP request if the relevant fields ("Last-Modified" and "ETag"
/// respectively) have been returned in the last response from
/// the server when the file has been downloaded.
pub fn build_req<P1, P2>(
    orig_req: ureq::Request,
    dst: P1,
    dst_meta: P2,
) -> (ureq::Request, Option<FileHttpMetadata>)
where
    P1: AsRef<path::Path>,
    P2: AsRef<path::Path>,
{
    let stored_meta = match_meta(dst, dst_meta);

    let req = match &stored_meta {
        None => orig_req,
        Some(meta) => match &meta.hdr_etag {
            Some(etag) => orig_req.set("If-None-Match", &etag),
            None => match &meta.hdr_last_modified {
                Some(last_modified) => orig_req.set("If-Modified-Since", &last_modified),
                None => orig_req,
            },
        },
    };
    (req, stored_meta)
}
