use std::fs;
use std::env;
use cc::Build;
use std::path::PathBuf;
use std::process::Command;

fn scan(build: &mut Build, path: &str, suffix: &str) {
    let paths = fs::read_dir(path).unwrap();

    for path in paths {
        let p = path.unwrap().path();
        let str = p.to_str().unwrap();

        if str.ends_with(suffix) {
            build.file(str);
        }
    }
}

fn main() {
    let mut c = cc::Build::new();
    c.warnings(false);
    c.cargo_metadata(false);

    scan(&mut c, "./device-detection-cxx/src/", ".c");
    scan(&mut c, "./device-detection-cxx/src/hash/", ".c");
    scan(&mut c, "./device-detection-cxx/src/common-cxx/", ".c");

    c.compile("51c");

    let source_files = vec!["src/shim/mod.rs"];

    let mut build = cxx_build::bridges(source_files);
    build.warnings(false);

    scan(&mut build, "./src/shim/", ".cpp");
    scan(&mut build, "./device-detection-cxx/src/", ".cpp");
    scan(&mut build, "./device-detection-cxx/src/hash/", ".cpp");
    scan(&mut build, "./device-detection-cxx/src/common-cxx/", ".cpp");

    let out = PathBuf::from(env::var("OUT_DIR").unwrap());

    build.cargo_metadata(false);
    build.include(".");
    build.flag("-std=c++11");
    build.compile("51c++");

    // prevent git submodule from being packaged
    fs::remove_dir_all(out.join("device-detection-cxx")).unwrap();

    /*
    see https://github.com/rust-lang/rust/issues/38095
    because the project is a mix of c and c++ we have to build two separate libraries, but c++
    depends on the c code. when a downstream project includes this crate, it will end up with
    "multiple definition of" errors, so we need to combine the two libraries and only link one.
     */

    Command::new("libtool").args(&["-static", "-o", "lib51.a", "lib51c.a", "lib51c++.a"])
        .current_dir(out.to_str().unwrap())
        .status().unwrap();

    println!("cargo:rustc-link-search=native={}", out.to_str().unwrap());
    println!("cargo:rustc-link-lib=static=51");

    println!("cargo:rerun-if-changed=src/shim/mod.rs");
    println!("cargo:rerun-if-changed=src/shim/shim.cpp");
    println!("cargo:rerun-if-changed=src/shim/shim.hpp");
}
