#[macro_use]
extern crate lazy_static;

mod db;
mod err;
mod runctx;

use std::env;
use std::fs::{self, File};
use std::io::{self, BufRead};
use std::path::{Path, PathBuf};
use std::time::Instant;

use env_logger::Env;

use hashbrown::{HashMap, HashSet};

use filetime::FileTime;

use walkdir::WalkDir;

use glob::glob;

use sha2::{Digest, Sha256};

use log::{debug, info, trace, warn};

use db::DB;
use runctx::RunContext;

use terminal_size::{terminal_size, Height, Width};

use err::Error;

type CmdProc = fn(&str, &[String]) -> Result<(), Error>;


lazy_static! {
  static ref CMDTBL: HashMap<&'static str, CmdProc> = {
    let mut map: HashMap<&str, CmdProc> = HashMap::new();
    map.insert("help", proc_help);
    map.insert("init", proc_init);
    map.insert("tag", proc_tag_untag);
    map.insert("untag", proc_tag_untag);
    map.insert("tags", proc_tags);
    map.insert("search", proc_search);
    map.insert("update", proc_update);
    map.insert("verify", proc_verify);
    map.insert("vacuum", proc_vacuum);
    map.insert("add-tag", proc_add_tag);
    map.insert("remove-tag", proc_remove_tag);
    map.insert("rename-tag", proc_rename_tag);
    map.insert("list-tags", proc_list_tags);
    map.insert("dups", proc_dups);
    map.insert("dedup", proc_dedup);
    map.insert("version", proc_version);
    map
  };
}


fn main() -> Result<(), Box<dyn std::error::Error>> {
  let env = Env::default().filter_or("LOG_LEVEL", "error");
  env_logger::init_from_env(env);

  //
  // Collect command line arguments into an array of utf-8 strings, and
  // terminate if there are none.
  //
  let args: Vec<String> = env::args().collect();
  if args.len() < 2 {
    println!("Nothing to do.  Use subcommand 'help' for list of commands.");
    return Ok(());
  }

  //
  // Look up subcommand and run its handler
  //
  if let Some(proc) = CMDTBL.get::<str>(&args[1]) {
    let start = Instant::now();

    proc(&args[1], &args[2..])?;

    let dur = Instant::now() - start;
    trace!("{} command handler took: {:?}", args[1], dur);
  } else {
    eprintln!("Unknown command '{}'", args[1]);
    std::process::exit(1);
  }

  Ok(())
}


/// Process `init` subcommand.
///
/// Creates a new database in the current directory, or fail if one already
/// exists.
fn proc_init(cmd: &str, args: &[String]) -> Result<(), Error> {
  trace!("Process {} command", cmd);

  //
  // Set up a RunContext with the dbfile path pointing to the current
  // directory and initialize a new database.
  // Fail if the dbfile already exists.
  //
  let ctx = RunContext::init(cmd, args)?;
  let _db = DB::init(&ctx)?;

  Ok(())
}


/// Process `tag` and `untag` subcommands.
///
/// `cmd` is either `tag` or `untag`.
///
/// By default `arg[0]` is the file being tagged, and the remaining arguments
/// are tag names or id's.
///
/// If one of the arguments is `--` it is used as a separator between files and
/// tags.
///
/// If either one of the files or tag names begin with a `@` the string
/// following it will be interpreted as a file name which lists, line by line,
/// the files/tags to process.
///
/// - `args[1]` is the pathname of the file to tag or untag.
/// - `args[2..]` are the tags to attach or detach from the file in args[1]
fn proc_tag_untag(cmd: &str, args: &[String]) -> Result<(), Error> {
  trace!("Process {} command", cmd);

  //
  // If there's no "--" in the argument list then assume the first argument is
  // the file name, and the rest are tags.  If there's a "--" then assume
  // everything before it is a list of file names and that everything after are
  // tags.
  //
  let (files, tags) = match args.iter().position(|s| s == "--") {
    Some(idx) => (&args[..idx], &args[idx + 1..]),
    None => (&args[0..1], &args[1..])
  };

  //
  // Iterate over files and tags lists and check if any of them are prefixed by
  // an '@' character.  If they are, then treat the string following it as a
  // file name from which to load files/tags
  //
  //println!("files: {:?}", files);
  //println!("tags: {:?}", tags);

  let files = at_transform(files)?;
  let tags = at_transform(tags)?;

  //println!("files: {:?}", files);
  //println!("tags: {:?}", tags);

  //
  // Locate base directory and open database.
  //
  let ctx = RunContext::find_base(cmd, args, false)?;
  let db = DB::open(&ctx)?;

  for fname in files {
    //
    // Generate the in-tree pathname of the file being tagged/untagged
    //
    let fname = ctx.gen_tree_path(fname);

    debug!("{} file {:?}", cmd, fname);

    if let Some(dbfi) = db.get_file_by_fname(fname.to_str().unwrap())? {
      for t in &tags {
        if cmd == "tag" {
          trace!("attaching tag {} to ufile_id {}", t, dbfi.ufile_id);
          match db.tag(dbfi.ufile_id, t) {
            Ok(_) => {}
            Err(e) => {
              eprintln!(
                "Unable to attach tag '{}' to ufile_id {}; {}",
                t, dbfi.ufile_id, e
              );
            }
          }
        } else {
          trace!("deatching tag {} from ufile_id {}", t, dbfi.ufile_id);
          match db.untag(dbfi.ufile_id, t) {
            Ok(_) => {}
            Err(e) => {
              eprintln!(
                "Unable to detach tag '{}' from ufile_id {}; {}",
                t, dbfi.ufile_id, e
              );
            }
          }
        }
      }
    } else {
      eprintln!(
        "File {:?} not found in database.  Perhaps an update is needed.",
        fname
      );
    }
  }

  Ok(())
}


/// Given a slice of strings, generate an output list where any entry with the
/// format `@<filename>` with the contents of the `filename`.
fn at_transform(input: &[String]) -> Result<Vec<String>, Error> {
  let mut ret = Vec::new();

  for n in input {
    let mut chit = n.chars();
    let first = chit.next();

    if let Some(ch) = first {
      if ch == '@' {
        let fname: String = chit.collect();
        let fname = Path::new(&fname);
        if fname.exists() {
          let lines = read_lines(&fname)?;
          for line in lines {
            let line = line?;
            ret.push(line);
          }
        } else {
          ret.push(n.to_string());
        }
      } else {
        ret.push(n.to_string());
      }
    } else {
      ret.push(n.to_string());
    }
  }

  Ok(ret)
}


fn read_lines<P>(
  filename: P
) -> std::io::Result<std::io::Lines<std::io::BufReader<File>>>
where
  P: AsRef<Path>
{
  let file = File::open(filename)?;
  Ok(io::BufReader::new(file).lines())
}


/// Process `tags` subcommand.
///
/// Lists all tags for a specific file.
fn proc_tags(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Find and open database in read-only mode
  //
  let ctx = RunContext::find_base(cmd, args, true)?;
  let db = DB::readonly(&ctx)?;

  //
  // Iterate over command line arguments, append them to the current subdir and
  // look up that entry's unique entry in the database and dump all of its
  // tags.
  //
  for fname in args {
    let fname = ctx.gen_tree_path(fname);

    println!("list tags for {}", fname.display());
    if let Some(dbfi) = db.get_file_by_fname(fname.to_str().unwrap())? {
      let tags = db.get_ufile_tags(dbfi.ufile_id)?;
      for (id, title) in tags {
        println!("{:4} {}", id, title);
      }
    }
  }

  Ok(())
}


/// Search for files matching a set of tags, and dump a list of files matching
/// the seach criteria.  A file must have _all_ tags for it to match.
fn proc_search(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Find and open database in read-only mode
  //
  let ctx = RunContext::find_base(cmd, args, true)?;
  let db = DB::readonly(&ctx)?;

  //
  // Iterate over tags and get all unique files with those tags
  //
  let lst = db.get_ufiles_by_tags(args)?;
  for ufid in lst {
    //println!("{}", ufid);

    //
    // Dump list of files which refer to this unique file
    //
    for (_fid, fname) in db.get_files_by_ufile(ufid)? {
      //println!("  {:4} {}", fid, fname)
      println!("{}", fname)
    }
  }

  Ok(())
}


fn proc_update(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Open database in current directory, fail if it isn't found.
  //
  let ctx = RunContext::open_here(cmd, args)?;
  let db = DB::open(&ctx)?;


  //
  // Load ignore list from database
  //
  let ignore = get_ignore(&db)?;

  debug!("ignore: {:?}", ignore);

  let rootdir = Path::new(".");

  for entry in WalkDir::new(".").into_iter().filter_entry(|e| {
    let p = e.path();

    // Ignore symlinks
    if let Ok(_pth) = p.read_link() {
      return false;
    }

    if let Some(path_str) = p.strip_prefix("./").unwrap().to_str() {
      !ignore.contains(path_str)
    } else {
      // Not unicode -- skip it
      false
    }
  }) {
    //
    // No idea what a DirEntry failure would look like, but we'll simply ignore
    // them for now.
    //
    let entry = match entry {
      Ok(entry) => entry,
      Err(e) => {
        eprintln!("Ignoring entry due to: {}", e);
        continue;
      }
    };

    //
    // Get a Path reference to the current entry.
    //
    let p = entry.path();

    //
    // Ignore the root "." entry
    //
    if p == rootdir {
      continue;
    }

    //
    // Strip the "./" prefix from all entries.
    // This would not apply to the root "." entry, but it was ignored above.
    //
    let p = p.strip_prefix("./")?;

    //
    // We only care about regular files
    //
    if !p.is_file() {
      continue;
    }

    //
    // Only work with valid unicode filenames
    //
    let fname = match p.to_str() {
      Some(fname) => fname,
      None => {
        warn!("Skipping non-unicode file name: {:?}", p);
        continue;
      }
    };


    //
    // If this point is reached then it's a file we care about
    //


    //
    // Get file's mtime in unix timestamp seconds in the filesystem
    //
    let metadata = fs::metadata(&fname).unwrap();
    let mtime = FileTime::from_last_modification_time(&metadata);
    let unix_seconds = mtime.unix_seconds();

    //
    // Get file information, by path, from database if it exists.
    // If an entry exists:
    // - If the mtime differs, then recalculate the file's hash.
    // - if the mtime is unchanged, assume the hash is unchanged.
    //
    trace!("Checking db to see if we have {} ..", fname);
    let dbfi = db.get_file_by_fname(fname)?;


    if let Some(e) = dbfi {
      trace!("Have file {:?} in database", p);

      // If the file exists, but its mtime has changed, then we should update
      // the hash.
      //
      // It's important that we don't update the hash unless the mtime is
      // changes, so we can use it to detect if the file's contents have
      // changed in some way (notably if the storage media is unstable).
      if e.mtime != unix_seconds {
        debug!("Timestamp has been updated, rehash the file");

        let hash = hashfile(p)?;
        let ufile_id = db.insert_or_get_ufile(&hash)?;
        match ufile_id {
          db::EntryId::New(ufile_id) | db::EntryId::Old(ufile_id) => {
            db.file_modified(e.id, unix_seconds, ufile_id)?;

            // For files which changed contents, migrate its old tags to the
            // new ufiles entry.
            db.migrate_tags(e.ufile_id, ufile_id)?;
          }
        }
      }
    } else {
      trace!("Don't have file {:?} in the database", p);

      //
      // Hash the file
      //
      let hash = hashfile(p)?;

      let ufile_id = db.insert_or_get_ufile(&hash)?;
      match ufile_id {
        db::EntryId::New(id) | db::EntryId::Old(id) => {
          db.add_file(fname, unix_seconds, id)?;
        }
      }
    };
  }

  //
  // When entries are added to the ignore list after an update has been run,
  // they will remain unless explicitly deleted.  So do that here.
  //
  info!("Removing ignored entries");
  db.remove_files_by_name(&ignore)?;

  //
  // Remove any files entries which have gone missing in the filesystem
  //
  info!("Remove missing entries");
  db.remove_missing()?;

  //
  // Remove any stale entries from the ufiles table.
  // These are ufiles entries which are no longer being referenced to by files
  // entries.
  //
  info!("Remove stale ufiles entries");
  db.delete_stale_ufiles()?;

  Ok(())
}


/// Generate the ignore list.
///
/// Load the ignore list from the database and apply glob to them and generate
/// a set of entries.
fn get_ignore(db: &DB) -> Result<HashSet<String>, Error> {
  let mut ignore: HashSet<String> = HashSet::new();

  //
  // Load ignore list from database into a set.
  //
  let set = db.get_ignore_set()?;

  //
  // Iterate over the set of ignore entries and apply glob rules.
  //
  for ig in set {
    for entry in glob(&ig).expect("Unable to read glob pattern") {
      let path = match entry {
        Ok(path) => path,
        Err(e) => {
          // The path matched but was unreadable, preventing its contents from
          // matching.
          warn!("Unable to glob match a path; {}", e);
          continue;
        }
      };

      let path = match path.to_str() {
        Some(path) => path.to_string(),
        None => {
          warn!("Skipping non-unicode path during glob matching");
          continue;
        }
      };

      ignore.insert(path);
    }
  }

  Ok(ignore)
}


fn hashfile(p: &Path) -> Result<String, Error> {
  let mut file = File::open(p)?;
  let mut sha256 = Sha256::new();
  io::copy(&mut file, &mut sha256)?;
  let hash = sha256.finalize();
  Ok(format!("{:x}", hash))
}


fn proc_verify(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Locate database and switch current working directory to it if it's higher
  // up in the tree, and open it up in read-only mode.
  //
  let ctx = RunContext::find_base(cmd, args, true)?;
  let db = DB::readonly(&ctx)?;

  //
  // Generate the verification path within the managed tree.
  //
  let verify_from = if let Some(subdir) = &ctx.subdir {
    // Not running verify from the base
    if args.is_empty() {
      // No arg was specified
      Some(subdir.clone())
    } else {
      // An arg was specified
      Some(subdir.join(&args[0]))
    }
  } else {
    // Running verify from the base
    if args.is_empty() {
      // No arg was specified
      None
    } else {
      // An arg was specified
      Some(PathBuf::from(&args[0]))
    }
  };

  //
  // Run verification
  //
  db.verify(verify_from)?;

  Ok(())
}


fn proc_vacuum(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Locate database and open database in read/write mode.
  //
  let ctx = RunContext::find_base(cmd, args, false)?;
  let db = DB::open(&ctx)?;

  //
  // Vacuum
  //
  db.vacuum()?;

  Ok(())
}


/// Process `add-tag` subcommand.
///
/// `args[0..]` is a list of tags to add to the database.  Pure-numeric tag
/// names are not allowed.
fn proc_add_tag(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Locate base directory and open database in read/write mode.
  //
  let ctx = RunContext::find_base(cmd, args, false)?;
  let db = DB::open(&ctx)?;

  for tag in args {
    if tag.parse::<i64>().is_ok() {
      eprintln!(
        "tag name '{}' ignored as pure numeric tag names are not allowed",
        tag
      );
      continue;
    }
    if db.add_tag(tag).is_err() {
      eprintln!("Unable to add tag '{}'", tag);
    }
  }

  Ok(())
}


/// Process `remove-tag` subcommand.
///
/// `args[0..]` is a list of tags to remove from the database.
/// Pure-numeric arguments are interpreted as database id's.  Non-numeric
/// arguments are treated as tag names.
fn proc_remove_tag(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Locate base directory and open database.
  //
  let ctx = RunContext::find_base(cmd, args, false)?;
  let db = DB::open(&ctx)?;

  //
  // Iterate over arguments and attempt to remove each entry
  //
  for tag in args {
    if db.remove_tag(tag).is_err() {
      eprintln!("Unable to remove tag '{}'", tag);
    }
  }

  Ok(())
}


/// Process `rename-tag` subcommand.
///
/// `arg[0]` is the tag being renamed.  If this is purely numeric it refers to
/// the tag id.  For non-numeric values this it is treated as a tag name.
fn proc_rename_tag(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Locate base directory and open database in read/write mode.
  //
  let ctx = RunContext::find_base(cmd, args, false)?;
  let db = DB::open(&ctx)?;

  //
  // Make sure the new name isn't purely numerical
  //
  if args[1].parse::<i64>().is_ok() {
    panic!("pure numeric tag names are not allowed");
  }

  db.rename_tag(args[0].as_ref(), args[1].as_ref())?;

  Ok(())
}


/// List all tags in the fidx database.
fn proc_list_tags(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Locate database and open it in read-onle mode
  //
  let ctx = RunContext::find_base(cmd, args, true)?;
  let db = DB::readonly(&ctx)?;

  //
  // Get list of all tags in the database
  //
  let lst = db.get_tag_list()?;

  let slist: Vec<(String, String)> = lst
    .iter()
    .map(|(id, title)| (id.to_string(), title.clone()))
    .collect();

  let mut max_id_len = 0;
  let mut max_title_len = 0;
  for (ids, title) in &slist {
    if ids.len() > max_id_len {
      max_id_len = ids.len();
    }
    if title.len() > max_title_len {
      max_title_len = title.len();
    }
  }

  let size = terminal_size();
  let dump_list = if let Some((Width(w), Height(h))) = size {
    if slist.len() > h as usize / 2 {
      //
      // Dump table
      //

      // single entry width  (longest id + 1 space + longest title + 2 spaces
      // to separate columns).
      // ToDo: This wastes some space unecessarily -- fix this.
      let swidth = max_id_len + 1 + max_title_len + 2;

      // Calculate the number of columns to use
      let ncols = w as usize / swidth;

      // Calculate rows
      let nrows = if slist.len() % ncols == 0 {
        slist.len() / ncols
      } else {
        slist.len() / ncols + 1
      };

      for row in 0..nrows {
        for col in 0..ncols {
          let idx = nrows * col + row;
          if idx < slist.len() {
            print!(
              "{ids:>idwidth$} {title:titlewidth$}",
              ids = slist[idx].0,
              title = slist[idx].1,
              idwidth = max_id_len,
              titlewidth = max_title_len
            );

            if col < ncols - 1 {
              print!("  ");
            }
          }
        }
        println!("");
      }
      false
    } else {
      true
    }
  } else {
    true
  };

  if dump_list {
    //
    // Dump list if table wasn't dumped
    //
    for (ids, title) in lst {
      println!(
        "{ids:>idwidth$} {title:titlewidth$}",
        idwidth = max_id_len,
        titlewidth = max_title_len
      );
    }
  }

  Ok(())
}


/// Process `dups` subcommand.
///
/// List all duplicate entries in the database.
fn proc_dups(cmd: &str, args: &[String]) -> Result<(), Error> {
  //
  // Locate database and open it in read-onle mode
  //
  let ctx = RunContext::find_base(cmd, args, true)?;
  let db = DB::readonly(&ctx)?;


  //
  // Get a list of all ufiles which have multiple files associated with them.
  //
  let lst = db.get_dup_ufiles()?;
  for ufid in lst {
    println!("{}", ufid);

    //
    // Get list of files associated with unique file and dump them
    //
    for (fid, fname) in db.get_files_by_ufile(ufid)? {
      println!("  {:4} {}", fid, fname)
    }
  }

  Ok(())
}


/// For files with the same hash, make sure they all belong to the same inode.
///
/// Before performing any actual deduplication make sure the files haven't
/// actually been modified -- don't blindly trust the values in the database.
fn proc_dedup(_cmd: &str, _args: &[String]) -> Result<(), Error> {
  unimplemented!("deduplication not implemented yet");
}


/// Just print the program name and version and then return.
fn proc_version(_cmd: &str, _args: &[String]) -> Result<(), Error> {
  println!("{} {}", env!("CARGO_PKG_NAME"), env!("CARGO_PKG_VERSION"));
  Ok(())
}


fn proc_help(_cmd: &str, _args: &[String]) -> Result<(), Error> {
  println!(
    r#"
help
  Show this help.
version
  Print program name and version.

init
  Initialize a new fidx base directory at the current location.
update
  Scan for file changes and update database to reflect changes.
verify [subdir|file]
  Givent the current state of the database, verify files in file system.

list-tags
  List tags in database.
add-tag [tag1 [tag2 [.. [tagN]]]]
  Add tags to the database.
rename-tag [old-name|id] new-name
  Rename a tag to a new name.
remove-tag [name1|id1 [.. [nameN|idN]]]
  Remove tags, by name or id, from the database.

tag <file> [tag1 [.. [tagN]]]
  Associate the specified file with the specified tags.
untag <file> [tag1 [.. [tagN]]]
  Disassociate the specified file from the specified tags.
tags <file>
  Write a list of tags the specified file is associated with.
search [tag1 [.. [tagN]]]
  Search for files having all specified tags.

dups
  List of all duplicate file entries in database.
dedup
  Attempt to deduplicate files.

vacuum
  Perform a full vacuum of the database.
"#
  );
  Ok(())
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
