use std::fmt;

#[derive(Debug)]
pub enum Error {
  IO(String),
  DbAlreadyExists,
  MissingDB,
  Parse(String),
  SQLite(String),
  Verify
}


impl std::error::Error for Error {}

impl From<std::io::Error> for Error {
  fn from(err: std::io::Error) -> Self {
    Error::IO(err.to_string())
  }
}

impl From<rusqlite::Error> for Error {
  fn from(err: rusqlite::Error) -> Self {
    Error::SQLite(err.to_string())
  }
}

impl From<std::path::StripPrefixError> for Error {
  fn from(err: std::path::StripPrefixError) -> Self {
    Error::Parse(err.to_string())
  }
}

impl fmt::Display for Error {
  fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
    match &*self {
      Error::DbAlreadyExists => {
        write!(f, "Database already exists")
      }
      Error::IO(s) => {
        write!(f, "I/O error; {}", s)
      }
      Error::MissingDB => {
        write!(f, "Missing required database file")
      }
      Error::Parse(s) => {
        write!(f, "Pasing error; {}", s)
      }
      Error::SQLite(s) => {
        write!(f, "SQLite error; {}", s)
      }
      Error::Verify => {
        write!(f, "Verification failed")
      }
    }
  }
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
