use std::path::{Path, PathBuf};

use crate::db;

use crate::err::Error;


pub struct RunContext<'a> {
  cmd: &'a str,
  args: &'a [String],

  /// Relative path and filename of database file.
  pub(crate) dbfile: PathBuf,

  /// The current subdirectory (relative to the base directory).
  pub(crate) subdir: Option<PathBuf>
}

impl<'a> RunContext<'a> {
  /// Create a new `Context` which assumes the current directory is the base
  /// directory.
  fn here(cmd: &'a str, args: &'a [String]) -> Result<Self, Error> {
    Ok(Self {
      cmd,
      args,
      dbfile: PathBuf::from(db::FNAME),
      subdir: None
    })
  }

  /// Initialize a new context.
  pub fn init(cmd: &'a str, args: &'a [String]) -> Result<Self, Error> {
    let ctx = Self::here(cmd, args)?;

    // When running init() the database file must not exist.
    if ctx.dbfile.is_file() {
      Err(Error::DbAlreadyExists)
    } else {
      Ok(ctx)
    }
  }

  /// Open up an existing `Context` at the current location.  Fail if the
  /// database can not be opened.
  pub fn open_here(cmd: &'a str, args: &'a [String]) -> Result<Self, Error> {
    let ctx = Self::here(cmd, args)?;

    // When running open_here() the database file must exist.
    if ctx.dbfile.is_file() {
      Ok(ctx)
    } else {
      Err(Error::MissingDB)
    }
  }

  /// Open up a new context, attempting to find the fidx database.  If it isn't
  /// found in the current directory, search upward in the directory tree until
  /// one is found (or fail if none can be found).
  ///
  /// If `switch` is set, it will change the current directory to the base
  /// directory, *after* determining the curdir.
  pub fn find_base(
    cmd: &'a str,
    args: &'a [String],
    switch: bool
  ) -> Result<Self, Error> {
    let fname = Path::new(db::FNAME);

    let (dbfile, subdir) = if fname.is_file() {
      // Found the database file in the current directory.
      (PathBuf::from(fname), None)
    } else {
      // Search for the database file.

      // Keep track of traversed directories.
      let mut v = Vec::new();

      // Keep track of the current directory.  Each leaf node will be poped off
      // this and put on the v stack until the database file is found.
      let mut curdir = std::env::current_dir()?;

      //
      // Traverse up the filesystem hierarchy until the database file has been
      // found.  Each time a step up has been performed, push the current
      // directory name onto the stack.
      //
      let dbfile = loop {
        // Get the current directory's name
        let dirname = Path::new(curdir.file_name().unwrap()).to_path_buf();

        // Push the directory name onto the stack
        v.push(dirname);

        // Step up one level
        curdir = curdir.parent().unwrap().to_path_buf();

        // Check if database file is in current directory
        let dbfile = curdir.join(db::FNAME);
        if dbfile.is_file() {
          // Found the database!
          if switch {
            std::env::set_current_dir(&curdir)?;
          }
          break dbfile;
        }
      };

      // The list of directories is in reverse order; create a new PathBuf
      // based on the reverse stack order.
      let mut subdir = PathBuf::new();
      for c in v.iter().rev() {
        subdir.push(c);
      }

      (dbfile, Some(subdir))
    };

    Ok(Self {
      cmd,
      args,
      dbfile,
      subdir
    })
  }


  /// Given an input path, generate a pathname as it would be found in the
  /// database.
  ///
  /// If the current subdirectory is `foo/bar` (relative the base directory)
  /// and the input path is `baz/file.txt` the output will be
  /// `foo/bar/baz/file.txt` (which is what the file's path should be in the
  /// database).
  ///
  /// # ToDo
  /// - Sanitize `pn`:
  ///   - Handle `./`
  ///   - Handle `..`
  pub fn gen_tree_path<P: AsRef<Path>>(&self, pn: P) -> PathBuf {
    if let Some(subdir) = &self.subdir {
      subdir.join(pn.as_ref())
    } else {
      PathBuf::from(pn.as_ref())
    }
  }
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
