use std::path::{Path, PathBuf};

use hashbrown::{HashMap, HashSet};

use rusqlite::{params, types::ToSql, Connection, OpenFlags};

use log::{debug, info, trace, warn};

use indicatif::ProgressBar;

#[cfg(unix)]
use std::os::unix::fs::MetadataExt;

use crate::runctx::RunContext;

use crate::err::Error;


pub const FNAME: &str = ".findex";


pub struct DB {
  pub conn: Connection
}

impl DB {
  /// Initialize a new fidx database.
  pub fn init(ctx: &RunContext) -> Result<Self, Error> {
    let db = Self::open(ctx)?;

    init(&db.conn)?;

    Ok(db)
  }

  /// Open a database in read/write mode.
  ///
  /// The database is created if it does not already exist.
  pub fn open(ctx: &RunContext) -> Result<Self, Error> {
    let conn = Connection::open(&ctx.dbfile)?;

    conn.pragma_update(None, "journal_mode", &"WAL")?;
    conn.pragma_update(None, "foreign_keys", &"ON")?;
    conn.pragma_update(None, "auto_vacuum", &"INCREMENTAL")?;

    Ok(DB { conn })
  }

  /// Open a database in read-only mode.
  ///
  /// It is implied that the database must exist.
  pub fn readonly(ctx: &RunContext) -> Result<Self, Error> {
    let flags = OpenFlags::SQLITE_OPEN_READ_ONLY;
    let conn = Connection::open_with_flags(&ctx.dbfile, flags)?;

    Ok(DB { conn })
  }


  /// Load ignore list from the database and put it into a [`HashSet`].
  pub fn get_ignore_set(&self) -> Result<HashSet<String>, Error> {
    get_ignore_set(&self.conn)
  }


  /// Verify file in the filesystem against the checksums in the database.
  ///
  /// If `scan` is `None` then a complete verifiction will be performed.  If it
  /// is `Some(pth)` then only items in the `pth` subtree will be checked (this
  /// can also perform single-file verifications if a file is specified).
  pub fn verify(&self, sub: Option<PathBuf>) -> Result<(), Error> {
    verify(&self.conn, sub)
  }


  /// Get a file entry from the database given a pathname.
  pub fn get_file_by_fname(
    &self,
    fname: &str
  ) -> Result<Option<DBFileEntry>, Error> {
    get_file_by_fname(&self.conn, fname)
  }


  /// Add a new tag to the database.
  pub fn add_tag(&self, title: &str) -> Result<i64, Error> {
    add_tag(&self.conn, title)
  }


  /// Remove a tag from the database.
  pub fn remove_tag(&self, tag: &str) -> Result<(), Error> {
    remove_tag(&self.conn, tag)
  }


  /// Rename a tag in the database.
  pub fn rename_tag(&self, tag: &str, new: &str) -> Result<(), Error> {
    rename_tag(&self.conn, tag, new)
  }


  /// Get a list of all tags in the database.
  pub fn get_tag_list(&self) -> Result<Vec<(i64, String)>, Error> {
    get_tags(&self.conn)
  }


  pub fn tag(&self, ufile_id: i64, tag: &str) -> Result<(), Error> {
    crate::db::tag(&self.conn, ufile_id, tag)
  }


  pub fn untag(&self, ufile_id: i64, tag: &str) -> Result<(), Error> {
    untag(&self.conn, ufile_id, tag)
  }


  pub fn add_file(
    &self,
    fname: &str,
    mtime: i64,
    ufile_id: i64
  ) -> Result<i64, Error> {
    add_file(&self.conn, fname, mtime, ufile_id)
  }


  pub fn file_modified(
    &self,
    file_id: i64,
    mtime: i64,
    ufile_id: i64
  ) -> Result<(), Error> {
    file_modified(&self.conn, file_id, mtime, ufile_id)
  }


  pub fn delete_stale_ufiles(&self) -> Result<(), Error> {
    delete_stale_ufiles(&self.conn)
  }


  pub fn insert_or_get_ufile(&self, hash: &str) -> Result<EntryId, Error> {
    insert_or_get_ufile(&self.conn, hash)
  }


  /// Get a list of tags for a unique file entry.
  pub fn get_ufile_tags(
    &self,
    ufile_id: i64
  ) -> Result<Vec<(i64, String)>, Error> {
    get_ufile_tags(&self.conn, ufile_id)
  }


  /// Given an iterator over tags, get a set of all unique file ids which match
  /// those tags.
  pub fn get_ufiles_by_tags<I, S>(
    &self,
    tags: I
  ) -> Result<HashSet<i64>, Error>
  where
    I: IntoIterator<Item = S>,
    S: AsRef<str>
  {
    get_ufiles_by_tags(&self.conn, tags)
  }


  /// Given a unique file id, get a list of all files referring to that unique
  /// file.
  pub fn get_files_by_ufile(
    &self,
    ufile_id: i64
  ) -> Result<Vec<(i64, String)>, Error> {
    get_files_by_ufile(&self.conn, ufile_id)
  }


  pub fn get_dup_ufiles(&self) -> Result<Vec<i64>, Error> {
    get_dup_ufiles(&self.conn)
  }


  /// Vacuum database.
  pub fn vacuum(&self) -> Result<(), Error> {
    self.conn.execute("VACUUM;", &[] as &[&dyn ToSql])?;
    Ok(())
  }


  pub fn migrate_tags(&self, old_id: i64, new_id: i64) -> Result<(), Error> {
    migrate_tags(&self.conn, old_id, new_id)
  }


  pub fn remove_missing(&self) -> Result<(), Error> {
    remove_missing(&self.conn)
  }


  pub fn remove_files_by_name(
    &self,
    ignore: &HashSet<String>
  ) -> Result<(), Error> {
    remove_files_by_name(&self.conn, ignore)
  }
}


fn init(conn: &Connection) -> Result<(), Error> {
  // Table of "unique" files -- specifically what file hashes exist in the file
  // system.
  //
  // id | hash
  // ---+-----
  //  1 | foo
  //  2 | baz
  conn.execute(
    "CREATE TABLE IF NOT EXISTS ufiles (
  id   INTEGER PRIMARY KEY,
  hash  TEXT UNIQUE NOT NULL,
  CHECK (length(hash) == 64)
);",
    &[] as &[&dyn ToSql]
  )?;

  conn.execute(
    "CREATE UNIQUE INDEX IF NOT EXISTS idx_ufiles_hash ON ufiles (hash);",
    &[] as &[&dyn ToSql]
  )?;


  // List of filenames and reference to their content's hash.
  // The mtime field is a unix timestamp (in seconds) and is used as an
  // optimization; if the mtime in the file system and the database match it
  // is assumed that the contents of the file has not been changed, so it is
  // not rehashed.  (This means that if the mtimes differ, the file will be
  // rehashed).
  //
  // The ufile_id is a reference to a "unique file" entry, which is determined
  // by the file content's hash.
  //
  // id | fname           | mtime | ufile_id
  // ---+-----------------+-------+---------
  //  1 | foo/bar/baz.txt |       |
  //  2 | cow.jpg         |       |
  //  3 | cats/meow.jpg   |       |
  conn.execute(
    "CREATE TABLE IF NOT EXISTS files (
  id   INTEGER PRIMARY KEY,
  fname TEXT UNIQUE NOT NULL,
  mtime INTEGER NOT NULL,
  ufile_id INTEGER NOT NULL REFERENCES ufiles(id) ON DELETE CASCADE,
  CHECK (length(fname) > 0)
);",
    &[] as &[&dyn ToSql]
  )?;

  conn.execute(
    "CREATE UNIQUE INDEX IF NOT EXISTS idx_files_fname ON files (fname);",
    &[] as &[&dyn ToSql]
  )?;


  // List of recognized tags.
  //
  // id | title
  // ---+------
  //  1 | foo
  //  2 | baz
  conn.execute(
    "CREATE TABLE IF NOT EXISTS tags (
  id   INTEGER PRIMARY KEY,
  title  TEXT UNIQUE NOT NULL,
  CHECK (length(title) > 0)
);",
    &[] as &[&dyn ToSql]
  )?;


  // Associate tags with file contents.
  // This means that a file's tags are lost when its contents change, which is
  // okay becase this tool is meant to immutable data.
  //
  // id | ufile_id | tag_id
  // ---+----------+--------
  //  1 | 1        | 1
  conn.execute(
    "CREATE TABLE IF NOT EXISTS ufiletags (
  id    INTEGER PRIMARY KEY,
  ufile_id INTEGER NOT NULL REFERENCES ufiles(id) ON DELETE CASCADE,
  tag_id INTEGER NOT NULL REFERENCES tags(id) ON DELETE CASCADE,
  UNIQUE(ufile_id, tag_id)
);",
    &[] as &[&dyn ToSql]
  )?;


  // List of pathnames to ignore.
  // These entries will be put through a glob:er at the very beginning of the
  // update process, so wildcards can be used.
  //
  // id | ignore
  // ---+---------------
  //  1 | **/.*.swp
  //  2 | ignorethis
  conn.execute(
    "CREATE TABLE IF NOT EXISTS ignore (
  id   INTEGER PRIMARY KEY,
  path TEXT NOT NULL UNIQUE,
  CHECK (length(path) > 0)
);",
    &[] as &[&dyn ToSql]
  )?;

  conn.execute(
    "INSERT OR IGNORE INTO ignore (path) VALUES (?);",
    params!(FNAME)
  )?;
  conn.execute(
    "INSERT OR IGNORE INTO ignore (path) VALUES (?);",
    params!(format!("{}-shm", FNAME))
  )?;
  conn.execute(
    "INSERT OR IGNORE INTO ignore (path) VALUES (?);",
    params!(format!("{}-wal", FNAME))
  )?;
  conn.execute(
    "INSERT OR IGNORE INTO ignore (path) VALUES (?);",
    params!(".findexer.log")
  )?;
  conn.execute(
    "INSERT OR IGNORE INTO ignore (path) VALUES (?);",
    params!("**/.*.swp")
  )?;


  Ok(())
}

pub struct DBFileEntry {
  pub id: i64,
  pub fname: String,
  pub mtime: i64,
  pub ufile_id: i64,
  pub hash: String
}


fn get_file_by_fname(
  conn: &Connection,
  fname: &str
) -> Result<Option<DBFileEntry>, Error> {
  const SQL: &str = r#"
SELECT f.id, f.mtime, f.ufile_id, uf.hash
FROM files AS f
LEFT JOIN ufiles AS uf ON f.ufile_id=uf.id
WHERE f.fname=?;
"#;
  let mut stmt = conn.prepare_cached(SQL)?;
  match stmt.query_row(params!(fname), |row| {
    Ok(DBFileEntry {
      id: row.get(0)?,
      fname: fname.to_string(),
      mtime: row.get(1)?,
      ufile_id: row.get(2)?,
      hash: row.get(3)?
    })
  }) {
    Ok(e) => Ok(Some(e)),
    Err(rusqlite::Error::QueryReturnedNoRows) => Ok(None),
    Err(e) => Err(e.into())
  }
}

pub fn get_ignore_set(conn: &Connection) -> Result<HashSet<String>, Error> {
  const SQL: &str = r#"
SELECT path
FROM ignore;
"#;
  let mut set = HashSet::new();
  let mut stmt = conn.prepare_cached(SQL)?;
  let mut rows = stmt.query([])?;
  while let Some(row) = rows.next()? {
    set.insert(row.get(0)?);
  }
  Ok(set)
}


pub enum EntryId {
  New(i64),
  Old(i64)
}


/// Given a hash, insert it if it is new and return the new id.  If the entry
/// already existed, return its id.
fn insert_or_get_ufile(
  conn: &Connection,
  hash: &str
) -> Result<EntryId, Error> {
  const SQL: &str = r#"
  SELECT id FROM ufiles WHERE hash=?;
"#;
  let mut stmt = conn.prepare_cached(SQL)?;

  //
  // Return existing id if one exists
  //
  match stmt.query_row(params!(hash), |row| Ok(EntryId::Old(row.get(0)?))) {
    Ok(e) => return Ok(e),
    Err(rusqlite::Error::QueryReturnedNoRows) => {}
    Err(e) => return Err(e.into())
  };

  //
  // Insert a new entry
  //
  const ISQL: &str = r#"
  INSERT INTO ufiles (hash) VALUES (?);
"#;
  let mut stmt = conn.prepare_cached(ISQL)?;
  stmt.execute(params![hash])?;

  let id = conn.last_insert_rowid();

  Ok(EntryId::New(id))
}


fn add_file(
  conn: &Connection,
  fname: &str,
  mtime: i64,
  ufile_id: i64
) -> Result<i64, Error> {
  const SQL: &str = r#"
INSERT INTO files (fname, mtime, ufile_id) VALUES (?, ?, ?);
"#;

  let mut stmt = conn.prepare_cached(SQL)?;
  stmt.execute(params![fname, mtime, ufile_id])?;

  Ok(conn.last_insert_rowid())
}


fn file_modified(
  conn: &Connection,
  file_id: i64,
  mtime: i64,
  ufile_id: i64
) -> Result<(), Error> {
  const SQL: &str = r#"
UPDATE files
SET mtime=?, ufile_id=?
WHERE id=?;
"#;

  let mut stmt = conn.prepare_cached(SQL)?;
  stmt.execute(params![mtime, ufile_id, file_id])?;

  Ok(())
}


/// Remove all `ufiles` which are no longer referenced to from `files`.
fn delete_stale_ufiles(conn: &Connection) -> Result<(), Error> {
  const SQL: &str = r#"
DELETE FROM ufiles
WHERE id IN (
  SELECT id
  FROM ufiles
  LEFT JOIN (
    SELECT ufile_id,COUNT(ufile_id) AS cnt
    FROM files
    GROUP BY ufile_id
  ) AS C
  ON C.ufile_id=id
  WHERE C.cnt IS NULL
  GROUP BY id
);
"#;

  let mut stmt = conn.prepare_cached(SQL)?;
  let n = stmt.execute(params![])?;

  info!("Removed {} ufiles rows", n);

  Ok(())
}


fn add_tag(conn: &Connection, title: &str) -> Result<i64, Error> {
  const SQL: &str = r#"
INSERT INTO tags (title) VALUES (?);
"#;

  let mut stmt = conn.prepare_cached(SQL)?;
  stmt.execute(params![title])?;

  Ok(conn.last_insert_rowid())
}

fn remove_tag(conn: &Connection, tag: &str) -> Result<(), Error> {
  if let Ok(id) = tag.parse::<i64>() {
    const SQL: &str = r#"
DELETE FROM tags where id=?;
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![id])?;
  } else {
    const SQL: &str = r#"
DELETE FROM tags where title=?;
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![tag])?;
  }

  Ok(())
}

fn rename_tag(conn: &Connection, tag: &str, new: &str) -> Result<(), Error> {
  if let Ok(id) = tag.parse::<i64>() {
    const SQL: &str = r#"
UPDATE tags SET title=? WHERE id=?;
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![new, id])?;
  } else {
    const SQL: &str = r#"
UPDATE tags SET title=? WHERE title=?;
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![new, tag])?;
  }

  Ok(())
}


fn get_tags(conn: &Connection) -> Result<Vec<(i64, String)>, Error> {
  const SQL: &str = r#"
SELECT id, title
FROM tags
ORDER BY title;
"#;
  let mut tags = Vec::new();
  let mut stmt = conn.prepare_cached(SQL)?;
  let mut rows = stmt.query([])?;
  while let Some(row) = rows.next()? {
    let id: i64 = row.get(0)?;
    let title: String = row.get(1)?;
    tags.push((id, title));
  }
  Ok(tags)
}


fn get_ufile_tags(
  conn: &Connection,
  ufile_id: i64
) -> Result<Vec<(i64, String)>, Error> {
  const SQL: &str = r#"
SELECT t.id,t.title
FROM ufiletags AS uft
LEFT JOIN tags AS t
ON t.id=uft.tag_id
WHERE uft.ufile_id=?
ORDER BY t.title;
"#;
  let mut tags = Vec::new();
  let mut stmt = conn.prepare_cached(SQL)?;
  let mut rows = stmt.query([ufile_id])?;
  while let Some(row) = rows.next()? {
    let id: i64 = row.get(0)?;
    let title: String = row.get(1)?;
    tags.push((id, title));
  }
  Ok(tags)
}


fn tag(conn: &Connection, ufile_id: i64, tag: &str) -> Result<(), Error> {
  if let Ok(id) = tag.parse::<i64>() {
    const SQL: &str = r#"
INSERT INTO ufiletags (ufile_id, tag_id) VALUES (?, ?);
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![ufile_id, id])?;
  } else {
    const SQL: &str = r#"
INSERT INTO ufiletags (ufile_id, tag_id)
VALUES (?, (SELECT id FROM tags WHERE title=?));
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![ufile_id, tag])?;
  }

  Ok(())
}

fn untag(conn: &Connection, ufile_id: i64, tag: &str) -> Result<(), Error> {
  if let Ok(id) = tag.parse::<i64>() {
    const SQL: &str = r#"
DELETE FROM ufiletags WHERE ufile_id=? AND tag_id=?;
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![ufile_id, id])?;
  } else {
    const SQL: &str = r#"
DELETE FROM ufiletags
WHERE ufile_id=? AND tag_id=(SELECT id FROM tags WHERE title=?);
"#;
    let mut stmt = conn.prepare_cached(SQL)?;
    stmt.execute(params![ufile_id, tag])?;
  }

  Ok(())
}


fn get_dup_ufiles(conn: &Connection) -> Result<Vec<i64>, Error> {
  const SQL: &str = r#"
SELECT c.id FROM (
  SELECT ufile_id AS id,COUNT(ufile_id) AS cnt
  FROM files
  GROUP BY ufile_id
) AS c
WHERE c.cnt > 1;
"#;
  let mut has_dups = Vec::new();
  let mut stmt = conn.prepare_cached(SQL)?;
  let mut rows = stmt.query([])?;
  while let Some(row) = rows.next()? {
    let id: i64 = row.get(0)?;
    has_dups.push(id);
  }
  Ok(has_dups)
}


fn get_files_by_ufile(
  conn: &Connection,
  ufile_id: i64
) -> Result<Vec<(i64, String)>, Error> {
  const SQL: &str = r#"
SELECT f.id,f.fname
FROM files AS f
WHERE f.ufile_id=?
ORDER BY f.fname;
"#;
  let mut lst = Vec::new();
  let mut stmt = conn.prepare_cached(SQL)?;
  let mut rows = stmt.query([ufile_id])?;
  while let Some(row) = rows.next()? {
    let id: i64 = row.get(0)?;
    let fname: String = row.get(1)?;
    lst.push((id, fname));
  }
  Ok(lst)
}


/// Given an iterator of tags, find all ufiles matching those tags.
fn get_ufiles_by_tags<I, S>(
  conn: &Connection,
  tags: I
) -> Result<HashSet<i64>, Error>
where
  I: IntoIterator<Item = S>,
  S: AsRef<str>
{
  //
  // Map which is key'd by tags and where the values are HashSet's of unique
  // file ids matching those tags.
  //
  let mut sets = HashMap::new();

  //
  // Iterate over tags in search set, find all unique file entries matching
  // that tag.  Add an entry to the sets map, mapping the tag to a set of
  // unique files
  //
  for tag in tags.into_iter() {
    if let Ok(id) = tag.as_ref().parse::<i64>() {
      //
      // Generate a set of all unique files having this tag
      //
      const SQL: &str = r#"
SELECT ufile_id FROM ufiletags WHERE tag_id=?;
"#;
      let mut stmt = conn.prepare_cached(SQL)?;
      let mut rows = stmt.query([id])?;
      let mut set = HashSet::new();
      while let Some(row) = rows.next()? {
        let id: i64 = row.get(0)?;
        set.insert(id);
      }

      //
      // Add entry mapping this tag to the set of unique files having the tag
      //
      sets.insert(tag.as_ref().to_string(), set);
    } else {
      //
      // Generate a set of all unique files having this tag
      //
      const SQL: &str = r#"
SELECT ufile_id
FROM ufiletags
WHERE tag_id=(SELECT id FROM tags WHERE title=?);
"#;
      let mut stmt = conn.prepare_cached(SQL)?;
      let mut rows = stmt.query([tag.as_ref()])?;
      let mut set = HashSet::new();
      while let Some(row) = rows.next()? {
        let id: i64 = row.get(0)?;
        set.insert(id);
      }

      //
      // Add entry mapping this tag to the set of unique files having the tag
      //
      sets.insert(tag.as_ref().to_string(), set);
    }
  }

  //
  // At this point the sets HashMap will contain tags (used as keys)
  // associated with a HashSet of all the unique file id's associated with
  // that tag.
  //

  //
  // Convert the "sets" HashMap into an iterator, then pull off the first
  // set of unique files it and set it as the current result set (to be
  // returned).
  // Return an empty HashSet if there were no entries in sets.
  //
  let mut it = sets.into_iter();
  let mut res = if let Some((_, set)) = it.next() {
    set
  } else {
    // No entries, return an empty set
    HashSet::new()
  };


  //
  // The current result set "res" needs to be constrained to only include
  // unique files matches for all other tags.  Iterate through the rest of the
  // HashSets and only retain entries in "res" which exist in all other
  // HashSets.
  //
  for (_, set) in it {
    res.retain(|k| set.contains(k));
  }

  Ok(res)
}


pub enum CheckFail {
  FileMissing(String),
  GetMetadata(String),
  HashingFailed(String),
  HashMismatch(String)
}

/// Iterate over files in the database and make sure their hashes match.
///
/// An assumption is made that the filesystem and the database have not
/// (intentionally) been modified since the last update.
fn verify(conn: &Connection, sub: Option<PathBuf>) -> Result<(), Error> {
  //
  // Count number of entries we'll be processing in the database and
  // create a progress bar with that value.
  //
  let num_files = get_num_files(conn, &sub)?;
  let pb = ProgressBar::new(num_files);

  //
  // Keep track of inodes encountered for files with nlinks > 1
  //
  //let mut checked_inodes = HashSet::new();

  let mut fail = Vec::new();

  let mut vals: Vec<&dyn ToSql> = Vec::new();

  // Need to keep this here to extend its scope
  let sub2;

  //let mut vals: Vec<&dyn ToSql> = Vec::new();
  let q = if let Some(sub) = &sub {
    sub2 = sub.to_str().unwrap();

    vals.push(&sub2);

    debug!("verify subset: {}", sub2);
    r#"
SELECT f.id, f.ufile_id, f.fname, uf.hash
FROM files AS f
LEFT JOIN ufiles AS uf ON f.ufile_id=uf.id
WHERE f.fname LIKE ? || '%'
ORDER BY f.fname;
"#
  } else {
    debug!("verify all");

    r#"
SELECT f.id, f.ufile_id, f.fname, uf.hash
FROM files AS f
LEFT JOIN ufiles AS uf ON f.ufile_id=uf.id
ORDER BY f.fname;
"#
  };

  let mut stmt = conn.prepare_cached(q)?;

  let mut rows = stmt.query(&vals[..])?;

  //
  // Keep track of inodes for files with > 1 nlinks
  //
  let mut checked_inodes = HashSet::new();

  while let Some(row) = rows.next()? {
    let fname: String = row.get(2)?;
    let dbhash: String = row.get(3)?;

    trace!("verify: {}", fname);

    // Make progress in progress bar
    pb.inc(1);

    // Generate a Path object and check if the file exists
    let p = Path::new(&fname);
    if !p.is_file() {
      eprintln!("Missing file: {}", fname);
      fail.push(CheckFail::FileMissing(fname));
      continue;
    }

    //
    // Get the file's metadata (so we can probe inode information)
    //
    let meta = match std::fs::metadata(p) {
      Ok(meta) => meta,
      Err(e) => {
        warn!("Skipping file {}; {}", fname, e);
        fail.push(CheckFail::GetMetadata(fname));
        continue;
      }
    };

    //
    // Don't recheck the same inodes
    //
    let (n_hard_links, inode) = get_hardlink_meta(&meta);
    if n_hard_links > 1 {
      trace!("File {} has {} hard links", fname, n_hard_links);

      if checked_inodes.contains(&inode) {
        // Already checked this inode -- skip it
        trace!("Already checked this inode -- skipping");
        continue;
      }

      // Remember that this inode has been checked already
      checked_inodes.insert(inode);
    }

    //
    // Calculate file's hash
    //
    let fhash = match crate::hashfile(p) {
      Ok(hash) => hash,
      Err(_) => {
        eprintln!("Unable to hash : {}", fname);
        fail.push(CheckFail::HashingFailed(fname));
        continue;
      }
    };

    if fhash != dbhash {
      eprintln!("Hash mismatch for: {}", fname);
      fail.push(CheckFail::HashMismatch(fname));
      continue;
    }
  }

  pb.finish_with_message("done");

  if fail.is_empty() {
    Ok(())
  } else {
    Err(Error::Verify)
  }
}


#[cfg(unix)]
fn get_hardlink_meta(meta: &std::fs::Metadata) -> (u64, u64) {
  let n_hard_links = meta.nlink();
  let inode = meta.ino();

  (n_hard_links, inode)
}

#[cfg(not(unix))]
fn get_hardlink_meta(meta: &std::fs::Metadata) -> (u64, u64) {
  (1, 0)
}


/// Get the number of file entries in the files table.
fn get_num_files(
  conn: &Connection,
  sub: &Option<PathBuf>
) -> Result<u64, Error> {
  if let Some(sub) = sub {
    //
    // Count subset
    //
    const SQL: &str = r#"
SELECT COUNT(id) FROM files WHERE fname LIKE ? || '%';
"#;

    // ToDo: Don't unwrap()
    let sub = sub.to_str().unwrap();

    let mut stmt = conn.prepare_cached(SQL)?;
    match stmt.query_row([sub], |row| row.get(0)) {
      Ok(n) => Ok(n),
      Err(e) => Err(e.into())
    }
  } else {
    //
    // Count all files
    //
    const SQL: &str = r#"
SELECT COUNT(id) FROM files;
"#;

    let mut stmt = conn.prepare_cached(SQL)?;
    match stmt.query_row([], |row| row.get(0)) {
      Ok(n) => Ok(n),
      Err(e) => Err(e.into())
    }
  }
}


fn remove_missing(conn: &Connection) -> Result<(), Error> {
  const SQL: &str = r#"
SELECT id, fname
FROM files;
"#;
  const DSQL: &str = r#"
DELETE FROM files WHERE id=?;
"#;

  //
  // Gather a list of file id's whose paths entries do not point to a file in
  // the file system.  (Presumably because that file has been removed).
  //
  let mut dellist = Vec::new();
  let mut stmt = conn.prepare_cached(SQL)?;
  let mut rows = stmt.query([])?;
  while let Some(row) = rows.next()? {
    let id: i64 = row.get(0)?;
    let fname: String = row.get(1)?;

    let p = Path::new(&fname);
    if !p.is_file() {
      debug!("Missing file {} -- mark for deletion", fname);
      dellist.push((id, fname));
    }
  }

  if !dellist.is_empty() {
    let mut stmt = conn.prepare_cached(DSQL)?;
    for (id, fname) in dellist {
      trace!("Removing file {} ({})", id, fname);
      stmt.execute(params![id])?;
    }
  }

  Ok(())
}


fn remove_files_by_name(
  conn: &Connection,
  ignore: &HashSet<String>
) -> Result<(), Error> {
  const SQL: &str = r#"
DELETE FROM files WHERE fname=?;
"#;
  let mut stmt = conn.prepare_cached(SQL)?;

  for ig in ignore {
    stmt.execute(params![ig])?;
  }

  Ok(())
}


/// Given two unique file ids, one "old" and one "new", migrate all tags
/// associated with the "old" entry to the "one" one.
fn migrate_tags(
  conn: &Connection,
  old_id: i64,
  new_id: i64
) -> Result<(), Error> {
  const SQL: &str = r#"
SELECT tag_id
FROM ufiletags
WHERE ufile_id=?;
"#;
  let mut lst = Vec::new();
  let mut stmt = conn.prepare_cached(SQL)?;
  let mut rows = stmt.query([old_id])?;
  while let Some(row) = rows.next()? {
    let id: i64 = row.get(0)?;
    lst.push(id);
  }

  const ISQL: &str = r#"
INSERT INTO ufiletags (ufile_id, tag_id) VALUES (?, ?);
"#;
  let mut stmt = conn.prepare_cached(ISQL)?;
  for tag_id in lst {
    // ToDo: Ignore collisions
    stmt.execute(params![new_id, tag_id])?;
  }

  Ok(())
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
