use cosmwasm_std::{Addr, StdError, StdResult, Storage};
use cw_storage_plus::{Key, Map, PrimaryKey};

use crate::access_control::role::Role;

/// `AddrRole` is a custom object that is the combination of an `Addr` and `Role` reference.
///
/// This is intended to be an internal implementation detail to exercise the multi-level storage
/// system in a convenient way
///
#[derive(Debug)]
struct AddrRole<'a> {
    addr: &'a Addr,
    role: &'a Role,
}

impl<'a> AddrRole<'a> {
    /// Create a new `AddrRole` from a specified `Addr` and `Role` references
    fn new(addr: &'a Addr, role: &'a Role) -> AddrRole<'a> {
        AddrRole { addr, role }
    }
}

impl<'a> PrimaryKey<'a> for &'a AddrRole<'a> {
    type Prefix = ();
    type SubPrefix = ();
    type Suffix = ();
    type SuperSuffix = ();

    /// returns a slice of key steps, which can be optionally combined
    fn key(&self) -> Vec<Key> {
        vec![
            Key::Ref(self.addr.as_bytes()),
            Key::Ref(self.role.as_bytes()),
        ]
    }
}

const ROLES_MAP: Map<&AddrRole, ()> = Map::new("access_control:roles");

/// Check to see if a specified `Addr` has the queried `Role`
///
/// # Arguments
///
/// * `storage` The reference to the underlying storage engine
/// * `addr` The reference to the address to query
/// * `role` The reference to the role to query
///
/// Will return an `Err` if there is a underlying storage issue, otherwise true / false
///
pub fn has_role(storage: &dyn Storage, addr: &Addr, role: &Role) -> StdResult<bool> {
    let res = ROLES_MAP.may_load(storage, &AddrRole::new(addr, role))?;
    Ok(res.is_some())
}

/// Ensure that a specified `Addr` has been granted the queried `Role`
///
/// # Arguments
///
/// * `storage` The reference to the underlying storage engine
/// * `addr` The reference to the address to query
/// * `role` The reference to the role to query
///
/// Will return an `Err` if there is an underlying storage issue or if the specified `Addr` has not
/// be granted the specified `Role`
///
pub fn ensure_has_role(storage: &dyn Storage, addr: &Addr, role: &Role) -> StdResult<()> {
    let result = has_role(storage, addr, role)?;

    if !result {
        return Err(StdError::generic_err("Role lookup failed"));
    }

    Ok(())
}

/// Adds a `Role` for the specified `Addr`
///
/// * `storage` The reference to the underlying storage engine
/// * `addr` The reference to the address to update
/// * `role` The reference to the role to added
///
/// Will return an `Err` if there is a underlying storage issue
///
pub fn add_role(storage: &mut dyn Storage, addr: &Addr, role: &Role) -> StdResult<()> {
    ROLES_MAP.save(storage, &AddrRole::new(addr, role), &())?;

    Ok(())
}

/// Removes a `Role` for the specified `Addr`
///
/// * `storage` The reference to the underlying storage engine
/// * `addr` The reference to the address to update
/// * `role` The reference to the role to removed
///
/// Will return an `Err` if there is a underlying storage issue
///
pub fn remove_role(storage: &mut dyn Storage, addr: &Addr, role: &Role) -> StdResult<()> {
    ROLES_MAP.remove(storage, &AddrRole::new(addr, role));

    Ok(())
}
