use cosmwasm_std::StdError;
use hmac_sha256::Hash;

/// `Role` is a convenience object that represents the SHA256 digest of a string.
///
/// For runtime performance all
///
/// # Examples
///
/// It is expected that by default users will construct roles from their string representations
///
/// ```
/// use fetchai_std::access_control::Role;
/// let admin_role = Role::from("admin");
/// ```
///
/// It is however to construct the role object from a slice reference so long as the size of the
/// slice has the correct length
///
/// ```
/// use fetchai_std::access_control::Role;
/// let role = Role::try_from([1u8; 32].as_slice()).unwrap();
/// ```
///
/// Or from a correctly sized array:
///
/// ```
/// use fetchai_std::access_control::Role;
/// let role = Role::from([1u8; 32]);
/// ```
#[derive(Debug)]
pub struct Role {
    key: [u8; 32],
}

impl From<&str> for Role {
    fn from(value: &str) -> Self {
        // compute the digest of the role
        let mut h = Hash::new();
        h.update(value.as_bytes());
        let key = h.finalize();

        Role { key }
    }
}

impl From<[u8; 32]> for Role {
    fn from(value: [u8; 32]) -> Self {
        Role { key: value }
    }
}

impl TryFrom<&[u8]> for Role {
    type Error = StdError;

    fn try_from(value: &[u8]) -> Result<Self, Self::Error> {
        let key = value
            .try_into()
            .map_err(|_| StdError::generic_err("Unable to convert slice into role"))?;

        Ok(Role { key })
    }
}

impl Role {
    pub fn as_bytes(&self) -> &[u8] {
        self.key.as_slice()
    }
}
