use serde_derive::*;

#[derive(Serialize, Deserialize, Debug)]
pub struct WebHookEmbed {
    pub title: Option<String>,
    pub description: Option<String>,
    pub color: Option<u32>,
    pub author: Option<WebHookEmbedAuthor>,
    pub fields: Option<Vec<WebHookEmbedField>>,
    pub footer: Option<WebHookEmbedFooter>,
    pub image: Option<WebHookEmbedImage>,
    pub thumbnail: Option<WebHookEmbedThumbnail>,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct WebHookEmbedAuthor {
    pub name: Option<String>,
    pub icon_url: Option<String>,
    pub url: Option<String>,
}

impl WebHookEmbedAuthor {
    pub fn new() -> WebHookEmbedAuthor {
        WebHookEmbedAuthor {
            name: None,
            icon_url: None,
            url: None,
        }
    }

    pub fn name<T: AsRef<str>>(mut self, name: T) -> WebHookEmbedAuthor {
        self.name = Some(name.as_ref().to_string());
        self
    }

    pub fn icon_url<T: AsRef<str>>(mut self, icon_url: T) -> WebHookEmbedAuthor {
        self.icon_url = Some(icon_url.as_ref().to_string());
        self
    }

    pub fn url<T: AsRef<str>>(mut self, url: T) -> WebHookEmbedAuthor {
        self.url = Some(url.as_ref().to_string());
        self
    }
}

#[derive(Serialize, Deserialize, Debug)]
pub struct WebHookEmbedField {
    pub name: Option<String>,
    pub value: Option<String>,
    pub inline: Option<bool>,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct WebHookEmbedFooter {
    pub text: Option<String>,
    pub icon_url: Option<String>,
}

impl WebHookEmbedFooter {
    pub fn new() -> WebHookEmbedFooter {
        WebHookEmbedFooter {
            text: None,
            icon_url: None,
        }
    }

    pub fn text<T: AsRef<str>>(mut self, text: T) -> WebHookEmbedFooter {
        self.text = Some(text.as_ref().to_string());
        self
    }

    pub fn icon_url<T: AsRef<str>>(mut self, icon_url: T) -> WebHookEmbedFooter {
        self.icon_url = Some(icon_url.as_ref().to_string());
        self
    }
}

#[derive(Serialize, Deserialize, Debug)]
pub struct WebHookEmbedImage {
    pub url: Option<String>,
}

impl WebHookEmbedImage {
    pub fn new() -> WebHookEmbedImage {
        WebHookEmbedImage { url: None }
    }

    pub fn url<T: AsRef<str>>(mut self, url: T) -> WebHookEmbedImage {
        self.url = Some(url.as_ref().to_string());
        self
    }
}

#[derive(Serialize, Deserialize, Debug)]
pub struct WebHookEmbedThumbnail {
    pub url: Option<String>,
}

impl WebHookEmbedThumbnail {
    pub fn new() -> WebHookEmbedThumbnail {
        WebHookEmbedThumbnail { url: None }
    }

    pub fn url<T: AsRef<str>>(mut self, url: T) -> WebHookEmbedThumbnail {
        self.url = Some(url.as_ref().to_string());
        self
    }
}

impl WebHookEmbed {
    pub fn new() -> WebHookEmbed {
        WebHookEmbed {
            title: None,
            description: None,
            color: None,
            author: None,
            fields: None,
            footer: None,
            image: None,
            thumbnail: None,
        }
    }

    pub fn title<T: AsRef<str>>(mut self, title: T) -> WebHookEmbed {
        self.title = Some(title.as_ref().to_string());
        self
    }

    pub fn description<T: AsRef<str>>(mut self, description: T) -> WebHookEmbed {
        self.description = Some(description.as_ref().to_string());
        self
    }

    pub fn color(mut self, color: u32) -> WebHookEmbed {
        self.color = Some(color);
        self
    }

    pub fn image<T: AsRef<str>>(mut self, url: T) -> WebHookEmbed {
        self.image = Some(WebHookEmbedImage {
            url: Some(url.as_ref().to_string()),
        });
        self
    }

    pub fn thumbnail<T: AsRef<str>>(mut self, url: T) -> WebHookEmbed {
        self.thumbnail = Some(WebHookEmbedThumbnail {
            url: Some(url.as_ref().to_string()),
        });
        self
    }

    pub fn author<F>(mut self, author: F) -> WebHookEmbed
    where
        F: FnOnce(WebHookEmbedAuthor) -> WebHookEmbedAuthor,
    {
        self.author = Some(author(WebHookEmbedAuthor {
            name: None,
            icon_url: None,
            url: None,
        }));
        self
    }

    pub fn footer<F>(mut self, footer: F) -> WebHookEmbed
    where
        F: FnOnce(WebHookEmbedFooter) -> WebHookEmbedFooter,
    {
        self.footer = Some(footer(WebHookEmbedFooter {
            text: None,
            icon_url: None,
        }));
        self
    }

    pub fn add_field<T: AsRef<str>>(mut self, name: T, value: T, inline: bool) -> WebHookEmbed {
        {
            if let Some(mut e) = self.fields {
                e.push(WebHookEmbedField {
                    name: Some(name.as_ref().to_string()),
                    value: Some(value.as_ref().to_string()),
                    inline: Some(inline),
                });
                self.fields = Some(e);
            } else {
                self.fields = Some(vec![WebHookEmbedField {
                    name: Some(name.as_ref().to_string()),
                    value: Some(value.as_ref().to_string()),
                    inline: Some(inline),
                }]);
            }
            self
        }
    }
}
