use serenity::{
    async_trait,
    model::{
        channel::Message,
        gateway::{Activity, Ready},
    },
    prelude::*,
};
use tokio;

use ferris_bot::cmd::{core, moderation};
use ferris_bot::config::conf;

/// Handler for events from the discord API
struct Handler;

#[async_trait]
impl EventHandler for Handler {
    async fn message(&self, ctx: Context, msg: Message) {
        if msg.content.as_str().chars().nth(0).unwrap() == '-' {
            let parts: Vec<&str> = msg.content.split_whitespace().collect();
            if parts[0].chars().nth(0) == Some('-') {
                if let Err(why) = match parts[0] {
                    // core commands
                    "-ping" => msg.channel_id.say(&ctx.http, core::ping()).await,
                    "-help" => {
                        msg.channel_id
                            .say(
                                &ctx.http,
                                core::help(
                                    conf::load_config("config.toml")
                                        .expect("expected a config file"),
                                )
                                .as_str(),
                            )
                            .await
                    }
                    "-getid" => {
                        msg.channel_id
                            .say(&ctx.http, core::get_id(msg).as_str())
                            .await
                    }
                    "-status" => {
                        msg.channel_id
                            .say(
                                &ctx.http,
                                core::status(
                                    conf::load_config("config.toml")
                                        .expect("expected a config file"),
                                )
                                .as_str(),
                            )
                            .await
                    }

                    // moderation commands
                    "-warn" => {
                        match conf::load_config("config.toml").expect("expected a config file").moderation {
                            Some(true) => {
                                msg.channel_id
                                .say(
                                    &ctx.http,
                                    moderation::warn(
                                        msg,
                                        conf::load_config("config.toml")
                                            .expect("expected a config file"),
                                        &ctx,
                                    )
                                    .await
                                    .as_str(),
                                )
                                .await
                            }
                            Some(false) | None => {
                                msg.channel_id.say(&ctx.http, "moderation has not been enabled. See `-status` for which modules are enabled").await
                            }
                        }
                    }
                    "-kick" => {
                        match conf::load_config("config.toml").expect("expected a config file").moderation {
                            Some(true) => {
                                msg.channel_id
                                .say(
                                    &ctx.http,
                                    moderation::kick(
                                        msg,
                                        conf::load_config("config.toml")
                                            .expect("expected a config file"),
                                        &ctx,
                                    )
                                    .await
                                    .as_str(),
                                )
                                .await
                            }
                            Some(false) | None => {
                                msg.channel_id.say(&ctx.http, "moderation has not been enabled. See `-status` for which modules are enabled").await
                            }
                        }
                    }
                    _ => Ok(msg),
                } {
                    println!("Error sending message: {:?}", why);
                }
            }
        }
    }

    async fn ready(&self, ctx: Context, ready: Ready) {
        ctx.set_activity(Activity::playing("-help")).await;
        println!("{} is connected!", ready.user.name);
    }
}

#[tokio::main]
async fn main() {
    let token = conf::load_config("config.toml").expect("Expected a config file");
    let token = token.token;
    let mut client = Client::builder(&token)
        .event_handler(Handler)
        .await
        .expect("Err creating client");

    if let Err(why) = client.start().await {
        println!("Client error: {:?}", why);
    }
}
