/// core commands
pub mod core {
    use crate::config::{conf::Config, val};
    use serenity::model::channel::Message;
    /// respond with pong (left in for debug purposes)
    pub fn ping() -> &'static str {
        "**Pong!**"
    }

    /// send a small help page to the user
    ///
    /// # Arguments
    ///
    /// * `conf`: the config to use
    ///
    /// # Return
    /// returns the help message as a String
    pub fn help(conf: Config) -> String {
        let mut help_string: String = String::from(
            "**HELP**
**core**
`-ping`: sends pong (used for debug purposes)
`-help`: sends this help message
`-status`: show the status of the bot (which modules are loaded)
`-getid [user]`: gets the user id of `user` or (if not specified) the sender",
        );
        match conf.moderation {
            // feature has been enabled in the config file
            Some(true) => {
                help_string.push_str(
                    "\n**moderation**
`-kick <user> [reason]`: kicks a user
`-ban <user> [reason]`: bans a user
`-warn <user> [reason]`: warns a user
`-mute <user> [reason]`: mutes a user
a `<user>` is represented by his ID
",
                );
            }
            // if set to false or not specified (default value) don't do anything
            // since the feature is disabled
            Some(false) | None => {}
        }
        help_string.push_str(format!("\nversion: {}", val::VERSION).as_str());
        help_string
    }

    /// get the ID of some user
    ///
    /// # Arguments
    ///
    /// * `msg`: the message to parse
    ///
    /// # Return
    ///
    /// returns the response content
    pub fn get_id(msg: Message) -> String {
        let components: Vec<&str> = msg.content.as_str().split_whitespace().collect();
        if components.len() == 1 {
            return format!("user ID of {}: {}", msg.author.name, msg.author.id.as_u64());
        } else if components.len() > 1 {
            if msg.mentions.len() > 0 {
                let mut out: String = String::new();
                for user in msg.mentions {
                    out.push_str(
                        format!("user ID of {}: {}\n", user.name, user.id.as_u64()).as_str(),
                    );
                }
                return out;
            } else {
                return String::from("if you want the ID of some user @mention them");
            }
        } else {
            return String::new();
        }
    }

    /// function to report the status of the bot
    ///
    /// # Arguments
    ///
    /// * `config`: the config to use
    ///
    /// # Return
    ///
    /// returns the message to send as a String
    pub fn status(config: Config) -> String {
        let mut out: String = String::from("**core**: enabled");
        match config.moderation {
            Some(true) => out.push_str("\n**moderation**: enabled"),
            Some(false) | None => out.push_str("\n**moderation**: disabled"),
        }
        out.push_str(val::VERSION);
        out
    }
}

/// moderation commands
pub mod moderation {
    use crate::config::conf::Config;
    use serenity::{
        client::Context,
        model::{
            channel::Message,
            id::{GuildId, RoleId},
            user::User,
        },
    };
    /// warn a user
    ///
    /// # Arguments
    ///
    /// * `msg`: the message to parse
    /// * `conf`: the configuration to use
    /// * `ctx`: the context to use for some checking things
    ///
    /// # Return
    ///
    /// The message to send as a String
    pub async fn warn(msg: Message, conf: Config, ctx: &Context) -> String {
        let context: Context = ctx.clone();
        let gid: GuildId = match msg.guild_id {
            Some(n) => n,
            None => {
                return String::from("you need to send this message in a server");
            }
        };
        let modrole: RoleId = match conf.moderating {
            Some(n) => RoleId(n.modrole),
            None => {
                return String::from("Modrole not specified in the config file");
            }
        };
        match msg.author.has_role(context, gid, modrole).await {
            Ok(true) => {
                if msg.mentions.len() != 1 {
                    return String::from("you need to mention the person you want to warn");
                } else {
                    let user: &User = &msg.mentions[0];
                    let parts: Vec<&str> = msg.content.as_str().split_whitespace().collect();
                    if parts.len() > 2 {
                        let reason_vec: Vec<&str> = parts[2..].to_vec();
                        let mut reason: String = String::new();
                        for i in reason_vec {
                            reason = format!("{} {}", reason, i);
                        }
                        reason = reason[1..].to_string();
                        return format!("{} has been warned. Reason: {}", user.name, reason);
                    } else {
                        return format!("{} has been warned", user.name);
                    }
                }
            }

            Ok(false) => {
                return String::from("you need to be a moderator to execute this command");
            }

            Err(e) => {
                return format!("Error while checking permissions: `{}`", e);
            }
        }
    }
    
    /// kick a user
    ///
    /// # Arguments
    ///
    /// * `msg`: the message to parse
    /// * `conf`: the configuration to use
    /// * `ctx`: the context to use for some checking things
    ///
    /// # Return
    ///
    /// The message to send as a String
    pub async fn kick(msg: Message, conf: Config, ctx: &Context) -> String {
        let context: Context = ctx.clone();
        let gid: GuildId = match msg.guild_id {
            Some(n) => n,
            None => {
                return String::from("you need to send this message in a server");
            }
        };
        let modrole: RoleId = match conf.moderating {
            Some(n) => RoleId(n.modrole),
            None => {
                return String::from("Modrole not specified in the config file");
            }
        };
        match msg.author.has_role(&context, gid, modrole).await {
            Ok(true) => {
                if msg.mentions.len() != 1 {
                    return String::from("you need to mention the person you want to kick");
                } else {
                    let user: &User = &msg.mentions[0];
                    let parts: Vec<&str> = msg.content.as_str().split_whitespace().collect();
                    if parts.len() > 2 {
                        let reason_vec: Vec<&str> = parts[2..].to_vec();
                        let mut reason: String = String::new();
                        for i in reason_vec {
                            reason = format!("{} {}", reason, i);
                        }
                        reason = reason[1..].to_string();
                        if let Err(e) = gid.kick(&context, user.id).await {
                            println!("failed to kick user: {:?}", e);
                            return format!("failed to kick user: {}", e);
                        }
                        return format!("{} has been kicked. Reason: {}", user.name, reason);
                    } else {
                        if let Err(e) = gid.kick(&context, user.id).await {
                            println!("failed to kick user: {:?}", e);
                            return format!("failed to kick user: {}", e);
                        }
                        return format!("{} has been kicked", user.name);
                    }
                }
            }

            Ok(false) => {
                return String::from("you need to be a moderator to execute this command");
            }

            Err(e) => {
                return format!("Error while checking permissions: `{}`", e);
            }
        }
    }
}
