use super::atoms::*;

use crate::builder::common::keywords::*;

use std::iter::{Peekable, Enumerate};
use std::str::Chars;
use std::fmt::{self, Write, Display};
use std::ascii;

use regex::Regex;

pub fn is_id_begin(ch: char) -> bool {
    ch.is_alphabetic() || ch == '_'
}

/// Is ch generally a character of identifier?
pub fn is_id(ch: char) -> bool {
    ch.is_alphanumeric() || ch == '_'
}


/// Gets the indentation width of a line
/// 
/// Indentation width is number of spaces divided by 4 and floored
/// 
/// Does not eat the '\n' before the identation
/// 
/// Returns: (width, scrolled_chars)
pub fn parse_indent(chars: &mut Peekable<Enumerate<Chars>>)
-> (usize, usize)
{
    let mut width  = 0usize;
    while let Some((_, ' ')) = chars.peek() {
        chars.next();
        width += 1;
    }

    let result: usize = width / 4;

    let mut scrolled = width;

    (result, scrolled)
}



/// Returns: (parsed_identifier, number_of_eaten_chars)
pub fn parse_id(chars: &mut Peekable<Enumerate<Chars>>) -> (TokenValue, usize) {
    let mut id = String::new();
    let mut eaten = 0;
    
    while chars.peek().is_some() && is_id(chars.peek().unwrap().1) {
        eaten += 1;
        id.push(chars.next().unwrap().1);
    }

    if is_keyword(&id) {
        (TokenValue::Kw(id), eaten)
    } else {
        (TokenValue::Id(id), eaten)
    }
}


/// Example: `parse_double_op(chars, '=', 'U')`
/// 
/// where chars is `[(0, '!'), (1, ..), ...]`
/// 
/// | Input | Output |
/// | ----- | ------ |
/// | "!"   | '!'    |
/// | "!="  | 'U'    |
pub fn parse_double_op(
    chars: &mut Peekable<Enumerate<Chars>>,
    possible_next: char, double_alias: char
) -> (TokenValue, usize)
{
    let (_, c) = chars.next().unwrap();
    
    if chars.peek().is_some() && chars.peek().unwrap().1 == possible_next {
        chars.next();
        return (TokenValue::Op(double_alias), 2);
    } else {
        return (TokenValue::Op(c), 1);
    }
}




fn parse_hex_digit(d: char) -> u64 {
    d.to_digit(16).unwrap() as u64
}

pub fn parse_hex_num(chars: &mut Peekable<Enumerate<Chars>>)
-> (Result<TokenValue, String>, usize)
{
    let mut result: u64 = 0;
    let mut len = 0;

    while chars.peek().is_some() && chars.peek().unwrap().1.is_ascii_hexdigit() {
        len += 1;

        if len > 16 {
            return (
                Err("number is too long (max 2^64-1, 16 digits)".to_string()),
                len
            );
        }

        result <<= 4;
        result += parse_hex_digit(chars.next().unwrap().1);
    }

    (Ok(TokenValue::Int(result)), len)
}




fn parse_bin_digit(d: char) -> u64 {
    d.to_digit(2).unwrap() as u64
}

fn is_bin_digit(c: char) -> bool {
    c == '0' || c == '1'
}

pub fn parse_bin_num(chars: &mut Peekable<Enumerate<Chars>>)
-> (Result<TokenValue, String>, usize)
{
    let mut result: u64 = 0;
    let mut len = 0;

    while chars.peek().is_some() && is_bin_digit(chars.peek().unwrap().1) {
        len += 1;

        if len > 64 {
            return (
                Err("number is too long (max 2^64-1, 64 digits)".to_string()),
                len
            );
        }

        result <<= 1;
        result += parse_bin_digit(chars.next().unwrap().1);
    }

    (Ok(TokenValue::Int(result)), len)
}



pub fn is_num_prefix(prefix: char) -> bool {
    prefix == 'x' || prefix == 'b'
}


pub fn parse_prefixed_num(prefix: char, mut chars: &mut Peekable<Enumerate<Chars>>)
-> (Result<TokenValue, String>, usize)
{
    match prefix {
        'x' => parse_hex_num(&mut chars),
        'b' => parse_bin_num(&mut chars),
        _ => (Err("invalid number prefix".to_string()), 0)
    }
}



pub fn parse_color(mut chars: &mut Peekable<Enumerate<Chars>>)
-> (Result<TokenValue, String>, usize)
{
    chars.next();
                
    if let Some((_, '#')) = chars.peek() {
        chars.next();
        let (parsed, len) = parse_id(&mut chars);
        if let TokenValue::Id(name) = parsed {
            return (Ok(TokenValue::ColorName(name)), len+2); // length of name + ##
        } else {
            panic!("Hmmm... Identifier is not identifier!")
        }
    }

    let mut len = 0;
    let mut source = String::new();
    while chars.peek().is_some() && chars.peek().unwrap().1.is_ascii_hexdigit() {
        len += 1;
        source.push(chars.next().unwrap().1);
    }

    match len {
        1..=4 | 6 | 8 => {},
        _ => {
            return (
                Err(format!(
                    "color was specified using invalid number of digits: {}",
                    len
                )),
                len+1 // + #
            );
        }
    }

    let num = u32::from_str_radix(&source, 16).unwrap();

    // Final color, 0xAARRGGBB
    let mut c: u32 = 0;

    match len {
        1 => c =
            0x00_11_11_11 * num,

        2 => c =
            0x11_00_00_00 * (num >> 4) +    // A
            0x00_11_11_11 * (num & 0x0F),   // RGB

        3 => c =
            0x00_11_00_00 * ( num          >> 8) + // R
            0x00_00_11_00 * ((num & 0x0F0) >> 4) + // G
            0x00_00_00_11 * ((num & 0x00F)     ),  // B
            
        4 => c = 
            0x11_00_00_00 * ( num           >> 12) +  // A
            0x00_11_00_00 * ((num & 0x0F00) >> 8 ) +  // R
            0x00_00_11_00 * ((num & 0x00F0) >> 4 ) +  // G
            0x00_00_00_11 * ((num & 0x000F)      ),   // B
        
        6|8 => c = num,
        
        _ => {}
    }

    (Ok(TokenValue::Color(c)), len+1) // +1 because of '#'
}


/// Returns: (parsed_result, scrolled_lines, scrolled_chars, overall_length)
pub fn parse_string(mut chars: &mut Peekable<Enumerate<Chars>>)
-> (Result<TokenValue, String>, usize, usize, usize)
{
    let unexp_eof = "unexpected EOF while reading a string".to_string();
    let mut result = String::new();
    let mut lines = 0;
    let mut cols = 0; // Character offset on the last line
    let mut length = 0; // Overall number of eaten chars

    // Eat ' or " or `
    let quote = chars.next().unwrap().1;
    cols += 1;
    length += 1;

    loop {
        if chars.peek().is_none() {
            return (Err(unexp_eof), lines, cols, length);
        }

        let (pos, c) = chars.next().unwrap();
        cols += 1;
        length += 1;
        if c == quote { break; } // Enclosing quote was eaten!

        if c == '\n' {
            lines += 1;
            cols = 0;
        }

        // Character is not escaped
        if quote == '`' || c != '\\' {
            result.push(c);
            continue;
        }

        // Parse the escaped character

        if chars.peek().is_none() {
            return (Err(unexp_eof), lines, cols, length);
        }

        let c = chars.next().unwrap().1;
        cols += 1;
        length += 1;

        match c {
            '0' => {}
            '\\'|'/'|'"'|'\'' => result.push(c),
            'n'  => result.push('\n'),

            'x'|'u'|'U' => {
                let len = match c {
                    'x' => 2,
                    'u' => 4,
                    'U' => 8,
                    _ => 0
                };

                let mut code = String::new();
                for _ in 0..len {
                    if chars.peek().is_none() {
                        return (Err(unexp_eof), lines, cols, length);
                    }
                    code.push(chars.next().unwrap().1);
                }
                cols += len;
                length += len;

                result.push(
                    char::from_u32(
                        u32::from_str_radix(&code, 16).unwrap_or_default()
                    ).unwrap_or_default()
                );
            }

            _ => result.push(c)
        }
    }

    (Ok(TokenValue::Str(result)), lines, cols, length)
}

