//! Syntactical analyzer of FjML

////////////////////////////////////////////////////////////////////////////////////////////
// Signature for all parsing functions:
// fn parse_X<'a>(mut reader: TokenReader<'a>) -> ParserResult<X<'a>>
//
// Signature for all parsing predecates:
// fn audit_X(mut reader: TokenReader<'a>) -> bool
////////////////////////////////////////////////////////////////////////////////////////////

pub mod ast;

mod token_reader;
mod basic;

use self::ast::*;
use self::token_reader::*;
use self::basic::*;

use super::tokenizer::{Token, TokenValue, self};
use crate::cli::Logger;
//use crate::config::{GlobalConfig, LocalConfig};




/// If errors occur, this function prints all the errors itself, but after certain stages it returns a verdict
/// saying why it stopped parsing
/// 
/// Requires tokens to be ended with the EOF token
pub fn parse<'a>(
    mut log: &mut Logger,
    tokens: &'a Vec<Token>
)-> Result<AST<'a>, String>
{
    let mut ast = AST::new();

    let mut token_reader = TokenReader::with_tokens(&tokens).expect("[bug] malformed tokens");

    loop {
        let result = parse_statement(token_reader);

        
        if let Err(err) = result {
            match err {
                ParserError::EOF => break,
                ParserError::InvalidToken{found: token, expected: s} => {
                    log.set_line(token.line);
                    log.set_pos(token.pos);
                    log.err(format!(r#"expected {}, found: "{}""#, s, token.value.to_brief_str()));
                }
            }

            return Err("syntax error".to_string());
        }
        
        let (statement, new_reader) = result.unwrap();
        token_reader = new_reader;
        
        ast.statements.push(statement);
    }

    Ok(ast)
}


/// Audits anything except EOL and EOF
fn audit_statement<'a>(mut reader: TokenReader<'a>) -> bool {
    if reader.peek_token().is_eof() {
        false
    } else if reader.peek_token().is_eol() {
        reader.skip_tokens(Skip::EOL);
        audit_block(reader.clone())
    } else {
        // HACK ; and } are specific for blocks
        // If finds ; or } then the token is not considered a statement
        audit_assert!{{!audit_value(reader.clone(), TokenValue::Op(';'))}}
        audit_assert!{{!audit_value(reader.clone(), TokenValue::Gr('}'))}}
        true
    }
}


/// The main parsing function
/// Handles EOF and EOL
fn parse_statement<'a>(mut reader: TokenReader<'a>) -> ParserResult<Statement<'a>>
{
    if reader.peek_token().is_eof() {
        ret_eof!();
    }

    let mut annos = Vec::<Annotation>::new();

    while audit_anno(reader.clone()) {
        let (anno, new_reader) = parse_anno(reader.clone())?;
        reader = new_reader;
        reader.skip_tokens(Skip::EOL);
        annos.push(anno);
    }

    let (stmt_value, new_reader) = parse_statement_value(reader.clone())?;
    reader = new_reader;

    // Eat EOL if present
    reader.require_value(TokenValue::EOL);

    let stmt = Statement{
        value: stmt_value,
        annos,
    };

    Ok((stmt, reader))
}

/// Helper parsing function
/// Does not handle EOF or EOL
fn parse_statement_value<'a>(mut reader: TokenReader<'a>) -> ParserResult<StatementValue<'a>>
{
    if audit_spec_statement(reader.clone()) {
        return parse_spec_statement(reader.clone());
    }

    else if audit_decl_statement(reader.clone()) {
        return parse_decl_statement(reader.clone());
    }

    /*
    else if audit_kw_statement(reader.clone()) {
        return parse_kw_statement(reader.clone());
    }
    */

    else if audit_block(reader.clone()) {
        let (body, new_reader) = parse_block(reader.clone())?;
        Ok((StatementValue::Block(body), new_reader))
    }

    else if audit_prim(reader.clone()) {
        let (prim, new_reader) = parse_prim(reader.clone())?;
        Ok((StatementValue::Prim(prim), new_reader))
    }

    else if audit_empty(reader.clone()) {
        return parse_empty_statement(reader.clone());
    }

    else {
        ret_invalid!(reader.get_token(), "statement");
    }
}


/// Audits `id: id` or `id: type` or `id statement`
fn audit_decl_statement<'a>(mut reader: TokenReader<'a>) -> bool {
    audit_assert!{audit_id(reader.clone())}

    reader.scroll(1);

    if reader.require_value(TokenValue::Op(':')).is_some() {
        if audit_id(reader.clone())
            {return true;}
        else if audit_exact_kw(reader.clone(), "type")
            {return true;}
    } else {
        return audit_statement(reader.clone());
    }

    false
}


/// Parses either a Declaration or a TypeDeclaration
fn parse_decl_statement<'a>(mut reader: TokenReader<'a>) -> ParserResult<StatementValue<'a>>
{
    let mut name: Option<Prim> = None;
    let mut this_is_type = false;

    let (mut type_name, new_reader) = parse_id(reader.clone())?;
    reader = new_reader;

    if reader.require_value(TokenValue::Op(':')).is_some() {
        name = Some(type_name.clone());

        if audit_exact_kw(reader.clone(), "type") {
            let (type_kw, new_reader) = parse_kw(reader.clone())?;
            reader = new_reader;
            this_is_type = true;
        }
        else {
            let (real_type_name, new_reader) = parse_id(reader.clone())
                .or_else(set_expectation!("type name"))
                ?;
            reader = new_reader;
            type_name = real_type_name;
        }
    }

    let mut body: Option<Box<Statement>> = None;
    
    if audit_statement(reader) {
        let (stmt, new_reader) = parse_statement(reader.clone())?;
        reader = new_reader;
        body = Some(Box::new(stmt));
    }

    if this_is_type {
        Ok((
            StatementValue::TypeDecl(
                Box::new(TypeDeclaration{
                    name: name.unwrap(), body
                })
            ),
            reader
        ))
    }
    else {
        Ok((
            StatementValue::Decl(
                Box::new(Declaration{
                    name, type_name, body
                })
            ),
            reader
        ))
    }
}



/// Audits `.id`
fn audit_spec_statement<'a>(mut reader: TokenReader<'a>) -> bool {
    audit_assert!{audit_value(reader.clone(), TokenValue::Op('.'))}
    reader.scroll(1);
    audit_assert!{audit_id(reader.clone())}

    true
}


fn parse_spec_statement<'a>(mut reader: TokenReader<'a>) -> ParserResult<StatementValue<'a>>
{
    reader.require_value(TokenValue::Op('.'));

    let (name, new_reader) = parse_id(reader.clone())
        .or_else(set_expectation!("label"))
        ?;
    reader = new_reader;

    let (body, new_reader) = parse_statement(reader.clone())
        .or_else(set_expectation!("specification body (empty specifications have no sense)"))
        ?;
    reader = new_reader;

    Ok((
        StatementValue::Spec(Box::new(Specification{
            name, body: Box::new(body)
        })),
        reader
    ))
}


/*
/// Audits `kw`
fn audit_kw_statement<'a>(reader: TokenReader<'a>) -> bool {
    audit_kw(reader)
}


// Keyword stmts are not yet defined by spec
fn parse_kw_statement<'a>(mut reader: TokenReader<'a>) -> ParserResult<StatementValue<'a>>
{
    let (keyword, new_reader) = parse_kw(reader.clone())?;
    reader = new_reader;

    let prim_arg: Option<Prim> = 
        if audit_prim(reader.clone()) {
            let (prim, new_reader) = parse_prim(reader.clone())?;
            reader = new_reader;
            Some(prim)
        }
        else {None};

    let statement_arg: Option<Box<Statement>> = 
        if audit_block(reader.clone()) {
            let (stmt, new_reader) = parse_statement(reader.clone())?;
            reader = new_reader;
            Some(Box::new(stmt))
        }
        else {None};

    Ok((
        StatementValue::Kw(Box::new(KeywordStatement{
            keyword, prim_arg, statement_arg
        })),
        reader
    ))
}
*/