use super::{atoms::*, compile_statement};

use crate::cli::Logger;

#[cfg(debug_assertions)]
#[cfg(feature = "perform_debug")]
pub fn perform_debug<'a> (
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>
) -> Result<(),()> {

    if let ast::StatementValue::Block(stmts) = &stmt.value {
        for stmt in stmts {
            for anno in &stmt.annos {
                match anno.name.get_text() {
                    "overview" => {
                        log.debug(format!("{:#?}", fjml));
                    }

                    "print" => {
                        if anno.args.len() > 0 {
                            log.debug(anno.args[0].get_text());
                        }
                    }

                    "inspect" => {
                        if anno.args.len() > 0 {
                            let name = anno.args[0].get_text();

                            let id = fjml.lookup_global(name);
                            if id.is_none() {
                                log.debug(format!("{}: no such global", name));
                                continue;
                            }
                            let id = id.unwrap();

                            let class =
                                if fjml.is_type(id) {"type"}
                                else if fjml.is_object(id) {"object"}
                                else if fjml.is_value(id) {"value"}
                                else {"unassigned"};

                            let value = match class {
                                "value" => fjml.values[&id].to_string(),
                                "type"|"object" => format!("{:#?}", fjml.objects[&id]),
                                _ => "[none]".to_string()
                            };

                            let prim = &fjml.names[&id];

                            log.debug(format!(
                                "@{}:{} #{} <{}> {} = {}",
                                prim.line, prim.pos, id,
                                name, class,
                                value
                            ));
                        }
                    }

                    _ => {}
                }
            }
        }
    }

    Ok(())
}


fn handle_definition<'a>(
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    name: Option<Prim<'a>>,
    parent: Option<ID>,
) -> Option<ID> {
    
    if let Some(val) = name {
        if fjml.lookup(parent, val.get_text()).is_some() {
            log.set_prim_pos(&val);
            log.err("already defined");
            let prev = fjml.lookup(parent, val.get_text()).unwrap();
            log.set_prim_pos(fjml.get_name(prev).unwrap());
            log.err("here");
            return None;
        }
    }

    let id = fjml.define_maybe_named(name);

    if parent.is_some() {
        fjml.add_local(parent.unwrap(), id);
    } else {
        fjml.make_global(id);
    }

    Some(id)
}


pub fn handle_decl<'a>(
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()> {
    let decl = stmt.value.get_decl().unwrap();

    let id = handle_definition(&mut fjml, &mut log, decl.name, parent);
    if id.is_none() {return Err(());}
    let id = id.unwrap();

    for anno in &stmt.annos {
        if anno.name.get_text() == "final" {
            fjml.make_final(id);
        }
    }

    let type_name = decl.type_name.get_text();

    let core_types = ["Num", "Str", "Color"];

    // If this is a primitive
    if core_types.contains(&type_name) {
        // The value is given
        if decl.body.is_some() {
            // The value is prim
            if let Some(val) = decl.body.as_ref().unwrap().value.get_prim() {
                // Identifier given - use the value of that identifier
                if let PrimValue::Id(global_name) = val.value {
                   /*  // This identifier exists
                    if let Some(global_id) = fjml.lookup_global(global_name) {
                        // This identifier refers to an prim
                        if fjml.is_value(id) {
                            // Clone taht value
                            fjml.make_value(id, fjml.values[&global_id].clone());
                        // The identifier does not refer to an prim
                        } else {
                            log.set_prim_pos(&val);
                            log.err("not an prim");
                            return true;
                        }
                    // This identifier does not exist
                    } else {
                        log.set_prim_pos(&val);
                        log.err("undefined");
                        return true;
                    } */
                    log.set_prim_pos(val);
                    log.err("cannot assign identifier");
                    return Err(());
                }
                // Just assign prim
                else {
                    fjml.make_value(id, val.clone());
                }
            // The value is not prim!
            } else {
                log.set_prim_pos(&decl.type_name);
                log.err("expected prim, got something else");
                return Err(());
            }
        // I should assign a default value for that type
        } else {
            let prim = match type_name {
                "Num" => Prim::new_num(),
                "Str" => Prim::new_str(),
                "Color" => Prim::new_color(),
                _ => { unimplemented!("new core type?"); }
            };
            fjml.make_value(id, prim);
        }

    }

    // Else, this is a structure
    else {
        
        let type_id = fjml.lookup_global(decl.type_name.get_text());
        if type_id.is_none() {
            log.set_prim_pos(&decl.type_name);
            log.err("undefined type");
            return Err(());
        }
        let type_id = type_id.unwrap();

        fjml.make_object(id);
        fjml.instantiate_type(id, type_id);
        fjml.set_parent(id, parent);

        if let Some(stmt) = &decl.body {
            return compile_statement(&mut fjml, &mut log, &stmt, Some(id));
        }

    }

    Ok(())
}


pub fn handle_type_decl<'a> (
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()> {

    let decl = stmt.value.get_type_decl().unwrap();

    if parent.is_some() {
        log.set_prim_pos(&decl.name);
        log.err("types must be defined in global scope");
        return Err(());
    }


    let id = handle_definition(&mut fjml, &mut log, Some(decl.name), None);
    if id.is_none() {
        return Err(());
    }
    let id = id.unwrap();

    fjml.make_object(id);
    fjml.make_type(id);

    // Process standard annotations
    for anno in &stmt.annos {
        match anno.name.get_text() {
            "class" => {    
                if anno.args.len() != 1 {
                    log.set_prim_pos(&anno.name);
                    log.warn("expected 1 argument: class name");
                } else {
                    if let PrimValue::String(s) = anno.args[0].value {
                        let cl = fjml.get_type_class(id);
                        if cl.is_some() && cl.unwrap() != s {
                            log.set_prim_pos(&anno.args[0]);
                            log.err(format!("type '{}' is of another class: {}", decl.name.get_text(), cl.unwrap()));
                            return Err(());
                        }

                        fjml.set_type_class(id, s);
                    } else {
                        log.set_prim_pos(&anno.args[0]);
                        log.warn("expected string: class name");
                    }
                }
            } // "class"


            "extend" => {
                for type_name_prim in &anno.args {
                    if let PrimValue::Id(type_name) = type_name_prim.value {
                        let lookup_result = fjml.lookup_global(type_name);
                        if lookup_result.is_none() {
                            log.set_prim_pos(type_name_prim);
                            log.err("undefined type");
                            return Err(());
                        }
                        let type_id = lookup_result.unwrap();

                        if !fjml.is_type(type_id) {
                            log.set_prim_pos(type_name_prim);
                            log.err("expected type");
                            return Err(());
                        }

                        if type_id == id {
                            log.set_prim_pos(type_name_prim);
                            log.err("type cannot extend itself");
                            return Err(());
                        }

                        let base_class = fjml.get_type_class(type_id);
                        let my_class = fjml.get_type_class(id);

                        if base_class.is_some()
                            && my_class.is_some()
                            && base_class.unwrap() != my_class.unwrap()
                        {
                            log.set_prim_pos(type_name_prim);
                            log.err(format!("type '{}' has class {} incompatible with class {}",
                                type_name, base_class.unwrap(), my_class.unwrap()
                            ));
                            return Err(());
                        }

                        if base_class.is_some() {
                            fjml.set_type_class(id, base_class.unwrap());
                        }

                        fjml.extend_type(id, type_id);

                    } else {
                        log.set_prim_pos(type_name_prim);
                        log.warn("expected identifier: base type name");
                        continue;
                    }
                }
            } // "extend"

            "final" => {
                fjml.make_final(id);
            }

            _ => {}
        }
    }

    if let Some(stmt) = &decl.body {
        return compile_statement(&mut fjml, &mut log, &stmt, Some(id));
    }

    Ok(())
}


pub fn handle_spec<'a> (
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()> {

    let spec = stmt.value.get_spec().unwrap();

    let lookup_result = fjml.lookup(parent, spec.name.get_text());

    if lookup_result.is_none() {
        log.set_prim_pos(&spec.name);
        log.err(format!("name '{}' does not exist in '{}'",
            spec.name.get_text(),
            fjml.get_scope_path(parent)
        ));
        return Err(());
    }

    let mut id = lookup_result.unwrap();

    if fjml.is_final(id) {
        log.set_prim_pos(&spec.name);
        log.err("cannot modify a final value");
        return Err(());
    }

    if let Some(parent_id) = parent {
        let type_id = fjml.get_type(parent_id);
        let field_name = spec.name.get_text();

        // If this ID is a type's field
        if let Some(field_id) = fjml.lookup_in_type(type_id, field_name) {
            // And it is still not overridden
            if !fjml.is_field_overridden(parent_id, field_id) {
                // Override!
                let overridden_id = fjml.clone_id(id);
                fjml.override_field(parent_id, field_id, overridden_id);
                id = overridden_id;
            }
        }
    }

    // One more check because `id` could have cheanged
    if fjml.is_final(id) {
        log.set_prim_pos(&spec.name);
        log.err("cannot modify a final value");
        return Err(());
    }

    for anno in &stmt.annos {
        if anno.name.get_text() == "final" {
            fjml.make_final(id);
        }
    }

    if fjml.is_value(id) {
        if let Some (prim) = spec.body.value.get_prim() {
            fjml.make_value(id, prim.clone());
        } else {
            log.set_prim_pos(&spec.name);
            log.err("expected value, got something else");
            return Err(());
        }
    } else {
        return compile_statement(&mut fjml, &mut log, &spec.body, Some(id));
    }

    Ok(())


}
