//! Let's make everything clear.
//! 
//! LOCAL is an entity owned by OBJECT but not its type.
//! 
//! FIELD is an entity specified by TYPE and owned by EITHER object or type.
//! If a field is owned by an object, we say that object OVERRIDES the type's field.

// TODO change Vec to HashSet

pub use crate::builder::parser::ast::{Prim, PrimValue};
pub use crate::builder::parser::ast;

use std::borrow::BorrowMut;
use std::collections::HashMap;

/// Index inside FjML.names
/// 
/// **All IDs are global to a single file**
pub type ID = u64;

#[derive(Debug, Clone)]
pub struct FjObject<'a> {
    pub parent: Option<ID>,

    pub locals: Vec<ID>,
    
    locals_lookup_map: HashMap<&'a str, ID>,
    
    /// If this is a type instance, this is set to the type's id
    pub type_id: Option<ID>,
    
    /// ID of the type's field -> ID of the type's default field or my local
    pub type_fields: HashMap<ID, ID>,

}


impl<'a> FjObject<'a> {
    pub fn new() -> Self {
        Self {
            parent: None,
            locals: Vec::new(),
            locals_lookup_map: HashMap::new(),
            type_id: None,
            type_fields: HashMap::new(),
        }
    }

    pub fn lookup_local(&self, name: &str) -> Option<ID> {
        self.locals_lookup_map.get(name)
            .and_then(|id| Some(*id))
    }

    pub fn add_lookup_name(&mut self, name: &'a str, id: ID) {
        self.locals_lookup_map.insert(name, id);
    }
}


#[derive(Debug)]
pub struct FjML<'a> {
    next_id: ID,
    
    /// Helps to look up global names
    globals_lookup_map: HashMap<&'a str, ID>,
    
    /// Contains names of all the named objects
    pub names: HashMap<ID, Prim<'a>>,

    // TODO externals -- modularity
    //pub externals: Vec<ID>,
    
    pub objects: HashMap<ID, FjObject<'a>>,
    
    pub values: HashMap<ID, Prim<'a>>,
    
    pub types: Vec<ID>,

    pub type_classes: HashMap<ID, &'a str>,

    /// Final things cannot be modified
    pub final_list: Vec<ID>,

    // TODO object annotations; display them in @inspect
}

impl<'a> FjML<'a> {
    pub fn new() -> Self {
        return Self {
            next_id: 0,
            globals_lookup_map: HashMap::new(),

            names: HashMap::new(),

            //externals: Vec::new(),
            objects: HashMap::new(),
            values: HashMap::new(),
            types: Vec::new(),

            type_classes: HashMap::new(),
            final_list: Vec::new()
        }
    }

    fn next_id(&mut self) -> ID {
        let id = self.next_id;
        self.next_id += 1;
        id
    }

    /// Returns: ID that corresponds to the name
    pub fn define_name(&mut self, val: Prim<'a>) -> ID {
        let id = self.next_id();
        self.names.insert(id, val);
        id
    }

    pub fn define_unnamed(&mut self) -> ID {
        self.next_id()
    }

    pub fn define_maybe_named(&mut self, name: Option<Prim<'a>>) -> ID {
        if let Some(val) = name {
            self.define_name(val)
        } else {
            self.define_unnamed()
        }
    }

    pub fn has_name(&self, id: ID) -> bool {
        self.names.contains_key(&id)
    }

    pub fn get_name(&self, id: ID) -> Option<&Prim<'a>> {
        self.names.get(&id)
    }

    pub fn lookup_global(&self, global_name: &str) -> Option<ID> {
        self.globals_lookup_map.get(global_name)
            .and_then(|id| Some(*id))
    }

    pub fn lookup_local(&self, parent_id: ID, field_name: &str) -> Option<ID> {
        let obj = self.objects.get(&parent_id).expect("requested ID is not an object");

        // Try to find inside the object
        if let Some(local_id) = obj.lookup_local(field_name) {
            return Some(local_id);
        }

        // Try to find inside its type
        if let Some(type_id) = obj.type_id {
            if let Some(field_id) = self.objects[&type_id].lookup_local(field_name) {
                // Objects can override their type's fields
                return Some(self.get_field(parent_id, field_id));
            }
        }

        None
    }

    /// If `parent_id` is none, looks up in the global scope
    pub fn lookup(&self, parent_id: Option<ID>, name: &str) -> Option<ID> {
        if let Some(id) = parent_id {
            self.lookup_local(id, name)
        } else {
            self.lookup_global(name)
        }
    }

    /// Looks up in type.
    /// If `type_id` is `None`, returns None (this means that object does not have a type)
    pub fn lookup_in_type(&self, type_id: Option<ID>, name: &str) -> Option<ID> {
        if let Some(id) = type_id {
            self.lookup_local(id, name)
        } else {
            None
        }
    }

    /// Adds `id` to the index of types
    pub fn make_type(&mut self, id: ID) {
        self.types.push(id);
    }

    /// Adds `id` to the index of values and assigns `value`
    pub fn make_value(&mut self, id: ID, value: Prim<'a>) {
        self.values.insert(id, value);
    }

    /// Adds `id` to the index of objects and assigns a brand new empty object
    pub fn make_object(&mut self, id: ID) {
        self.objects.insert(id, FjObject::new());
    }

    /// If object has a name, it is added to the globals lookup index
    pub fn make_global(&mut self, id: ID) {
        if self.has_name(id) {
            self.globals_lookup_map.insert(self.names[&id].get_text(), id);
        }

        if self.is_object(id) {
            self.objects.get_mut(&id).unwrap().parent = None;
        }
    }


    /// Is id a type?
    pub fn is_type(&self, id: ID) -> bool {
        self.types.contains(&id)
    }

    /// Does id contain a primitive?
    pub fn is_value(&self, id: ID) -> bool {
        self.values.contains_key(&id)
    }

    /// Is id a complex structure (type or type instance)?
    pub fn is_object(&self, id: ID) -> bool {
        self.objects.contains_key(&id)
    }

    /// Gets the type id of the object
    pub fn get_type(&self, object_id: ID) -> Option<ID> {
        self.objects[&object_id].type_id
    }

    /// Adds type fields to the object
    pub fn instantiate_type(&mut self, object_id: ID, type_id: ID) {
        // Copy type fields now because later we won't be able to reference self.objects
        let type_fields = self.objects[&type_id].locals.clone();

        let mut obj = self.objects.get_mut(&object_id).expect("requested ID is not an object");

        obj.type_id = Some(type_id);
        
        for field in type_fields {
            obj.type_fields.insert(field, field);
        }
    }

    /// Adds a local item to the object
    pub fn add_local(&mut self, object_id: ID, item_id: ID) {
        self.objects.get_mut(&object_id).expect("requested ID is not an object")
            .locals.push(item_id);

        if self.has_name(item_id) {
            let name = self.get_name(item_id).unwrap().get_text();
            self.objects.get_mut(&object_id).unwrap()
                .add_lookup_name(name, item_id);
        }
    }

    pub fn set_parent(&mut self, object_id: ID, parent_id: Option<ID>) {
        if self.is_object(object_id) {
            self.objects.get_mut(&object_id).unwrap()
                .parent = parent_id;
        }
    }

    /// Overrides type's field (`types_id`) with object's own field (`overridden_id`)
    pub fn override_field(&mut self, object_id: ID, types_id: ID, overridden_id: ID) {
        self.objects.get_mut(&object_id).expect("requested ID is not an object")
            .type_fields.insert(types_id, overridden_id);
    }

    pub fn is_field_overridden(&self, object_id: ID, field_id: ID) -> bool {
        self.objects[&object_id].type_fields[&field_id] != field_id
    }

    pub fn get_field(&self, object_id: ID, field_id: ID) -> ID {
        self.objects[&object_id].type_fields[&field_id]
    }

    pub fn clone_value(&mut self, value_id: ID) -> ID {
        /* let id = self.define_maybe_named(
            self.get_name(value_id)
                .and_then( |p| Some(p.clone()) )
        ); */
        let id = self.define_unnamed();
        self.make_value(id, self.values[&value_id].clone());
        id
    }

    /// Does NOT clone types
    pub fn clone_object(&mut self, object_id: ID) -> ID {
        /* let id = self.define_maybe_named(
            self.get_name(object_id)
                .and_then( |p| Some(p.clone()) )
        ); */
        let id = self.define_unnamed();

        let locals = self.objects[&object_id].locals.clone();
        let mut new_locals = Vec::<ID>::new();
        new_locals.resize(locals.len(), 0);
        for i in 0..locals.len() {
            new_locals[i] = self.clone_id(locals[i]);
        }

        self.objects.insert(id, 
            FjObject {
                locals: new_locals,
                ..self.objects[&object_id].clone()
            }
        );

        id
    }

    /// Clones anything with the specified ID.
    /// Does NOT clone types!
    pub fn clone_id(&mut self, id: ID) -> ID {
        if self.is_value(id) {
            self.clone_value(id)
        } else {
            self.clone_object(id)
        }
    }

    /// Creates a string that represents a scope path (full name)
    pub fn get_scope_path(&self, scope_id: Option<ID>) -> String {
        let mut path = String::new();

        if scope_id.is_none() {
            return "global scope".to_string();
        }

        let mut id = scope_id.unwrap();
        
        if self.is_value(id) {
            return "[primitive]".to_string();
        }

        loop {
            path = self.get_name(id)
                .and_then(|v| Some(v.get_text()))
                .unwrap_or("#")
                .to_string()
                + "::"
                + &path;

            if let Some(parent_id) = self.objects[&id].parent {
                id = parent_id;
            } else {
                break;
            }
        }

        // Pop the last "::"
        path.pop();
        path.pop();

        path
    }

    pub fn set_type_class(&mut self, type_id: ID, class_name: &'a str) {
        self.type_classes.insert(type_id, class_name);
    }

    pub fn get_type_class(&self, type_id: ID) -> Option<&'a str> {
        self.type_classes.get(&type_id)
            .and_then(|cl| Some(*cl))
    }

    /// Requires all classes stuff to be checked
    /// 
    /// Merges all type locals (they definitely have different IDs)
    /// and name lookup maps
    pub fn extend_type(&mut self, type_id: ID, base_type_id: ID) {
        let base_locals = self.objects[&base_type_id].locals.clone();
        let base_lookup_map = self.objects[&base_type_id].locals_lookup_map.clone();    

        let mut obj = self.objects.get_mut(&type_id).expect("not an object");
        
        obj.locals.extend(base_locals);
        obj.locals_lookup_map.extend(base_lookup_map);
    }

    pub fn make_final(&mut self, id: ID) {
        self.final_list.push(id);
    }

    pub fn is_final(&self, id: ID) -> bool {
        self.final_list.contains(&id)
    }
}