//! Builds FjML project from given configs

pub mod tokenizer;
pub mod parser;
pub mod compiler;
pub mod common;

use self::tokenizer::*;
use self::parser::*;
use self::compiler::*;

use crate::config::{LocalConfig, GlobalConfig};
use crate::cli::{Logger, BuildStage};

use glob;

use std::fs;
use std::path::{Path, PathBuf};
use std::env;


pub fn build(mut log: &mut Logger, gc: &GlobalConfig, lc_path: &str) {
    log.set_stage(BuildStage::Reading);

    // Load local configuration

    let lc = LocalConfig::load(lc_path);
    if let Err(msg) = lc {
        log.set_file(lc_path.to_string());
        log.err(msg);
        return;
    }
    let lc = lc.unwrap();

    // Process-local current dir
    env::set_current_dir(&lc.path);

    log.hilog(format!("Building project '{}' v{}", lc.project.name, lc.project.version));

    // Check Fejix version used by the project
    if let Some(build_config) = &lc.project.build {
        if let Some(fejix_version) = &build_config.fejix_version {
            if fejix_version.major != gc.fejix.version.major {
                log.unset_stage();
                log.err(format!("Project requires Fejix version {} while current is {} (major version mismatches)",
                    fejix_version, gc.fejix.version
                ));
                return;
            }

            if fejix_version.minor > gc.fejix.version.minor {
                log.unset_stage();
                log.warn(format!("Project requires Fejix version {} while current is {} (minor version is bigger)",
                    fejix_version, gc.fejix.version
                ));
            }
        }
    }

    log.set_stage(BuildStage::Reading);

    // Make a list of all files used in the project

    let result = glob::glob(&(lc.project.source + "/**/*.fj"));
    if let Err(err) = result {
        log.err(err.msg);
        return;
    }

    let paths = result.unwrap();

    for path in paths {
        log.set_stage(BuildStage::Reading);
        
        if let Err(err) = path {
            log.set_file(err.path().display().to_string());
            log.err(err.to_string());
            return;
        }

        let path = path.unwrap();

        let path_str = path.to_str().unwrap();
        let file_name = path.file_name().unwrap().to_str().expect("A file has invalid non-UTF8 name");

        log.log(format!("Building '{}'...", file_name));
        log.set_file(file_name.to_string());
        let file = fs::read_to_string(&path);
        if let Err(msg) = file {
            log.err(msg.to_string());
            return;
        }

        log.set_stage(BuildStage::Tokenizing);

        let result = tokenize(file.unwrap());
        if let Err(err) = result {
            log.set_token_pos(&err.token);
            log.err(err.message);
            log.log(format!(">> {}", err.source));
            return;
        }
        let tokens = result.unwrap();
        
        log.set_stage(BuildStage::Parsing);

        let result = parse(&mut log, &tokens);

        log.unset_pos();

        if let Err(msg) = result {
            log.err(msg);
            return;
        }
        let ast = result.unwrap();

        log.set_stage(BuildStage::Compiling);
        
        let result = compile(&mut log, &ast, &lc.project.build);

        log.unset_pos();

        if let Err(msg) = result {
            log.err(msg);
            return;
        }
        let fjml = result.unwrap();

    }
}