pub use crate::builder::parser::ast::{Rvalue, Primitive};
pub use crate::builder::parser::ast;

use std::borrow::BorrowMut;
use std::collections::HashMap;

/// Index inside FjML.names
/// 
/// **All IDs are global to a single file**
pub type ID = u64;

#[derive(Debug)]
pub struct FjObject<'a> {
    pub parent: Option<ID>,

    pub locals: Vec<ID>,
    
    locals_lookup_map: HashMap<&'a str, ID>,
    
    /// If this is a type instance, this is set to the type's id
    pub type_id: Option<ID>,
    
    /// Type's field ID -> my overridden object ID
    pub type_overrides: HashMap<ID, ID>,

    // TODO object annotations
}


impl<'a> FjObject<'a> {
    pub fn new() -> Self {
        Self {
            parent: None,
            locals: Vec::new(),
            locals_lookup_map: HashMap::new(),
            type_id: None,
            type_overrides: HashMap::new(),
        }
    }

    pub fn lookup_local(&self, name: &str) -> Option<ID> {
        self.locals_lookup_map.get(name)
            .and_then(|id| Some(*id))
    }

    pub fn add_lookup_name(&mut self, name: &'a str, id: ID) {
        self.locals_lookup_map.insert(name, id);
    }
}


#[derive(Debug)]
pub struct FjML<'a> {
    next_id: ID,
    
    /// Helps to look up global names
    globals_lookup_map: HashMap<&'a str, ID>,
    
    /// Contains names of all the named objects
    pub names: HashMap<ID, Rvalue<'a>>,

    //pub externals: Vec<ID>,
    
    pub objects: HashMap<ID, FjObject<'a>>,
    
    pub values: HashMap<ID, Rvalue<'a>>,
    
    pub types: Vec<ID>,
}

impl<'a> FjML<'a> {
    pub fn new() -> Self {
        return Self {
            next_id: 0,
            globals_lookup_map: HashMap::new(),

            names: HashMap::new(),

            //externals: Vec::new(),
            objects: HashMap::new(),
            values: HashMap::new(),
            types: Vec::new(),
        }
    }

    fn next_id(&mut self) -> ID {
        let id = self.next_id;
        self.next_id += 1;
        id
    }

    /// Returns: ID that corresponds to the name
    pub fn define_name(&mut self, val: Rvalue<'a>) -> ID {
        let id = self.next_id();
        self.names.insert(id, val);
        id
    }

    pub fn define_unnamed(&mut self) -> ID {
        self.next_id()
    }

    pub fn define_maybe_named(&mut self, name: Option<Rvalue<'a>>) -> ID {
        if let Some(val) = name {
            self.define_name(val)
        } else {
            self.define_unnamed()
        }
    }

    pub fn has_name(&self, id: ID) -> bool {
        self.names.contains_key(&id)
    }

    pub fn get_name(&self, id: ID) -> Option<&Rvalue<'a>> {
        self.names.get(&id)
    }

    pub fn lookup_global(&self, global_name: &str) -> Option<ID> {
        self.globals_lookup_map.get(global_name)
            .and_then(|id| Some(*id))
    }

    pub fn lookup_local(&self, parent_id: ID, field_name: &str) -> Option<ID> {
        self.objects.get(&parent_id).expect("requested ID is not an object")
            .lookup_local(field_name)
    }

    pub fn lookup(&self, parent_id: Option<ID>, name: &str) -> Option<ID> {
        if let Some(id) = parent_id {
            self.lookup_local(id, name)
        } else {
            self.lookup_global(name)
        }
    }

    /// Adds `id` to the index of types
    pub fn make_type(&mut self, id: ID) {
        self.types.push(id);
    }

    /// Adds `id` to the index of values and assigns `value`
    pub fn make_value(&mut self, id: ID, value: Rvalue<'a>) {
        self.values.insert(id, value);
    }

    /// Adds `id` to the index of objects and assigns a brand new empty object
    pub fn make_object(&mut self, id: ID) {
        self.objects.insert(id, FjObject::new());
    }

    /// If object has a name, it is added to the globals lookup index
    pub fn make_global(&mut self, id: ID) {
        if self.has_name(id) {
            self.globals_lookup_map.insert(self.names[&id].get_text(), id);
        }

        if self.is_object(id) {
            self.objects.get_mut(&id).unwrap().parent = None;
        }
    }


    /// Is id a type?
    pub fn is_type(&self, id: ID) -> bool {
        self.types.contains(&id)
    }

    /// Does id contain a primitive?
    pub fn is_value(&self, id: ID) -> bool {
        self.values.contains_key(&id)
    }

    /// Is id a complex structure (type or type instance)?
    pub fn is_object(&self, id: ID) -> bool {
        self.objects.contains_key(&id)
    }

    /// Gets the type id of the object
    pub fn get_type(&self, object_id: ID) -> Option<ID> {
        self.objects[&object_id].type_id
    }

    /// Sets `type_id` of the object
    pub fn set_type(&mut self, object_id: ID, type_id: ID) {
        self.objects.get_mut(&object_id).expect("requested ID is not an object")
            .type_id = Some(type_id);
    }

    /// Adds item to object
    pub fn add_to_object(&mut self, object_id: ID, item_id: ID) {
        self.objects.get_mut(&object_id).expect("requested ID is not an object")
            .locals.push(item_id);

        if self.has_name(item_id) {
            let name = self.get_name(item_id).unwrap().get_text();
            self.objects.get_mut(&object_id).unwrap()
                .add_lookup_name(name, item_id);
        }
    }

    pub fn set_parent(&mut self, object_id: ID, parent_id: Option<ID>) {
        if self.is_object(object_id) {
            self.objects.get_mut(&object_id).unwrap()
                .parent = parent_id;
        }
    }

    /// Overrides type's field (`types_id`) with object's own field (`overridden_id`)
    pub fn override_field(&mut self, object_id: ID, types_id: ID, overridden_id: ID) {
        self.objects.get_mut(&object_id).expect("requested ID is not an object")
            .type_overrides.insert(types_id, overridden_id);
    }

    /// Creates a string that represents a scope path (full name)
    pub fn get_scope_path(&self, scope_id: Option<ID>) -> String {
        let mut path = String::new();

        if scope_id.is_none() {
            return "global scope".to_string();
        }

        let mut id = scope_id.unwrap();
        
        if self.is_value(id) {
            return "[rvalue]".to_string();
        }

        loop {
            path = self.get_name(id)
                .and_then(|v| Some(v.get_text()))
                .unwrap_or("#")
                .to_string()
                + "::"
                + &path;

            if let Some(parent_id) = self.objects[&id].parent {
                id = parent_id;
            } else {
                break;
            }
        }

        // Pop the last "::"
        path.pop();
        path.pop();

        path
    }
}