#![allow(unused)]

mod config;
use config::*;

mod cli;

mod info;
mod builder;

use std::env;
use std::path;
use std::path::PathBuf;

fn main() {
    let mut log = cli::Logger::new();

    let my_path = env::current_exe().unwrap();
    let my_dir = my_path.parent().unwrap();

    let default_config = my_dir.join("fejix_config.toml");

    let default_config =
        if let Ok(path) = env::var("FEJIX_CONFIG") {
            let config = path::PathBuf::from(path);
            if !config.exists() {
                log.err("environment variable FEJIX_CONFIG is defined but is invalid");
                return;
            }
            config
        } else {
            default_config
        };

    let args = cli::get_args(&default_config.to_str().unwrap());

    log.quiet = args.is_present("quiet");
    log.colorful = !args.is_present("no_color");
    log.debug = args.is_present("compiler_debug");
    log.set_stage(cli::BuildStage::Reading);

    // Save CWD
    let cwd = env::current_dir().expect("Cannot access current working directory");

    // Get global config path
    // And set its parent directory to current (needed to resolve relative paths)
    let global_config_path = args.value_of("global_config").unwrap();
    let global_config_parsed_path = PathBuf::from(global_config_path);
    let global_config_dir = global_config_parsed_path.parent()
        .expect("Global config does not have a parent directory");
    env::set_current_dir(global_config_dir);
    
    // Load global config (resolves relative paths)
    let gconfig = GlobalConfig::load(global_config_path);
    if let Err(msg) = gconfig {
        log.set_file(global_config_path.to_string());
        log.err(msg);
        return;
    }
    let gconfig = gconfig.unwrap();

    // Restore cwd
    env::set_current_dir(cwd);

    // If color wasn't disabled by args, it is controlled by global config
    if log.colorful {
        log.colorful = gconfig.output.use_color;
    }


    let local_config_path = args.value_of("local_config").unwrap();


    if args.is_present("info") {
        info::info(
            &mut log, local_config_path,
            args.value_of("specific_link"), args.is_present("no_links")
        );
    } else {
        // All the important magic happens there
        builder::build(&mut log, &gconfig, local_config_path);
    }
}