//! Manages reading configuration from .toml files

use serde::Deserialize;
use toml;
pub use semver;

use std::fs;
use std::path::{PathBuf, Path};
use std::str::FromStr;
use std::collections::HashMap;

fn deserialize_version<'de, D>(deserializer: D) -> Result<semver::Version, D::Error>
    where D: serde::de::Deserializer<'de>
{
    let s = String::deserialize(deserializer)?;
    match semver::Version::parse(&s) {
        Err(msg) => Err(serde::de::Error::custom(msg)),
        Ok(ver) => Ok(ver)
    }
}

fn deserialize_opt_version<'de, D>(deserializer: D) -> Result<Option<semver::Version>, D::Error>
    where D: serde::de::Deserializer<'de>
{
    let s = String::deserialize(deserializer)?;
    match semver::Version::parse(&s) {
        Err(msg) => Err(serde::de::Error::custom(msg)),
        Ok(ver) => Ok(Some(ver))
    }
}

/// Global configuration
#[derive(Deserialize)]
pub struct GlobalConfig {
    pub fejix: FejixConfig,
    pub output: OutputConfig,
}

/// Console output
#[derive(Deserialize)]
pub struct OutputConfig {
    pub use_color: bool,
}

/// Fejix compiler configuration
#[derive(Deserialize)]
pub struct FejixConfig {
    #[serde(deserialize_with="deserialize_version")]
    pub version: semver::Version,

    pub paths: FejixPathsConfig,
}


#[derive(Deserialize)]
pub struct FejixPathsConfig {
    pub stdlib: String,
}

/// Local configuration
#[derive(Deserialize)]
pub struct LocalConfig {
    pub project: ProjConfig,

    #[serde(skip)]
    pub path: PathBuf,
}

/// Project configuration
#[derive(Deserialize)]
pub struct ProjConfig {
    pub name: String,

    #[serde(deserialize_with="deserialize_version")]
    pub version: semver::Version,

    pub source: String,

    pub description: Option<String>,
    pub links: Option<HashMap<String, String>>,

    pub build: Option<BuildConfig>,
}

#[derive(Deserialize)]
pub struct BuildConfig {
    pub compiler_debug: Option<bool>,

    // pub use_stdlib: Option<bool>,
    // pub use_stdlib_prelude: Option<bool>,

    #[serde(deserialize_with="deserialize_opt_version")]
    #[serde(default)] // Defaults to None
    pub fejix_version: Option<semver::Version>,
}


/// Converts all relative paths to absolute (i.e. canonicalizes them)
fn resolve_rel_paths(gc: GlobalConfig) -> Result<GlobalConfig, String> {
    let mut gc = gc;
    
    let result = fs::canonicalize(gc.fejix.paths.stdlib);
    if let Err(msg) = result {
        return Err(String::from("Cannot resolve relative path for fejix.paths.stdlib"));
    }
    gc.fejix.paths.stdlib = result.unwrap().to_str().unwrap().to_string();

    Ok(gc)
}


/// Read global configuration of Fejix compiler
impl GlobalConfig {
    pub fn load(path: &str) -> Result<GlobalConfig, String> {
        match fs::read_to_string(path) {
            Err(msg) => Err(msg.to_string()),
            Ok(s) => match toml::from_str(&s) {
                Err(msg) => Err(msg.to_string()),
                Ok(cfg) =>
                    resolve_rel_paths(cfg)
            }
        }
    }   
}


/// Reads local (project) configuration
impl LocalConfig {
    pub fn load(path: &str) -> Result<LocalConfig, String> {
        let mut path_is_dir = false;

        let pathbuf = {
            match PathBuf::from_str(path) {
                Err(_) => return Err(String::from(path)),
                Ok(mut buf) => {
                    if buf.is_dir() {
                        buf.push("fejix.toml");
                        path_is_dir = true;
                    }
                    buf
                }
            }
        };

        match fs::read_to_string(&pathbuf) {
            Err(msg) => {
                let mut total_msg = String::new();
                if path_is_dir { total_msg.push_str("cannot read fejix.toml: "); }
                total_msg.push_str(&msg.to_string());
                Err(total_msg)
            },
            Ok(s) =>
                match toml::from_str(&s) {
                    Err(msg) => Err(msg.to_string()),
                    Ok(cfg) => {
                        let parent = pathbuf.parent()
                            .expect("Local config file does not have parent directory!");
                        Ok(LocalConfig {
                            path: parent.to_path_buf(),
                            ..cfg
                        })
                    }
                }
        }
    }
}