//! Handles all the command arguments and logging

use crate::builder::tokenizer::Token;
use crate::builder::parser::ast::Rvalue;

use clap::{ArgMatches, App, Arg};
use colored::*;

use std::fmt;
use std::fmt::Display;

#[cfg(debug_assertions)]
const COMPILE_MODE: &'static str = "debug";
#[cfg(not(debug_assertions))]
const COMPILE_MODE: &'static str = "release";


pub fn get_args<'a>(default_config: &'a str) -> ArgMatches {
    App::new("Fejix compiler")
    .version(format!("0.1.1-{}", COMPILE_MODE).as_str())
    .author("Mark Lagodych <github.com/MarkLagodych>")
    
    .about("The build tool for Fejix Markup Language")

    .arg(Arg::new("quiet")
        .help("Do not print any information except errors")
        .required(false)
        .short('q')
        .long("quiet")
    )

    .arg(Arg::new("no_color")
        .help("Do not colorize output. Also triggered by NO_COLOR variable")
        .required(false)
        .short('C')
        .long("no-color")
    )

    .arg(Arg::new("global_config")
        .help("Fejix global configuration file. Overriden by FEJIX_CONFIG variable")
        .short('f')
        .long("config")
        .takes_value(true)
        .value_name("FILE")
        .default_value(default_config)
    )

    .arg(Arg::new("local_config")
        .help("Project description file; path or parent directory")
        .required(false)
        .takes_value(true)
        .value_name("FILE")
        .default_value("fejix.toml")
    )

    //.subcommand(SubCommand::with_name("build")
    //    .about("Build project in CWD or using specified .toml file")
    //)

    .arg(Arg::new("info")
        .help("Shows the description of the project")
        .required(false)
        .long("info")
        .short('i')
    )

    .arg(Arg::new("specific_link")
        .help("Forces -i to display only the contents of a specified link")
        .required(false)
        .long("--link")
        .short('l')
        .takes_value(true)
        .value_name("INFO_LINK")
    )

    .arg(Arg::new("no_links")
        .help("Prevents -i from displaying links")
        .required(false)
        .long("--no-links")
        .short('L')
    )

    .arg(Arg::new("compiler_debug")
        .help("Allows debugging output (mostly emitted in DEBUG configuration)")
        .required(false)
        .long("--compiler-debug")
        .short('Z')
    )

    .get_matches()
}


#[derive(Clone, Copy, Debug)]
pub enum BuildStage {
    Reading,
    Tokenizing,
    Parsing,
    Compiling,
}

impl Display for BuildStage {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Self::Reading => write!(f, "reading"),
            Self::Tokenizing => write!(f, "tokenizing"),
            Self::Parsing => write!(f, "parsing"),
            Self::Compiling => write!(f, "compiling"),
        }
    }
}

#[derive(Clone, Debug)]
pub struct Logger {
    // Flags -- manually set
    pub quiet: bool,
    pub colorful: bool,
    pub debug: bool,

    // Message parameters -- use set_X to modify
    file: Option<String>,
    line: Option<usize>,
    pos: Option<usize>,
    stage: Option<BuildStage>,
}


/// Important: unset_X also unsets all properties specific to X
impl Logger {
    pub fn new() -> Self {
        Self {
            quiet: false,
            colorful: true,
            debug: false,
            file: None,
            line: None,
            pos: None,
            stage: Some(BuildStage::Reading),
        }
    }

    fn highlight(&self, s: String) -> colored::ColoredString {
        if self.colorful {
            s.white().bold()
        } else {
            s.normal()
        }
    }

    fn unhighlight(&self, s: String) -> colored::ColoredString {
        if self.colorful {
            s.cyan()
        } else {
            s.normal()
        }
    }

    fn warning(&self) -> colored::ColoredString {
        if self.colorful {
            "WARNING".yellow().bold()
        } else {
            "WARNING".normal()
        }
    }

    fn error(&self) -> colored::ColoredString {
        if self.colorful {
            "ERROR".red().bold()
        } else {
            "ERROR".normal()
        }
    }


    pub fn set_file(&mut self, file: String) {
        self.file = Some(file);
    }

    /// Also unsets `line` and `pos`
    pub fn unset_file(&mut self) {
        self.file = None;
        self.unset_pos();
    }

    pub fn set_line(&mut self, line: usize) {
        self.line = Some(line);
    }

    /// Also unsets `pos`
    pub fn unset_line(&mut self) {
        self.line = None;
        self.pos = None;
    }

    pub fn set_pos(&mut self, pos: usize) {
        self.pos = Some(pos);
    }

    /// Also unsets `line`
    pub fn unset_pos(&mut self) {
        self.pos = None;
        self.line = None;
    }

    pub fn set_token_pos(&mut self, t: &Token) {
        self.set_line(t.line);
        self.set_pos(t.pos);
    }

    pub fn set_rvalue_pos(&mut self, val: &Rvalue) {
        self.set_line(val.line);
        self.set_pos(val.pos);
    }

    pub fn set_stage(&mut self, stage: BuildStage) {
        self.stage = Some(stage);
    }

    pub fn unset_stage(&mut self) {
        self.stage = None;
    }

    pub fn log<MsgType: Display>(&self, msg: MsgType) {
        if !self.quiet {
            println!("{}", msg);
        }
    }

    /// Highlighted log
    pub fn hilog<MsgType: Display>(&self, msg: MsgType) {
        if !self.quiet {
            println!("{}", self.highlight(msg.to_string()));
        }
    }

    /// Use for debugging messages
    pub fn debug<MsgType: Display>(&self, msg: MsgType) {
        if !self.quiet && self.debug {
            println!("{}", self.unhighlight(msg.to_string()));
        }
    }

    /// Gather all the known info like file, line, building stage
    fn info(&self) -> String {
        let mut s = String::new();

        if let Some(stage) = &self.stage {
            s += &format!(" while {}", stage);

            if let Some(file_name) = &self.file {
                s += &format!(" '{}'", self.highlight(file_name.to_string()));
            }
    
            if let Some(line) = self.line {
                s += &format!(" on line {}", self.highlight(line.to_string()));
            }

            if let Some(pos) = self.pos {
                s += &format!(" pos {}", pos);
            }
        }

        s
    }

    pub fn warn<MsgType: Display>(&self, msg: MsgType) {
        if !self.quiet {
            eprintln!("{}{}: {}", self.warning(), self.info(), msg);
        }
    }

    pub fn err<MsgType: Display>(&self, msg: MsgType) {
        eprintln!("{}{}: {}", self.error(), self.info(), self.highlight(msg.to_string()));
    }
}