use std::fmt;

#[derive(Debug)]
pub struct TokenizerError {
    pub token: Token,
    pub source: String,
    pub message: String,
}

impl TokenizerError {
    pub fn new(token: Token, message: String, source: String) -> Self {
        return Self { token, message, source }
    }

    pub fn from<MsgT, SrcT>(token: Token, message: MsgT, source: SrcT) -> Self
        where MsgT: ToString, SrcT: ToString
    {
        return Self { token, message: message.to_string(), source: source.to_string() }
    }
}


#[derive(Debug, Clone, PartialEq)]
pub enum TokenValue {
    /// End Of File
    EOF,

    /// End Of Line
    EOL,

    /// Represents the fact that a line is indented more then the previous one
    Indent,
    /// Not indent
    Dedent,

    Str(String),
    Id(String),
    Kw(String),

    Color(u32),
    ColorName(String),
    
    Float(f32),
    Int(u64),
    
    /// Operator
    ///
    /// | Operator |    Representation    |
    /// | -------- | -------------------- |
    /// | `==`     | `E` (Equals)         |
    /// | `!=`     | `U` (Unequals)       |
    /// | `::`     | `C` (Colons)         |
    /// | Other    | Same as the operator |
    Op(char),

    /// Grouper
    /// 
    /// `( ) [ ] { }`
    Gr(char)
}

impl TokenValue {
    pub fn to_brief_str(&self) -> String {
        match self {
            Self::EOF => String::from("EOF"),
            Self::EOL => String::from("EOL"),
            Self::Indent => format!("indent"),
            Self::Dedent => format!("dedent"),
            //Self::MetaExpr(_) => String::from("meta"),
            Self::Color(num) => format!("color ({:#x})", num),
            Self::ColorName(name) => format!("color ({})", name),
            Self::Id(name) => format!("identifier '{}'", name),
            Self::Kw(name) => format!("keyword '{}'", name),
            Self::Str(_) => format!("string"),
            Self::Float(num) => format!("number {}", num),
            Self::Int(num) => format!("number {}", num),
            Self::Gr(g) => String::from(*g),
            Self::Op(op) => 
                match *op {
                    'E' => "==".to_string(),
                    'U' => "!=".to_string(),
                    'C' => "::".to_string(),
                    _ => op.to_string(),
                },
        }
    }
}


#[derive(Debug, Clone)]
pub struct Token {
    pub line: usize,
    pub pos: usize,
    pub value: TokenValue,
}

impl Token {
    /// **!!! Creates EOF !!!**
    pub fn new() -> Self {
        Self {
            line: 1,
            pos: 1,
            value: TokenValue::EOF,
        }
    }

    pub fn from(line: usize, pos: usize, value: TokenValue) -> Self {
        Self {
            line, pos, value
        }
    }

    pub fn is_eof(&self) -> bool {
        if let &TokenValue::EOF = &self.value {
            true
        } else {
            false
        }
    }

    pub fn is_eol(&self) -> bool {
        if let &TokenValue::EOL = &self.value {
            true
        } else {
            false
        }
    }

    pub fn is_indentation(&self) -> bool {
        if let &TokenValue::Indent = &self.value {
            true
        } else if let &TokenValue::Dedent = &self.value {
            true
        } else {
            false
        }
    }

    pub fn rewind(&mut self) {
        self.line = 1;
        self.pos = 1;
    }

    /// Relatively change position
    pub fn scroll_pos(&mut self, pos: usize) {
        self.pos += pos;
    }

    /// line += 1; pos = 0
    pub fn incr_line(&mut self) {
        self.line += 1;
        self.pos = 1;
    }

    pub fn scroll_lines(&mut self, lines: usize) {
        for _ in 0..lines {
            self.incr_line();
        }
    }

    pub fn set_value(&mut self, token_value: TokenValue) {
        self.value = token_value;
    }
}