//! Provides a convenient way to read tokens through the [TokenReader] struct
//! 
//! Important terminology: `scroll` means to relatively change, flip through. If we have a list of tokens and a pointer
//! to a token inside the list, the phrase `to scroll 3 tokens` means to increase the pointer by 3.
//! If you find this word stupid, I will be happy to change it :D

use bitflags::bitflags;

use std::rc::Rc;
use std::cell::RefCell;
use std::str::ParseBoolError;

use crate::cli::Logger;
use crate::builder::tokenizer::{Token, TokenValue};
use crate::builder::parser::*;

bitflags! {
    /// Token skipping flags
    pub struct Skip: u32 {
        /// Do not skip any tokens
        const NONE = 0;
        /// Skip `Indent` or `Dedent` token
        const INDENT = 0b_0000_0001;
        /// Skip `EOL` token
        const EOL = 0b_0000_0010;
        /// Skip `EOL` and `Indent`
        const WHITE = Self::INDENT.bits | Self::EOL.bits;
    }
}


/// Provides convenient access to tokens
#[derive(Clone, Copy, Debug)]
pub struct TokenReader<'a> {
    /// Guaranteed to have EOF as the last token
    tokens: &'a Vec<Token>,
    token_index: usize,
}

impl<'a> TokenReader<'a> {
    /// Constructs a new token reader from the given well-formed token list
    /// 
    /// See what is "well-formed": [crate::builder::tokenizer::tokenize]
    pub fn with_tokens(tokens: &'a Vec<Token>) -> Option<Self> {
        let last = tokens.len() - 1;

        if tokens.get(last).is_none() {
            return None
        }

        if let TokenValue::EOF = tokens.get(last).unwrap().value
            {}
        else {
            return None
        }

        Some(Self {
            tokens,
            token_index: 0
        })
    }

    /// Gets the current token index (starts with 0)
    pub fn index(&self) -> usize { self.token_index }

    /// Relatively changes token index
    /// 
    /// If applying delta causes the token index to go out of bounds, sets it to min/max
    pub fn scroll(&mut self, delta: isize) {
        if self.token_index > (isize::MAX as usize) && delta > 0 {
            return;
        }

        let new_index = self.token_index as isize + delta;

        if new_index < 0 {
            self.token_index = 0;
        } else if new_index as usize > self.tokens.len()-1 {
            self.token_index = self.tokens.len()-1;
        } else {
            self.token_index = new_index as usize;
        }
    }

    /// Returns the current token (if the end is reached, always returns EOF)
    pub fn peek_token(&self) -> &'a Token {
        match self.tokens.get(self.index()) {
            None => &self.tokens[self.tokens.len()-1],
            Some(token) => &token,
        }
    }

    /// Same as [TokenReader::peek_token] but increases token index (scrolls 1 token)
    /// 
    /// Does not scroll if reached EOF
    pub fn get_token(&mut self) -> &'a Token {
        let token = self.peek_token();
        self.scroll(1);
        token
    }

    /// Finds the next token, skipping tokens of specified types
    /// 
    /// Returns the found token and number of scrolled tokens (including the token found)
    pub fn peek_next_token(&mut self, skip: Skip) -> (&'a Token, isize) {
        let mut scrolled = 0;

        while !self.peek_token().is_eof() {

            match self.peek_token().value {
                TokenValue::Indent | TokenValue::Dedent => {
                    if !skip.contains(Skip::INDENT) {
                        break;
                    }
                },

                TokenValue::EOL => {
                    if !skip.contains(Skip::EOL) {
                        break;
                    }
                },

                _ => { break; }
            }

            self.scroll(1);
            scrolled += 1;

        }

        let token = self.peek_token();
        self.scroll(-scrolled);

        (token, scrolled+1)
    }

    /// Same as [TokenReader::peek_next_token] but automatically increases token index
    pub fn get_next_token(&mut self, skip: Skip) -> &'a Token {
        let (token, delta) = self.peek_next_token(skip);
        self.scroll(delta);
        token
    }

    /// Same as [TokenReader::get_token] but checks if the value is the same
    /// 
    /// If value differs, does not scroll
    pub fn require_value(&mut self, value: TokenValue) -> Option<&'a Token> {
        let token = self.peek_token();
        if token.value == value {
            self.scroll(1);
            Some(token)
        } else {
            None
        }
    }

    /// Same as [TokenReader::require_value] but also skips specified tokens
    /// 
    /// If value differs, does not scroll
    pub fn require_next_value(&mut self, skip: Skip, value: TokenValue) -> Option<&'a Token>
    {
        let (token, delta) = self.peek_next_token(skip);
        if token.value == value {
            self.scroll(delta);
            Some(token)
        } else {
            None
        }
    }


    pub fn skip_tokens(&mut self, skip: Skip) {
        let (_, delta) = self.peek_next_token(skip);
        self.scroll(delta-1); // Saves the found token
    }

}


#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_peek_next_token() {
        let a = TokenValue::Id("a".to_string());
        let b = TokenValue::Id("b".to_string());
        let c = TokenValue::Id("c".to_string());

        let tokens = vec![
            Token{ line: 1, pos: 1, value: a.clone() },
            Token{ line: 1, pos: 2, value: TokenValue::EOL },

            Token{ line: 2, pos: 1, value: TokenValue::Indent },
            Token{ line: 2, pos: 2, value: b.clone() },
            Token{ line: 2, pos: 3, value: TokenValue::EOL },
            
            Token{ line: 3, pos: 1, value: TokenValue::Dedent },
            Token{ line: 3, pos: 2, value: c.clone() },
            Token{ line: 3, pos: 3, value: TokenValue::EOL },

            Token{ line: 3, pos: 4, value: TokenValue::EOF },
        ];

        let mut reader = TokenReader::with_tokens(&tokens).unwrap();

        assert_eq!(
            a,
            reader.require_next_value(Skip::NONE, a.clone())
                .unwrap()
                .value
        );

        assert_eq!(
            b,
            reader.require_next_value(Skip::WHITE, b.clone())
                .unwrap()
                .value
        );

        assert_eq!(
            c,
            reader.require_next_value(Skip::WHITE, c.clone())
                .unwrap()
                .value
        );
    }
}