#![macro_use]

use super::*;
use crate::builder::tokenizer::{Token, TokenValue, self};

#[derive(Debug, Clone)]
pub enum ParserError<'a> {
    /// Not an error, just the file ended.
    EOF,

    InvalidToken {
        found: &'a Token,
        expected: &'static str,
    }
}


pub type ParserResult<'a, T> = Result<(T, TokenReader<'a>), ParserError<'a>>;


impl<'a> ParserError<'a> {
    pub fn with_expectation(self, expectation: &'static str) -> Self {
        match self {
            Self::InvalidToken{found, expected} =>
                Self::InvalidToken{found, expected: expectation},

            s => s
        }
    }

    pub fn is_eof(&self) -> bool {
        if let ParserError::EOF = self {
            true
        } else {
            false
        }
    }
}


macro_rules! set_expectation {
    () => {
        compile_error!("Usage: xxx.on_err(set_expectation!('Expectation')) where xxx is ParserResult");
    };

    ($expectation:expr) => {
        |err| ParserResult::Err(err.with_expectation($expectation))
    };
}


macro_rules! on_eof {
    () => {
        compile_error!("Usage: Usage: xxx.on_err(on_eof!{{...}}) where xxx is ParserResult");
    };

    ($body:block) => {
        |err| {
            if err.is_eof()
                $body
            Err(err)
        }
    };
}


macro_rules! ret_invalid {
    () => {
        compile_error!("Usage: ret_invalid(token, 'expected this');");
    };

    ($token:expr, $expected:expr) => {
        return ParserResult::Err(
            ParserError::InvalidToken{
                found: $token,
                expected: $expected,
            }
        )
    };
}

macro_rules! ret_eof {
    () => {
        return ParserResult::Err(ParserError::EOF)
    };
}


macro_rules! audit_assert {
    () => {
        compile_error!("Usage: audit_assert(audit_xxx(...)); If fails, returns false from current function");
    };

    ($e:expr) => {
        if !$e { return false; }
    };
}


pub fn audit_empty<'a>(reader: TokenReader<'a>) -> bool {
    reader.peek_token().value == TokenValue::Op('!')
}


pub fn parse_empty_statement<'a>(mut reader: TokenReader<'a>) -> ParserResult<StatementValue<'a>> {
    if reader.require_value(TokenValue::Op('!')).is_some() {
        Ok((StatementValue::Empty, reader))
    } else {
        ret_invalid!(reader.peek_token(), "'!'");
    }
}


pub fn audit_value<'a>(reader: TokenReader<'a>, value: TokenValue) -> bool {
    reader.peek_token().value == value
}


pub fn audit_rvalue<'a>(reader: TokenReader<'a>) -> bool {
    Rvalue::audit(reader.peek_token())
}


pub fn parse_rvalue<'a>(mut reader: TokenReader<'a>) -> ParserResult<Rvalue<'a>> {
    let token = reader.get_token();
    match Rvalue::from(token) {
        None => ret_invalid!(token, "rvalue"),
        Some(rvalue) => Ok((rvalue, reader))
    }
}


pub fn audit_id<'a>(reader: TokenReader<'a>) -> bool {
    if !audit_rvalue(reader.clone()) {
        return false;
    }

    let (rvalue, _) = parse_rvalue(reader).unwrap();

    if let Primitive::Id(_) = rvalue.prim {
        true
    } else {
        false
    }
}

/// Only parses an identifier (makes life much easier!)
pub fn parse_id<'a>(mut reader: TokenReader<'a>) -> ParserResult<Rvalue>
{
    let (rvalue, new_reader) = parse_rvalue(reader.clone())
        .or_else(set_expectation!("identifier"))?;

    if let Primitive::Id(_) = rvalue.prim {
        Ok((rvalue, new_reader))
    }
    else {
        let token = reader.get_token();
        ret_invalid!(token, "identifier");
    }
}


pub fn audit_kw<'a>(reader: TokenReader<'a>) -> bool {
    if let TokenValue::Kw(_) = reader.peek_token().value {
        true
    }
    else {
        false
    }
}

pub fn audit_exact_kw<'a>(reader: TokenReader<'a>, exact_kw: &'a str) -> bool {
    if let TokenValue::Kw(kw) = &reader.peek_token().value {
        kw == exact_kw
    }
    else {
        false
    } 
}


pub fn parse_kw<'a>(mut reader: TokenReader<'a>) -> ParserResult<Keyword<'a>>
{
    let token = reader.get_token();
    if let TokenValue::Kw(kw) = &token.value {
        Ok((Keyword{kw: &kw, token}, reader))
    } else {
        ret_invalid!(token, "keyword")
    }
}


/// Parses a comma-separated list
pub fn parse_rvalue_list<'a>(mut reader: TokenReader<'a>)
-> ParserResult< Vec<Rvalue<'a>> >
{

    let mut prims = Vec::<Rvalue>::new();

    while let Ok((prim, new_reader)) = parse_rvalue(reader.clone()) {
        prims.push(prim);
        reader = new_reader;
        if reader.require_value(TokenValue::Op(',')).is_none() {
            break;
        }
    }

    Ok((prims, reader))
}


pub fn audit_anno<'a>(reader: TokenReader<'a>) -> bool {
    reader.peek_token().value == TokenValue::Op('@')
}


pub fn parse_anno<'a>(mut reader: TokenReader<'a>)-> ParserResult< Annotation<'a> >
{
    reader.require_value(TokenValue::Op('@'));

    let (name, new_reader) = parse_id(reader.clone())
        .or_else(set_expectation!("annotation name"))
        ?;

    reader = new_reader;

    let mut args = Vec::<Rvalue<'a>>::new();

    if reader.require_value(TokenValue::Gr('(')).is_some() {
        let (rvalues, new_reader) = parse_rvalue_list(reader.clone())
            .or_else(set_expectation!("annotation arguments"))
            ?;

        reader = new_reader;
     
        if reader.require_value(TokenValue::Gr(')')).is_none() {
            ret_invalid!(reader.get_token(), "')' (end of annotation arguments)");
        }

        args = rvalues;
    }

    Ok((
        Annotation{name, args},
        reader
    ))
}


pub fn audit_block<'a>(mut reader: TokenReader<'a>) -> bool {
    if reader.require_next_value(Skip::WHITE, TokenValue::Gr('{')).is_some() {
        true
    } else if reader.require_next_value(Skip::EOL, TokenValue::Indent).is_some() {
        true
    } else {
        false
    }
}


pub fn parse_block<'a>(mut reader: TokenReader<'a>)-> ParserResult< Vec<Statement<'a>> >
{
    let mut body = Vec::<Statement>::new();

    // {;} block
    if let Some(_) = reader.require_next_value(Skip::WHITE, TokenValue::Gr('{')) {

        loop {
            if reader.require_value(TokenValue::Op(';')).is_some() {
                continue;
            }

            if reader.require_value(TokenValue::Gr('}')).is_some() {
                break;
            }

            let (stmt, new_reader) = parse_statement(reader.clone())
                .or_else( on_eof!{{ ret_invalid!(reader.get_token(), "end of block") }} )
                ?;
            reader = new_reader;
                
            body.push(stmt);

            if reader.require_value(TokenValue::Gr('}')).is_some() {
                break;
            }

            if reader.require_value(TokenValue::Op(';')).is_none() {
                ret_invalid!(reader.get_token(), "statement separator ';'")
            }
        }
    }

    // Indent-EOL block
    else {

        reader.require_next_value(Skip::EOL, TokenValue::Indent);

        loop {

            let (stmt, new_reader) = parse_statement(reader.clone())?;
            reader = new_reader;

            body.push(stmt);

            if reader.require_next_value(Skip::EOL, TokenValue::Dedent).is_some() {
                break;
            }

        }

    }

    // Eat EOL if present
    reader.require_value(TokenValue::EOL);

    Ok((body, reader))
}