//! Defines all structures needed for AST (Abstract Syntax Tree)

use crate::builder::common::color_name::color_name_to_num;
use crate::builder::tokenizer::{Token, TokenValue};

use std::fmt;


/// Holds u64 or f32
#[derive(Debug, Clone, Copy)]
pub enum Number {
    Int(u64),
    Real(f32),
}

impl Number {
    pub fn to_string(&self) -> String {
        match self {
            Self::Int(i) => i.to_string(),
            Self::Real(r) => r.to_string()
        }
    }
}

#[derive(Debug, Clone, Copy)]
pub struct Color {
    pub a: u8,
    pub r: u8,
    pub g: u8,
    pub b: u8,
}

#[derive(Debug, Clone, Copy)]
pub enum Primitive<'a> {
    Number(Number),
    String(&'a str),
    Id(&'a str),
    Color(Color),
}

#[derive(Debug, Clone, Copy)]
pub struct Rvalue<'a> {
    pub line: usize,
    pub pos: usize,
    pub prim: Primitive<'a>,
}


/// Abstract Syntax Tree
#[derive(Debug)]
pub struct AST<'a> {
    pub statements: Vec<Statement<'a>>
}

impl<'a> AST<'a> {
    pub fn new() -> Self {
        Self {
            statements: Vec::<Statement>::new()
        }
    }
}


#[derive(Debug)]
pub enum StatementValue<'a> {
    Empty,
    Rvalue(Rvalue<'a>),
    //RvalueList(Vec<Rvalue<'a>>),
    Block(Vec<Statement<'a>>),
    Kw(Box<KeywordStatement<'a>>),
    Decl(Box<Declaration<'a>>),
    TypeDecl(Box<TypeDeclaration<'a>>),
    Spec(Box<Specification<'a>>),
}


impl<'a> StatementValue<'a> {
    pub fn get_rvalue(&self) -> Option<&Rvalue<'a>> {
        if let Self::Rvalue(x) = self {Some(&x)} else {None}
    }

    pub fn get_decl(&self) -> Option<&Declaration<'a>> {
        if let Self::Decl(x) = self {Some(&x)} else {None}
    }

    pub fn get_type_decl(&self) -> Option<&TypeDeclaration<'a>> {
        if let Self::TypeDecl(x) = self {Some(&x)} else {None}
    }

    pub fn get_spec(&self) -> Option<&Specification<'a>> {
        if let Self::Spec(x) = self {Some(&x)} else {None}
    }

    pub fn get_block(&self) -> Option<&Vec<Statement<'a>>> {
        if let Self::Block(x) = self {Some(&x)} else {None}
    }
}


#[derive(Debug)]
pub struct Statement<'a> {
    pub value: StatementValue<'a>,
    pub annos: Vec<Annotation<'a>>,
}


#[derive(Debug)]
pub struct Annotation<'a> {
    pub name: Rvalue<'a>,
    pub args: Vec<Rvalue<'a>>,
}




impl<'a> Rvalue<'a> {
    pub fn from(token: &'a Token) -> Option<Self> {
        let mut me = Self {
            line: token.line,
            pos: token.pos,
            // Will be set later
            prim: Primitive::Number(Number::Real(0.0)),
        };

        match &token.value {
            TokenValue::Int(num) => me.prim = Primitive::Number(Number::Int(*num)),
            TokenValue::Float(num) => me.prim = Primitive::Number(Number::Real(*num)),
            TokenValue::Id(name) => me.prim = Primitive::Id(name),
            TokenValue::Color(color_num) => me.prim = Primitive::Color(Color::from(*color_num)),
            TokenValue::Str(s) => me.prim = Primitive::String(s),

            TokenValue::ColorName(name) => {
                let num = color_name_to_num(name)?;
                me.prim = Primitive::Color(Color::from(num));
            }

            _ => return None
        }

        Some(me)
    }

    pub fn new_num() -> Self {
        Self {
            line: 0, pos: 0,
            prim: Primitive::Number(Number::Int(0))
        }
    }

    pub fn new_str() -> Self {
        Self {
            line: 0, pos: 0,
            prim: Primitive::String("")
        }
    }

    pub fn new_color() -> Self {
        Self {
            line: 0, pos: 0,
            prim: Primitive::Color(Color::from(0x00_00_00_00))
        }
    }


    /// Returns: true if token can be converted to rvalue
    pub fn audit(token: &'a Token) -> bool {
        match &token.value {
            TokenValue::Int(_)
            | TokenValue::Float(_)
            | TokenValue::Id(_)
            | TokenValue::Color(_)
            | TokenValue::Str(_)
            | TokenValue::ColorName(_)
                => true,

            _ => false
        }
    }


    /// Light-weight method that just gets the contents of Id or String
    pub fn get_text(&self) -> &'a str {
        match &self.prim {
            Primitive::Id(id) => id,
            Primitive::String(s) => s,
            _ => "[non-text value]"
        }
    }

    /// Creates a string representation of the value
    /// 
    /// See also: [Self::get_text]
    pub fn to_string(&self) -> String {
        match self.prim {
            Primitive::Id(id) => id.to_string(),
            Primitive::String(s) => s.to_string(),
            Primitive::Color(c) => c.to_string(),
            Primitive::Number(n) => n.to_string(),
        }
    }
}




impl Color {
    pub fn from(color_num: u32) -> Self {
        // 0x04_03_02_01 -> [4, 3, 2, 1]
        let bytes = color_num.to_be_bytes();
        Self {
            a: bytes[0],
            r: bytes[1],
            g: bytes[2],
            b: bytes[3] 
        }
    }

    pub fn to_string(&self) -> String {
        format!("(A {}, R {}, G {}, B {})",
            self.a, self.r, self.g, self.b
        )
    }
}


#[derive(Debug, Clone)]
pub struct Keyword<'a> {
    pub kw: &'a str,
    pub token: &'a Token,
}




#[derive(Debug)]
pub struct Declaration<'a> {
    pub name: Option<Rvalue<'a>>,
    pub type_name: Rvalue<'a>,
    pub body: Option< Box<Statement<'a>> >,
}


#[derive(Debug)]
pub struct TypeDeclaration<'a> {
    pub name: Rvalue<'a>,
    pub body: Option< Box<Statement<'a>> >,
}


#[derive(Debug)]
pub struct Specification<'a> {
    pub name: Rvalue<'a>,
    pub body: Box<Statement<'a>>,
}


/// Keyword statement
#[derive(Debug)]
pub struct KeywordStatement<'a> {
    pub keyword: Keyword<'a>,
    pub rvalue_arg: Option<Rvalue<'a>>,
    pub statement_arg: Option< Box<Statement<'a>> >,
}