use super::{atoms::*, compile_statement};

use crate::cli::Logger;

#[cfg(debug_assertions)]
#[cfg(feature = "perform_debug")]
pub fn perform_debug<'a> (
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>
) -> Result<(),()> {

    if let ast::StatementValue::Block(stmts) = &stmt.value {
        for stmt in stmts {
            for anno in &stmt.annos {
                match anno.name.get_text() {
                    "overview" => {
                        log.debug(format!("{:#?}", fjml));
                    }

                    "print" => {
                        if anno.args.len() > 0 {
                            log.debug(anno.args[0].get_text());
                        }
                    }

                    "inspect" => {
                        if anno.args.len() > 0 {
                            let name = anno.args[0].get_text();

                            let id = fjml.lookup_global(name);
                            if id.is_none() {
                                log.debug(format!("{}: no such global", name));
                                continue;
                            }
                            let id = id.unwrap();

                            let class =
                                if fjml.is_type(id) {"type"}
                                else if fjml.is_object(id) {"object"}
                                else if fjml.is_value(id) {"value"}
                                else {"unassigned"};

                            let value = match class {
                                "value" => fjml.values[&id].to_string(),
                                "type"|"object" => format!("{:#?}", fjml.objects[&id]),
                                _ => "[none]".to_string()
                            };

                            let rvalue = &fjml.names[&id];

                            log.debug(format!(
                                "@{}:{} #{} <{}> {} = {}",
                                rvalue.line, rvalue.pos, id,
                                name, class,
                                value
                            ));
                        }
                    }

                    _ => {}
                }
            }
        }
    }

    Ok(())
}


fn handle_definition<'a>(
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    name: Option<Rvalue<'a>>,
    parent: Option<ID>,
) -> Option<ID> {
    
    if let Some(val) = name {
        if fjml.lookup(parent, val.get_text()).is_some() {
            log.set_rvalue_pos(&val);
            log.err("already defined");
            let prev = fjml.lookup(parent, val.get_text()).unwrap();
            log.set_rvalue_pos(fjml.get_name(prev).unwrap());
            log.err("here");
            return None;
        }
    }

    let id = fjml.define_maybe_named(name);

    if parent.is_some() {
        fjml.add_to_object(parent.unwrap(), id);
    } else {
        fjml.make_global(id);
    }

    Some(id)
}


pub fn handle_decl<'a>(
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()> {
    let decl = stmt.value.get_decl().unwrap();

    let id = handle_definition(&mut fjml, &mut log, decl.name, parent);
    if id.is_none() {return Err(());}
    let id = id.unwrap();
    

    let type_name = decl.type_name.get_text();

    let core_types = ["Num", "Str", "Color"];

    // If this is a primitive
    if core_types.contains(&type_name) {
        // The value is given
        if decl.body.is_some() {
            // The value is rvalue
            if let Some(val) = decl.body.as_ref().unwrap().value.get_rvalue() {
                // Identifier given - use the value of that identifier
                if let Primitive::Id(global_name) = val.prim {
                   /*  // This identifier exists
                    if let Some(global_id) = fjml.lookup_global(global_name) {
                        // This identifier refers to an rvalue
                        if fjml.is_value(id) {
                            // Clone taht value
                            fjml.make_value(id, fjml.values[&global_id].clone());
                        // The identifier does not refer to an rvalue
                        } else {
                            log.set_rvalue_pos(&val);
                            log.err("not an rvalue");
                            return true;
                        }
                    // This identifier does not exist
                    } else {
                        log.set_rvalue_pos(&val);
                        log.err("undefined");
                        return true;
                    } */
                    log.set_rvalue_pos(val);
                    log.err("cannot assign identifier");
                    return Err(());
                }
                // Just assign rvalue
                else {
                    fjml.make_value(id, val.clone());
                }
            // The value is not rvalue!
            } else {
                log.set_rvalue_pos(&decl.type_name);
                log.err("expected rvalue, got something else");
                return Err(());
            }
        // I should assign a default value for that type
        } else {
            let rvalue = match type_name {
                "Num" => Rvalue::new_num(),
                "Str" => Rvalue::new_str(),
                "Color" => Rvalue::new_color(),
                _ => { unimplemented!("new type?"); }
            };
            fjml.make_value(id, rvalue);
        }

    }

    // Else, this is a structure
    else {
        
        let type_id = fjml.lookup_global(decl.type_name.get_text());
        if type_id.is_none() {
            log.set_rvalue_pos(&decl.type_name);
            log.err("undefined type");
            return Err(());
        }
        let type_id = type_id.unwrap();

        fjml.make_object(id);
        fjml.set_type(id, type_id);
        fjml.set_parent(id, parent);

        if let Some(stmt) = &decl.body {
            return compile_statement(&mut fjml, &mut log, &stmt, Some(id));
        }

    }

    Ok(())
}


pub fn handle_type_decl<'a> (
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()> {

    let decl = stmt.value.get_type_decl().unwrap();

    if parent.is_some() {
        log.set_rvalue_pos(&decl.name);
        log.err("types cannot be toplevel");
        return Err(());
    }


    let id = handle_definition(&mut fjml, &mut log, Some(decl.name), None);
    if id.is_none() {
        return Err(());
    }
    let id = id.unwrap();

    fjml.make_object(id);
    fjml.make_type(id);

    if let Some(stmt) = &decl.body {
        return compile_statement(&mut fjml, &mut log, &stmt, Some(id));
    }

    // TODO extends and classes - needs annos

    Ok(())
}


pub fn handle_spec<'a> (
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()> {

    let spec = stmt.value.get_spec().unwrap();

    let lookup_result = fjml.lookup(parent, spec.name.get_text());

    if lookup_result.is_none() {
        log.set_rvalue_pos(&spec.name);
        log.err(format!("name '{}' does not exist in '{}'",
            spec.name.get_text(),
            fjml.get_scope_path(parent)
        ));
        return Err(());
    }

    let id = lookup_result.unwrap();

    if fjml.is_value(id) {
        if let Some (val) = spec.body.value.get_rvalue() {
            fjml.values.insert(id, val.clone());
        } else {
            log.set_rvalue_pos(&spec.name);
            log.err("expected value, got something else");
            return Err(());
        }
    } else {
        return compile_statement(&mut fjml, &mut log, &spec.body, Some(id));
    }

    Ok(())


}
