//! Semantic analyzer of AST
//! 
//! Only compiles one AST at a time, performs no linkage

///////////////////////////////////////////////////////////////////
/// Syntax for all compiler functions:
/*
pub fn handle_X<'a> (
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()> {
    // ...
}
*/
/// Returns: true if errors occured
/////////////////////////////////////////////////////////////////////

pub mod atoms;

mod util;

use std::process::Command;

pub use atoms::*;

use util::*;

use crate::cli::Logger;
use crate::config::BuildConfig;


pub fn compile<'a>(
    mut log: &mut Logger,
    ast: &'a ast::AST<'a>,
    cfg: &Option<BuildConfig>,
) -> Result<FjML<'a>, String>
{
    let mut fjml = FjML::new();

    let mut error = false;

    for stmt in &ast.statements {

        if stmt.annos.len() > 0 && stmt.annos[0].name.get_text() == "__debug__" {
            #[cfg(all(debug_assertions, feature = "perform_debug"))] {
                perform_debug(&mut fjml, &mut log, &stmt, None);
            }
            continue;
        }

        error = compile_statement(&mut fjml, &mut log, stmt, None).is_err();
    }

    if !error {
        Ok(fjml)
    } else {
        Err("code contains errors".to_string())
    }
}


fn compile_statement<'a>(
    mut fjml: &mut FjML<'a>,
    mut log: &mut Logger,
    stmt: &ast::Statement<'a>,
    parent: Option<ID>,
) -> Result<(),()>
{
    match &stmt.value {
        ast::StatementValue::Decl(decl) => {
            return handle_decl(&mut fjml, &mut log, &stmt, parent);
        }

        ast::StatementValue::TypeDecl(tdecl) => {
            return handle_type_decl(&mut fjml, &mut log, &stmt, parent);
        }

        ast::StatementValue::Spec(spec) => {
            return handle_spec(&mut fjml, &mut log, &stmt, parent);
        }

        ast::StatementValue::Block(stmts) => {
            for stmt in stmts {
                compile_statement(&mut fjml, &mut log, stmt, parent)?;
            }
        }

        _ => {}
    }

    Ok(())
}