use crate::{EntryID, FeedID};
use serde_derive::{Deserialize, Serialize};

#[derive(Debug, Deserialize)]
pub struct Entry {
    pub id: EntryID,
    pub feed_id: FeedID,
    pub title: Option<String>,
    pub url: String,
    pub extracted_content_url: String,
    #[serde(default)]
    pub author: Option<String>,
    #[serde(default)]
    pub content: Option<String>,
    #[serde(default)]
    pub summary: Option<String>,
    pub published: String,
    pub created_at: String,
    #[serde(default)]
    pub original: Option<OriginalContent>,
    #[serde(default)]
    pub images: Option<EntryImages>,
    #[serde(default)]
    pub enclosure: Option<EntryEnclosure>,
    #[serde(default)]
    pub extracted_articles: Option<Vec<ExtractedArticle>>,
}

impl Entry {
    #[allow(clippy::type_complexity)]
    pub fn decompose(
        self,
    ) -> (
        EntryID,
        FeedID,
        Option<String>,
        String,
        String,
        Option<String>,
        Option<String>,
        Option<String>,
        String,
        String,
        Option<OriginalContent>,
        Option<EntryImages>,
        Option<EntryEnclosure>,
        Option<Vec<ExtractedArticle>>,
    ) {
        (
            self.id,
            self.feed_id,
            self.title,
            self.url,
            self.extracted_content_url,
            self.author,
            self.content,
            self.summary,
            self.published,
            self.created_at,
            self.original,
            self.images,
            self.enclosure,
            self.extracted_articles,
        )
    }
}

#[derive(Debug, Serialize)]
pub struct UnreadEntriesInput {
    pub unread_entries: Vec<i64>,
}

#[derive(Debug, Deserialize)]
pub struct EntryImages {
    pub original_url: String,
    #[serde(default)]
    pub size_1: Option<SizeOneImage>,
}

impl EntryImages {
    pub fn decompose(self) -> (String, Option<SizeOneImage>) {
        (self.original_url, self.size_1)
    }
}

#[derive(Debug, Deserialize)]
pub struct SizeOneImage {
    pub cdn_url: String,
    pub width: u32,
    pub height: u32,
}

impl SizeOneImage {
    pub fn decompose(self) -> (String, u32, u32) {
        (self.cdn_url, self.width, self.height)
    }
}

#[derive(Debug, Deserialize)]
pub struct EntryEnclosure {
    pub enclosure_url: String,
    pub enclosure_type: String,
    #[serde(default)]
    pub enclosure_length: Option<String>,
    #[serde(default)]
    pub itunes_duration: Option<String>,
    #[serde(default)]
    pub itunes_image: Option<String>,
}

impl EntryEnclosure {
    pub fn decompose(
        self,
    ) -> (
        String,
        String,
        Option<String>,
        Option<String>,
        Option<String>,
    ) {
        (
            self.enclosure_url,
            self.enclosure_type,
            self.enclosure_length,
            self.itunes_duration,
            self.itunes_image,
        )
    }
}

#[derive(Debug, Deserialize)]
pub struct ExtractedArticle {
    pub url: String,
    #[serde(default)]
    pub title: Option<String>,
    #[serde(default)]
    pub host: Option<String>,
    #[serde(default)]
    pub author: Option<String>,
    #[serde(default)]
    pub content: Option<String>,
}

impl ExtractedArticle {
    pub fn decompose(
        self,
    ) -> (
        String,
        Option<String>,
        Option<String>,
        Option<String>,
        Option<String>,
    ) {
        (self.url, self.title, self.host, self.author, self.content)
    }
}

#[derive(Debug, Deserialize)]
pub struct OriginalContent {
    #[serde(default)]
    pub author: Option<String>,
    #[serde(default)]
    pub content: Option<String>,
    #[serde(default)]
    pub title: Option<String>,
    #[serde(default)]
    pub url: Option<String>,
    #[serde(default)]
    pub entry_id: Option<String>,
    #[serde(default)]
    pub published: Option<String>,
}

impl OriginalContent {
    #[allow(clippy::type_complexity)]
    pub fn decompose(
        self,
    ) -> (
        Option<String>,
        Option<String>,
        Option<String>,
        Option<String>,
        Option<String>,
        Option<String>,
    ) {
        (
            self.author,
            self.content,
            self.title,
            self.url,
            self.entry_id,
            self.published,
        )
    }
}

#[derive(Debug, Serialize)]
pub struct UpdateEntryUnreadInput {
    pub unread_entries: Vec<EntryID>,
}

#[derive(Debug, Serialize)]
pub struct UpdateEntryStarredInput {
    pub starred_entries: Vec<EntryID>,
}

#[cfg(test)]
mod tests {
    use super::Entry;

    #[tokio::test(flavor = "current_thread")]
    async fn entry_json() {
        #[allow(clippy::invisible_characters)]
        let json = r#"[{"id":2508416375,"feed_id":125584,"title":"Ein Fall wie aus dem Lehrbuch","author":"Udo Vetter","summary":"Manche echten Fälle klingen wie aus einem Jura-Lehrbuch Zivilrecht. So mussten die Gerichte ein Frankfurt nun entscheiden, ob der Verkauf eines BMW auf ebay wirksam ist, obwohl der Kaufpreis für das Fahrzeug nur einen Euro betrug. Wert ist das Auto nämlich","content":"\u003cp\u003eManche echten Fälle klingen wie aus einem Jura-Lehrbuch Zivilrecht. So mussten die Gerichte ein Frankfurt nun entscheiden, ob der Verkauf eines BMW auf ebay wirksam ist, obwohl der Kaufpreis für das Fahrzeug nur einen Euro betrug. Wert ist das Auto nämlich mindestens 12.000,00 Euro. \u003c/p\u003e\n\u003cp\u003eDer Verkäufer hatte wohl eine Auktion starten wollen, ermöglichte aber versehentlich einen Sofortkauf zum Startpreis. Dieses Schnäppchen wollte sich der Käufer nicht entgehen lassen. Er klagte bis zum Oberlandesgericht in Frankfurt. \u003c/p\u003e\n\u003cp\u003eLetztlich unterlag der Käufer, denn nicht nur aus dem versehentlich zum Sofortkauf freigegebenen Startpreis, sondern auch aus dem Angebotstext ergab sich laut dem Gericht, dass der Verkäufer dieses Geschäft jedenfalls so nicht wollte. Somit gab es eben keine übereinstimmenden Willenserklärungen, die für einen Vertragsschluss notwendig sind. Den Fall kann somit an sich jeder Student im dritten Semester recht zuverlässig lösen. Der Anwalt des Klägers ist übrigens nicht unbedingt schuld. Der Mandant bekommt halt seinen Willen, wenn er unbedingt darauf besteht. \u003c/p\u003e\n\u003cp\u003e\u003ca href=\"https://www.spiegel.de/wirtschaft/ebay-panne-verkaeufer-muss-bmw-nicht-fuer-einen-euro-herausgeben-a-1718e4cb-b644-4130-9c45-f3f118af8a60\"\u003eNäheres kann man auf Spiegel Online nachlesen.\u003c/a\u003e  \u003c/p\u003e\n","url":"https://www.lawblog.de/index.php/archives/2020/07/09/ein-fall-wie-aus-dem-lehrbuch/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/575f4dba8737b4352a95a1b9cbdb464f526f2dc6?base64_url=aHR0cHM6Ly93d3cubGF3YmxvZy5kZS9pbmRleC5waHAvYXJjaGl2ZXMvMjAyMC8wNy8wOS9laW4tZmFsbC13aWUtYXVzLWRlbS1sZWhyYnVjaC8=","published":"2020-07-09T13:00:27.000000Z","created_at":"2020-07-09T13:34:41.447286Z"},{"id":2508581568,"feed_id":64661,"title":"\u003cNextGear\u003e Boah ey, dieser Corona ist echt anstrengend!","author":null,"summary":"\u003cNextGear\u003e Boah ey, dieser Corona ist echt anstrengend! \u003cB166ER\u003e Stimmt, wenn diese ganze Coronakrise vorbei ist bleib ich erstmal 2 Wochen zuhause. \u003cFAME\u003e Wie geil xD","content":"\u0026lt;NextGear\u0026gt; Boah ey, dieser Corona ist echt anstrengend!\u003cbr\u003e\n\u0026lt;B166ER\u0026gt; Stimmt, wenn diese ganze Coronakrise vorbei ist bleib ich erstmal 2 Wochen zuhause.\u003cbr\u003e\n\u0026lt;FAME\u0026gt; Wie geil xD","url":"http://www.ibash.de/zitat_60539.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/3d3f2804cdf6f0f4dd1ea2a0c5ced7aa35728984?base64_url=aHR0cDovL3d3dy5pYmFzaC5kZS96aXRhdF82MDUzOS5odG1s","published":"2020-07-09T15:54:37.997464Z","created_at":"2020-07-09T15:54:37.997438Z"},{"id":2509606430,"feed_id":125584,"title":"Kinder u.a. müssen draußen bleiben","author":"Udo Vetter","summary":"Darf ein Tagungs- und Wellnesshotel Kinder unter 16 Jahren als Gäste ablehnen? Diese Frage hat der Bundesgerichtshof nun beantwortet. Man sieht schon an der Frage unschwer, dass die Entscheidung weitreichende Bedeutung haben kann. Denn auch Restaurants,","content":"\u003cp\u003eDarf ein Tagungs- und Wellnesshotel Kinder unter 16 Jahren als Gäste ablehnen? Diese Frage hat der Bundesgerichtshof nun beantwortet. Man sieht schon an der Frage unschwer, dass die Entscheidung weitreichende Bedeutung haben kann. Denn auch Restaurants, Cafés und andere Anbieter führen immer mal wieder eine Adults-only-Regel ein – und ernten dafür meist massive Shitstorms.\u003c/p\u003e\n\u003cp\u003eUnrechtmäßig ist das Verhalten des Hotels jedenfalls nicht, urteilt der Bundesgerichtshof. Eine unzulässige Benachteiligung von Kindern liege nur vor, wenn es keinen sachlichen Grund für ihren Ausschluss gibt. Das sei bei dem Hotel aber gerade nicht der Fall; der Betrieb sei vom Angebot her auf Ruhe und Entspannung angelegt. Das sei ausreichend. \u003c/p\u003e\n\u003cp\u003eDemgegenüber hätten die fünf Kinder, deren Mutter im Hotel ein Angebot eingeholt hatte und abgewiesen wurde, kein durchgreifendes Interesse, ausgerechnet in diesem Hotel unterzukommen. Was sich inbesondere daraus ergibt, dass es in der fraglichen Region wohl noch genug andere Hotels gab, in denen Kinder als Gäste willkommen sind.\u003c/p\u003e\n\u003cp\u003eDamit sind wir beim großen Aber in der Entscheidung. Je mehr es um Güter und Dienstleistungen geht, die „zur täglichen Lebensgestaltung oder zur Befriedigung zentraler Lebensbedürfnisse“ geht, desto enger sei der Spielraum für den Unternehmer beim Ausschluss „unliebsamer“ Gäste. Was für ein Tagungshotel gilt, muss also nicht fürs Kiezcafé zulässig sein. Letztendlich läuft es im Einzelfall, wie so oft, auf eine Abwägung der Interessen des Unternehmers und der Kunden hinaus. Das heißt, im konkreten Fall hat ein örtliches Gericht das letzte Wort.  \u003c/p\u003e\n\u003cp\u003eImmerhin verdeutlicht das Urteil aber sehr klar, dass ein entsprechendes Geschäftskonzept gewisse Personengruppen ausschließen kann und keine unzulässige Diskrimnierung damit verbunden ist – sofern die Regel nur sachlich begründbar ist. Ein Urteil auf jeden Fall, das die unternehmerische Freiheit ziemlich deutlich in den Mittelpunkt stellt. Der Shitstorm für das Gericht dürfte damit absehbar sein. \u003c/p\u003e\n\u003cp\u003e\u003ca href=\"http://juris.bundesgerichtshof.de/cgi-bin/rechtsprechung/document.py?Gericht=bgh\u0026amp;Art=en\u0026amp;az=VIII%20ZR%20401/18\u0026amp;nr=107827\"\u003eLinks zum Urteil \u003c/a\u003e\u003c/p\u003e\n","url":"https://www.lawblog.de/index.php/archives/2020/07/10/kinder-u-a-muessen-draussen-bleiben/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/b12af43a481a69a69d89bb998c991d681f0d2a2d?base64_url=aHR0cHM6Ly93d3cubGF3YmxvZy5kZS9pbmRleC5waHAvYXJjaGl2ZXMvMjAyMC8wNy8xMC9raW5kZXItdS1hLW11ZXNzZW4tZHJhdXNzZW4tYmxlaWJlbi8=","published":"2020-07-10T10:37:49.000000Z","created_at":"2020-07-10T10:57:43.206305Z"},{"id":2509802446,"feed_id":125584,"title":"Freitagsgespräche","author":"Udo Vetter","summary":"„Ach, gut dass Sie anrufen. Wann kann ich denn mit dem Kostenvorschuss rechnen?“ „Haben Sie das Geld noch nicht bekommen? Habe ich Donnerstag vor einer Woche gesendet.“ „Ich habe bislang nichts auf dem Konto.“ „Ich habe das Geld per Post geschickt.“ Seufz.","content":"\u003cp\u003e„Ach, gut dass Sie anrufen. Wann kann ich denn mit dem Kostenvorschuss rechnen?“\u003c/p\u003e\n\u003cp\u003e„Haben Sie das Geld noch nicht bekommen? Habe ich Donnerstag vor einer Woche gesendet.“\u003c/p\u003e\n\u003cp\u003e„Ich habe bislang nichts auf dem Konto.“\u003c/p\u003e\n\u003cp\u003e„Ich habe das Geld per Post geschickt.“\u003c/p\u003e\n\u003cp\u003eSeufz. \u003c/p\u003e\n","url":"https://www.lawblog.de/index.php/archives/2020/07/10/freitagsgespraeche/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/1d131aeefe238c5dbec3453229faad3122e1783a?base64_url=aHR0cHM6Ly93d3cubGF3YmxvZy5kZS9pbmRleC5waHAvYXJjaGl2ZXMvMjAyMC8wNy8xMC9mcmVpdGFnc2dlc3ByYWVjaGUv","published":"2020-07-10T14:00:04.000000Z","created_at":"2020-07-10T14:06:07.044777Z"},{"id":2510324708,"feed_id":184804,"title":"Hamster Ball 2","author":null,"summary":"","content":"\u003cimg src=\"https://imgs.xkcd.com/comics/hamster_ball_2.png\" title=\"The worst is being teased by responsible, mask-wearing teens. They even spritz the hamster ball with disinfectant before rolling it, carefully avoiding the filter vents.\" alt=\"The worst is being teased by responsible, mask-wearing teens. They even spritz the hamster ball with disinfectant before rolling it, carefully avoiding the filter vents.\"\u003e","url":"https://xkcd.com/2331/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/40a19eb2575ba3a03c00cdd7a030c6b4bfb54906?base64_url=aHR0cHM6Ly94a2NkLmNvbS8yMzMxLw==","published":"2020-07-10T00:00:00.000000Z","created_at":"2020-07-10T22:03:58.742537Z"},{"id":2511847291,"feed_id":1397625,"title":"Warts","author":null,"summary":"","content":"\u003cdiv style=\"display:block; background-color: #ccc;\"\u003e\n\u003cp\u003e\u003ca href=\"https://www.oglaf.com/warts/\"\u003e\u003cimg src=\"https://media.oglaf.com/story/ttwarts.gif\"\u003e\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003e\u003ca href=\"https://www.oglaf.com/warts/\"\u003e\u003cimg src=\"https://media.oglaf.com/archive/arc-warts.png\" width=\"400\" height=\"100\" border=\"0\" alt=\"https://www.oglaf.com/warts/\"\u003e\u003c/a\u003e\u003c/p\u003e\n\u003c/div\u003e\n","url":"https://www.oglaf.com/warts/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/cb0fec7785af37d2bd355143980c817f9f13f4c6?base64_url=aHR0cHM6Ly93d3cub2dsYWYuY29tL3dhcnRzLw==","published":"2020-07-12T00:00:00.000000Z","created_at":"2020-07-12T15:18:53.258091Z"},{"id":2513692557,"feed_id":1520516,"title":"Open Folders in Any Terminal App from Nautilus","author":"Joey Sneddon","summary":"Being able to open a folder in a terminal from Nautilus is handy — but what if you want to use a different terminal app? Use this plucky Nautilus plugin. This post, Open Folders in Any Terminal App from Nautilus is from OMG! Ubuntu!. Do not reproduce","content":"\u003cp\u003e\u003cimg width=\"406\" height=\"232\" src=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/04/best-ubuntu-apps-tilix-406x232.jpg\" class=\"attachment-post-list size-post-list wp-post-image\" alt=\"\" srcset=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/04/best-ubuntu-apps-tilix-406x232.jpg 406w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/04/best-ubuntu-apps-tilix-350x200.jpg 350w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/04/best-ubuntu-apps-tilix-840x480.jpg 840w \" sizes=\"(max-width: 406px) 100vw, 406px\"\u003eBeing able to open a folder in a terminal from Nautilus is handy — but what if you want to use a different terminal app? Use this plucky Nautilus plugin.\u003c/p\u003e\n\u003cp\u003eThis post, \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk/2020/07/open-folder-in-terminal-ubuntu-plugin\"\u003eOpen Folders in Any Terminal App from Nautilus\u003c/a\u003e is from \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk\"\u003eOMG! Ubuntu!\u003c/a\u003e. Do not reproduce elsewhere without permission.\u003c/p\u003e\n","url":"https://www.omgubuntu.co.uk/2020/07/open-folder-in-terminal-ubuntu-plugin","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/dc347ea04e40500bad31b682a98cad2e30323491?base64_url=aHR0cHM6Ly93d3cub21ndWJ1bnR1LmNvLnVrLzIwMjAvMDcvb3Blbi1mb2xkZXItaW4tdGVybWluYWwtdWJ1bnR1LXBsdWdpbg==","published":"2020-07-14T05:34:00.000000Z","created_at":"2020-07-14T05:39:16.332659Z"},{"id":2514374566,"feed_id":1520516,"title":"Star Labs’ Newest Linux Laptop is Small But Perfectly Formed","author":"Joey Sneddon","summary":"The Star Labs Lite Mk III is a small, full-formed Linux laptop perfect for on-the-go computing. We share the laptop's specs, price, and where to buy. This post, Star Labs’ Newest Linux Laptop is Small But Perfectly Formed is from OMG! Ubuntu!. Do not","content":"\u003cp\u003e\u003cimg width=\"406\" height=\"232\" src=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop-406x232.jpg\" class=\"attachment-post-list size-post-list wp-post-image\" alt=\"\" srcset=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop-406x232.jpg 406w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop-300x172.jpg 300w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop-750x429.jpg 750w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop-350x200.jpg 350w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop-768x440.jpg 768w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop-840x480.jpg 840w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/star-labs-lite-laptop.jpg 950w \" sizes=\"(max-width: 406px) 100vw, 406px\"\u003eThe Star Labs Lite Mk III is a small, full-formed Linux laptop perfect for on-the-go computing. We share the laptop's specs, price, and where to buy.\u003c/p\u003e\n\u003cp\u003eThis post, \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk/2020/07/starlabs-lite-mk-iii-linux-laptop\"\u003eStar Labs’ Newest Linux Laptop is Small But Perfectly Formed\u003c/a\u003e is from \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk\"\u003eOMG! Ubuntu!\u003c/a\u003e. Do not reproduce elsewhere without permission.\u003c/p\u003e\n","url":"https://www.omgubuntu.co.uk/2020/07/starlabs-lite-mk-iii-linux-laptop","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/e29eec0a4ef1e6918dd84c21bcae8b7435e7d71f?base64_url=aHR0cHM6Ly93d3cub21ndWJ1bnR1LmNvLnVrLzIwMjAvMDcvc3RhcmxhYnMtbGl0ZS1tay1paWktbGludXgtbGFwdG9w","published":"2020-07-14T16:48:52.000000Z","created_at":"2020-07-14T17:02:32.578959Z"},{"id":2515484275,"feed_id":125584,"title":"Ein Foto","author":"Udo Vetter","summary":"Der Mandant schickt via WhatsApp ein Foto vom Anhörungsbogen, den er von der Polizei bekommen hat. Ihm wird Fahren ohne Fahrerlaubnis vorgeworfen, womöglich eins seiner wenigen Laster.","content":"\u003cp\u003eDer Mandant schickt via WhatsApp ein Foto vom Anhörungsbogen, den er von der Polizei bekommen hat.\u003c/p\u003e\n\u003cp\u003e\u003ca href=\"https://www.lawblog.de/index.php/archives/2020/07/15/__trashed-9/amg/\" rel=\"attachment wp-att-41611\"\u003e\u003cimg src=\"https://www.lawblog.de/wp-content/uploads/2020/07/AMG.jpg\" alt=\"\" width=\"375\" height=\"271\" class=\"aligncenter size-full wp-image-41611\" srcset=\"https://www.lawblog.de/wp-content/uploads/2020/07/AMG.jpg 375w, https://www.lawblog.de/wp-content/uploads/2020/07/AMG-300x217.jpg 300w\" sizes=\"(max-width: 375px) 100vw, 375px\"\u003e\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eIhm wird Fahren ohne Fahrerlaubnis vorgeworfen, womöglich eins seiner wenigen Laster. \u003c/p\u003e\n","url":"https://www.lawblog.de/index.php/archives/2020/07/15/ein-foto/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/ef995dfeaa9e62d64806a633ae53de9f3c9e4b23?base64_url=aHR0cHM6Ly93d3cubGF3YmxvZy5kZS9pbmRleC5waHAvYXJjaGl2ZXMvMjAyMC8wNy8xNS9laW4tZm90by8=","published":"2020-07-15T13:38:31.000000Z","created_at":"2020-07-15T13:53:46.887042Z"},{"id":2515543269,"feed_id":1520516,"title":"Android 10 is Coming to the PinePhone (Updated)","author":"Joey Sneddon","summary":"GloDroid is a project that wants to bring a fully open source version of Android 10 to low-cost devices including the Orange Pi, PinePhone and PineTab. This post, Android 10 is Coming to the PinePhone (Updated) is from OMG! Ubuntu!. Do not reproduce","content":"\u003cp\u003e\u003cimg width=\"406\" height=\"232\" src=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-406x232.jpg\" class=\"attachment-post-list size-post-list wp-post-image\" alt=\"Android on the PinePhone\" srcset=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-406x232.jpg 406w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-300x172.jpg 300w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-750x430.jpg 750w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-350x200.jpg 350w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-768x440.jpg 768w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-1536x880.jpg 1536w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone-840x480.jpg 840w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/android-on-the-pinephone.jpg 1682w \" sizes=\"(max-width: 406px) 100vw, 406px\"\u003eGloDroid is a project that wants to bring a fully open source version of Android 10 to low-cost devices including the Orange Pi, PinePhone and PineTab.\u003c/p\u003e\n\u003cp\u003eThis post, \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk/2020/07/glodroid-android-for-pinephone-allwinner\"\u003eAndroid 10 is Coming to the PinePhone (Updated)\u003c/a\u003e is from \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk\"\u003eOMG! Ubuntu!\u003c/a\u003e. Do not reproduce elsewhere without permission.\u003c/p\u003e\n","url":"https://www.omgubuntu.co.uk/2020/07/glodroid-android-for-pinephone-allwinner","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/3715912ce04cf0b8b3c1f06e6ee0cc306eb43d0c?base64_url=aHR0cHM6Ly93d3cub21ndWJ1bnR1LmNvLnVrLzIwMjAvMDcvZ2xvZHJvaWQtYW5kcm9pZC1mb3ItcGluZXBob25lLWFsbHdpbm5lcg==","published":"2020-07-15T14:34:27.000000Z","created_at":"2020-07-15T14:43:54.361750Z"},{"id":2515556412,"feed_id":197252,"title":"pics/corona/idioten-vermehrung.jpg","author":null,"summary":"","content":"\u003cimg src=\"http://fun.drno.de/pics/corona/idioten-vermehrung.jpg\"\u003e","url":"http://fun.drno.de/pics/corona/idioten-vermehrung.jpg","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/fa429b7884bc9e24d6eb2e26960c50482600bde9?base64_url=aHR0cDovL2Z1bi5kcm5vLmRlL3BpY3MvY29yb25hL2lkaW90ZW4tdmVybWVocnVuZy5qcGc=","published":"2020-07-15T14:52:58.946836Z","created_at":"2020-07-15T14:52:58.946810Z"},{"id":2515648260,"feed_id":1520516,"title":"New PinePhone with 3GB RAM and USB Dock Goes on Sale","author":"Joey Sneddon","summary":"The PinePhone postmarketOS Community Edition is now available to order direct from Pine64 priced at $149 — but I don’t think you should buy it. Hear me out, hear me out — there is a […] This post, New PinePhone with 3GB RAM and USB Dock Goes on Sale is","content":"\u003cp\u003e\u003cimg width=\"406\" height=\"232\" src=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/06/postmarket-os-community-edition-406x232.jpg\" class=\"attachment-post-list size-post-list wp-post-image\" alt=\"pinephone postmarketOS community edition\" srcset=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/06/postmarket-os-community-edition-406x232.jpg 406w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/06/postmarket-os-community-edition-300x171.jpg 300w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/06/postmarket-os-community-edition-750x429.jpg 750w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/06/postmarket-os-community-edition-350x200.jpg 350w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/06/postmarket-os-community-edition-768x439.jpg 768w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/06/postmarket-os-community-edition.jpg 840w \" sizes=\"(max-width: 406px) 100vw, 406px\"\u003eThe PinePhone postmarketOS Community Edition is now available to order direct from Pine64 priced at $149 — but I don’t think you should buy it. Hear me out, hear me out — there is a […]\u003c/p\u003e\n\u003cp\u003eThis post, \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk/2020/07/buy-pinephone-postmarketos\"\u003eNew PinePhone with 3GB RAM and USB Dock Goes on Sale\u003c/a\u003e is from \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk\"\u003eOMG! Ubuntu!\u003c/a\u003e. Do not reproduce elsewhere without permission.\u003c/p\u003e\n","url":"https://www.omgubuntu.co.uk/2020/07/buy-pinephone-postmarketos","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/a9148882378c794174fd03de2d607c510cfe115e?base64_url=aHR0cHM6Ly93d3cub21ndWJ1bnR1LmNvLnVrLzIwMjAvMDcvYnV5LXBpbmVwaG9uZS1wb3N0bWFya2V0b3M=","published":"2020-07-15T15:51:03.000000Z","created_at":"2020-07-15T16:00:02.292400Z"},{"id":2515683671,"feed_id":184804,"title":"COVID Risk Chart","author":null,"summary":"","content":"\u003cimg src=\"https://imgs.xkcd.com/comics/covid_risk_chart.png\" title=\"First prize is a free ticket to the kissing booth.\" alt=\"First prize is a free ticket to the kissing booth.\"\u003e","url":"https://xkcd.com/2333/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/d51391bfa9e2e47d03919105545142139ec1725d?base64_url=aHR0cHM6Ly94a2NkLmNvbS8yMzMzLw==","published":"2020-07-15T00:00:00.000000Z","created_at":"2020-07-15T16:31:43.235982Z"},{"id":2516081020,"feed_id":1520516,"title":"GNOME Opens an Official Merch Shop Selling T-Shirts, More","author":"Joey Sneddon","summary":"GNOME fans can now buy t-shirts, hoodies, and even socks bearing the famous logo through a new, dedicated online store. We take a look at what's on offer… This post, GNOME Opens an Official Merch Shop Selling T-Shirts, More is from OMG! Ubuntu!. Do not","content":"\u003cp\u003e\u003cimg width=\"406\" height=\"232\" src=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-406x232.jpg\" class=\"attachment-post-list size-post-list wp-post-image\" alt=\"gnome merchandise\" srcset=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-406x232.jpg 406w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-300x172.jpg 300w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-750x430.jpg 750w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-350x200.jpg 350w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-768x440.jpg 768w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-1536x880.jpg 1536w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch-840x480.jpg 840w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-merch.jpg 1682w \" sizes=\"(max-width: 406px) 100vw, 406px\"\u003eGNOME fans can now buy t-shirts, hoodies, and even socks bearing the famous logo through a new, dedicated online store. We take a look at what's on offer…\u003c/p\u003e\n\u003cp\u003eThis post, \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk/2020/07/gnome-merchandise-shop\"\u003eGNOME Opens an Official Merch Shop Selling T-Shirts, More\u003c/a\u003e is from \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk\"\u003eOMG! Ubuntu!\u003c/a\u003e. Do not reproduce elsewhere without permission.\u003c/p\u003e\n","url":"https://www.omgubuntu.co.uk/2020/07/gnome-merchandise-shop","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/8f6572bd1b73ff046cb062f1079d4f76fe861582?base64_url=aHR0cHM6Ly93d3cub21ndWJ1bnR1LmNvLnVrLzIwMjAvMDcvZ25vbWUtbWVyY2hhbmRpc2Utc2hvcA==","published":"2020-07-15T22:03:15.000000Z","created_at":"2020-07-15T22:06:38.612262Z"},{"id":2516258851,"feed_id":904489,"title":"Gary Davidian and His 68000 Emulator ↦","author":"Jason Snell","summary":"A timely little story by Hansen Hsu from the Computer History Museum about a previous Apple chip transition: Apple did this the very first time in the early 1990s, with the move from Motorola 68000 (a.k.a. 68K) to PowerPC. Motorola’s 68K chips were losing","content":"\u003cp\u003eA timely little story by Hansen Hsu from the Computer History Museum about \u003ca href=\"https://computerhistory.org/blog/transplanting-the-macs-central-processor-gary-davidian-and-his-68000-emulator/\"\u003ea previous Apple chip transition\u003c/a\u003e:\u003c/p\u003e\n\u003cblockquote\u003e\n\u003cp\u003e\n  Apple did this the very first time in the early 1990s, with the move from Motorola 68000 (a.k.a. 68K) to PowerPC. Motorola’s 68K chips were losing steam, and in order to keep up with IBM PC compatibles running Intel processors, Apple got together with Motorola and IBM to define a new processor architecture, the PowerPC. In order to make this move while still supporting all of the Mac’s existing software, including its operating system, Apple needed to create an emulator for 68K instructors that would run on the PowerPC. This was a multi-year project, which was started by engineer Gary Davidian at Apple in the middle of 1990….\u003c/p\u003e\n\u003cp\u003e  Davidian’s emulator served as a proof of concept that emulation of 68K code could be done without too big of a hit in software speed. This convinced managers at Apple that emulation was a workable solution for users…. In October 1991, Apple announced the PowerPC alliance with IBM and Motorola, and work began to shift Davidian’s emulator over to the PowerPC architecture. New prototype plug-in boards and then full machines were built with the new chip, whose bugs were still being worked out. It took several more years, until 1994, for the first Macs with PowerPC processors, the Power Macintosh \u003ca href=\"https://www.computerhistory.org/collections/catalog/102647305\"\u003e6100\u003c/a\u003e, \u003ca href=\"https://www.computerhistory.org/collections/catalog/102647308\"\u003e7100\u003c/a\u003e, and \u003ca href=\"https://www.computerhistory.org/collections/catalog/102661920\"\u003e8100\u003c/a\u003e, to be released to customers. The PowerPC would be the Mac’s CPU until 2006, when Apple replaced it with Intel’s processors.\n\u003c/p\u003e\n\u003c/blockquote\u003e\n\u003cp\u003eThe CHM article links to a couple of its oral-history interviews with Davidian. Nobody thinks about the PowerPC transition much now, but it set the bar for Mac chip transitions. Rosetta 2, coming this fall to Apple silicon Macs near you, is the inheritor of Davidian’s legacy.\u003c/p\u003e\u003cp\u003e\u003ca href=\"https://computerhistory.org/blog/transplanting-the-macs-central-processor-gary-davidian-and-his-68000-emulator/\"\u003eGo to the linked site\u003c/a\u003e.\u003c/p\u003e\u003cp\u003e\u003ca href=\"https://sixcolors.com/link/2020/07/gary-davidian-and-his-68000-emulator/\"\u003eRead on Six Colors\u003c/a\u003e.\u003c/p\u003e","url":"https://sixcolors.com/link/2020/07/gary-davidian-and-his-68000-emulator/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/b2fdcc63881750ec41019284279fcbe64bb3c8b3?base64_url=aHR0cHM6Ly9zaXhjb2xvcnMuY29tL2xpbmsvMjAyMC8wNy9nYXJ5LWRhdmlkaWFuLWFuZC1oaXMtNjgwMDAtZW11bGF0b3Iv","published":"2020-07-16T01:07:37.000000Z","created_at":"2020-07-16T01:36:39.385951Z"},{"id":2516494009,"feed_id":1214779,"title":"PSA und FCA fusionieren zu Stellantis","author":null,"summary":"Der gemeinsame Name steht, ein Logo soll bald folgen. Doch noch ist der Zusammenschluss von PSA und Fiat Chrysler nicht genehmigt.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/news/PSA-und-FCA-fusionieren-zu-Stellantis-4845091.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/7/1/7/5/MEDIA-690a4485076af183.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eDer gemeinsame Name steht, ein Logo soll bald folgen. Doch noch ist der Zusammenschluss von PSA und Fiat Chrysler nicht genehmigt.\u003c/p\u003e","url":"https://www.heise.de/news/PSA-und-FCA-fusionieren-zu-Stellantis-4845091.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/9019859d2335ed8f9e01b450e76acc08e7bc45a2?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvbmV3cy9QU0EtdW5kLUZDQS1mdXNpb25pZXJlbi16dS1TdGVsbGFudGlzLTQ4NDUwOTEuaHRtbD93dF9tYz1yc3MucmVkLmF1dG9zLmF1dG9zLmF0b20uYmVpdHJhZy5iZWl0cmFn","published":"2020-07-16T07:00:00.000000Z","created_at":"2020-07-16T07:15:02.881907Z"},{"id":2516561278,"feed_id":1214779,"title":"Land Rover: Diesel-Mildhybrid-Antriebe für den Range Rover Sport","author":null,"summary":"Statt der V6- und V8-Diesel bekommt der Range Rover Sport einen Reihensechser in drei Leistungsvarianten, der mit 48-Volt-Startergenerator zum Mildhybrid wird.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/hintergrund/Land-Rover-Diesel-Mildhybrid-Antriebe-fuer-den-Range-Rover-Sport-4844761.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/6/9/8/1/RRS_21MY_150720_12-0cd439ef8f478202.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eStatt der V6- und V8-Diesel bekommt der Range Rover Sport einen Reihensechser in drei Leistungsvarianten, der mit 48-Volt-Startergenerator zum Mildhybrid wird. \u003c/p\u003e","url":"https://www.heise.de/hintergrund/Land-Rover-Diesel-Mildhybrid-Antriebe-fuer-den-Range-Rover-Sport-4844761.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/a1893a7bb78b718cd2d3fcf0994ab7474d047a78?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvaGludGVyZ3J1bmQvTGFuZC1Sb3Zlci1EaWVzZWwtTWlsZGh5YnJpZC1BbnRyaWViZS1mdWVyLWRlbi1SYW5nZS1Sb3Zlci1TcG9ydC00ODQ0NzYxLmh0bWw_d3RfbWM9cnNzLnJlZC5hdXRvcy5hdXRvcy5hdG9tLmJlaXRyYWcuYmVpdHJhZw==","published":"2020-07-16T08:31:00.000000Z","created_at":"2020-07-16T08:41:20.814464Z"},{"id":2516807495,"feed_id":29836,"title":"Bock auf eine Bundestagsrede? Die hier lohnt sich. Geht um die Bafin, Wirecard und Cum Ex.","author":null,"summary":null,"content":null,"url":"http://blog.fefe.de/?ts=a1ee8c0c","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/6a00a3583cb7d0c5eed3a9c93b2af22bc745c0b1?base64_url=aHR0cDovL2Jsb2cuZmVmZS5kZS8_dHM9YTFlZThjMGM=","published":"2020-07-16T12:55:11.228729Z","created_at":"2020-07-16T12:55:11.228712Z"},{"id":2516872896,"feed_id":1214779,"title":"Fisker plant E-SUV \"Ocean\" auf Volkswagens MEB-Basis","author":null,"summary":"Statt mit eigener Technik und Festkörperbatterien will Fisker sein SUV-Modell Ocean auf Basis des modularen Elektrobaukastens von Volkswagen bauen.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/news/Fisker-plant-E-SUV-Ocean-auf-Volkswagens-MEB-Basis-4845754.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/7/6/2/4/Fisker_Track-2330-FOR_WEB-E2-effaf7335811f532.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eStatt mit eigener Technik und Festkörperbatterien will Fisker sein SUV-Modell Ocean auf Basis des modularen Elektrobaukastens von Volkswagen bauen.\u003c/p\u003e","url":"https://www.heise.de/news/Fisker-plant-E-SUV-Ocean-auf-Volkswagens-MEB-Basis-4845754.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/2c6dd8de2d094ebe8433eb3a8dcee6de2cd3275a?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvbmV3cy9GaXNrZXItcGxhbnQtRS1TVVYtT2NlYW4tYXVmLVZvbGtzd2FnZW5zLU1FQi1CYXNpcy00ODQ1NzU0Lmh0bWw_d3RfbWM9cnNzLnJlZC5hdXRvcy5hdXRvcy5hdG9tLmJlaXRyYWcuYmVpdHJhZw==","published":"2020-07-16T13:41:00.000000Z","created_at":"2020-07-16T13:48:48.035172Z"},{"id":2516986969,"feed_id":1405087,"title":"Motherboard: Hackers convinced Twitter employee to help them hijack accounts","author":"Dave Mark","summary":"Joseph Cox, Motherboard: A Twitter insider was responsible for a wave of high profile account takeovers on Wednesday, according to leaked screenshots obtained by Motherboard and two sources who took over accounts. And: “We used a rep that literally done","content":"\u003cp\u003eJoseph Cox, Motherboard:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eA Twitter insider was responsible for a wave of high profile account takeovers on Wednesday, according to leaked screenshots obtained by Motherboard and two sources who took over accounts.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eAnd:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003e“We used a rep that literally done all the work for us,” one of the sources told Motherboard. The second source added they paid the Twitter insider. Motherboard granted the sources anonymity to speak candidly about a security incident. A Twitter spokesperson told Motherboard that the company is still investigating whether the employee hijacked the accounts themselves or gave hackers access to the tool.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eAnd:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eAfter a wave of account takeovers, screenshots of an internal Twitter user administration tool are being shared in the hacking underground.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eAnd this response from Twitter:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eAfter the publication of this piece, Twitter said in a tweet that “We detected what we believe to be a coordinated social engineering attack by people who successfully targeted some of our employees with access to internal systems and tools.”\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eWere the employees duped by social engineering? Or was there complicity here, was a Twitter insider paid, as indicated by the article.\u003c/p\u003e\n\n\u003cp\u003eAlso, there is some question as to whether the bitcoin scam was the hackers’ endgame. Or if access to the accounts opened a door that could be exploited later.\u003c/p\u003e\n\n\u003cp\u003eBeyond alarming.\u003c/p\u003e\n\u003cp\u003e\u003ca href=\"https://www.loopinsight.com/2020/07/16/motherboard-hackers-convinced-twitter-employee-to-help-them-hijack-accounts/\" rel=\"bookmark\" title=\"Permanent link to 'Motherboard: Hackers convinced Twitter employee to help them hijack accounts'\" class=\"glyph\"\u003e∞ Read this on The Loop\u003c/a\u003e\u003c/p\u003e\n","url":"https://www.vice.com/en_us/article/jgxd3d/twitter-insider-access-panel-account-hacks-biden-uber-bezos","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/287519b53a093af1e06f9721f0ae88e506126f77?base64_url=aHR0cHM6Ly93d3cudmljZS5jb20vZW5fdXMvYXJ0aWNsZS9qZ3hkM2QvdHdpdHRlci1pbnNpZGVyLWFjY2Vzcy1wYW5lbC1hY2NvdW50LWhhY2tzLWJpZGVuLXViZXItYmV6b3M=","published":"2020-07-16T15:03:56.000000Z","created_at":"2020-07-16T15:14:54.805137Z"},{"id":2517128749,"feed_id":1405087,"title":"Camo uses your iPhone or iPad as a pro webcam for your Mac","author":"Shawn King","summary":"I’ve been a beta tester for Camo for a couple of weeks and, while it works mostly as advertised, $40/year to rent camera software seems steep for personal use considering simply plugging your iPhone into your Mac via USB and using the latest version of","content":"\u003cp\u003eI’ve been a beta tester for Camo for a couple of weeks and, while it works mostly as advertised, $40/year to rent camera software seems steep for personal use considering simply plugging your iPhone into your Mac via USB and using the latest version of QuickTime accomplishes the same thing for free.\u003c/p\u003e\n\u003cp\u003e\u003ca href=\"https://www.loopinsight.com/2020/07/16/camo-uses-your-iphone-or-ipad-as-a-pro-webcam-for-your-mac/\" rel=\"bookmark\" title=\"Permanent link to 'Camo uses your iPhone or iPad as a pro webcam for your Mac'\" class=\"glyph\"\u003e∞ Read this on The Loop\u003c/a\u003e\u003c/p\u003e\n","url":"https://appleinsider.com/articles/20/07/16/camo-uses-your-iphone-or-ipad-as-a-pro-webcam-for-your-mac","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/5a47f2722caf30f021d4c84f6d483079a19bead5?base64_url=aHR0cHM6Ly9hcHBsZWluc2lkZXIuY29tL2FydGljbGVzLzIwLzA3LzE2L2NhbW8tdXNlcy15b3VyLWlwaG9uZS1vci1pcGFkLWFzLWEtcHJvLXdlYmNhbS1mb3IteW91ci1tYWM=","published":"2020-07-16T17:04:18.000000Z","created_at":"2020-07-16T17:12:07.262803Z"},{"id":2517265404,"feed_id":1389029,"title":"Quest might just be the best starter RPG ever designed","author":"Matthew Rossi","summary":"It’s no secret that I’m old and jaded — I’ve been old and jaded for a long time now — so when I say that I’m impressed by Quest, it’s not just because I love internal rhymes. I get recommended a lot of indie RPGs. It’s just one of the things that happens","content":"\u003cdiv style=\"margin: 5px 5% 10px 5%;\"\u003e\u003cimg src=\"https://blizzardwatch.com/wp-content/uploads/2020/07/QuestRPG.png\" width=\"675\" height=\"378\" title=\"\" alt=\"\"\u003e\u003c/div\u003e\u003cdiv\u003eIt’s no secret that I’m old and jaded — I’ve been old and jaded for a long time now — so when I say that I’m impressed by Quest, it’s not just because I love internal rhymes. I get recommended a lot of indie RPGs. It’s just one of the things that happens when you’ve...\u003c/div\u003e","url":"https://blizzardwatch.com/2020/07/16/quest-tabletop-rpg/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/54b572267a58e940b4826deb7c3a6d835d35dc20?base64_url=aHR0cHM6Ly9ibGl6emFyZHdhdGNoLmNvbS8yMDIwLzA3LzE2L3F1ZXN0LXRhYmxldG9wLXJwZy8=","published":"2020-07-16T19:00:00.000000Z","created_at":"2020-07-16T19:07:56.289741Z"},{"id":2517387411,"feed_id":1389029,"title":"Overwatch Workshop addition lets you make heroes massive and/or tiny","author":"Tyler Colp","summary":"What if Widowmaker, but big? What if Sombra, but tiny? In the latest Overwatch PTR update, you can turn any hero into a Godzilla-sized monster or an ant-sized critter with the new Workshop tools. One of the major features implemented into the July 15 PTR","content":"\u003cdiv style=\"margin: 5px 5% 10px 5%;\"\u003e\u003cimg src=\"https://blizzardwatch.com/wp-content/uploads/2020/07/Overwatch-hero-scale-widowmaker-ptr.jpg\" width=\"675\" height=\"380\" title=\"\" alt=\"\"\u003e\u003c/div\u003e\u003cdiv\u003eWhat if Widowmaker, but big? What if Sombra, but tiny? In the latest Overwatch PTR update, you can turn any hero into a Godzilla-sized monster or an ant-sized critter with the new Workshop tools. One of the major features implemented into the July 15 PTR patch is the ability for you to scale hero size...\u003c/div\u003e","url":"https://blizzardwatch.com/2020/07/16/overwatch-workshop-hero-size/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/d94d151803f48ec5e5950737772158555ba32e9e?base64_url=aHR0cHM6Ly9ibGl6emFyZHdhdGNoLmNvbS8yMDIwLzA3LzE2L292ZXJ3YXRjaC13b3Jrc2hvcC1oZXJvLXNpemUv","published":"2020-07-16T21:00:01.000000Z","created_at":"2020-07-16T21:11:46.554574Z"},{"id":2517473764,"feed_id":29836,"title":"Die \"Frauen sind besser\"-Ideologen sind so erfolgreich mit ihrer Propaganda, dass in einem Jugendknast ein verhaltensauffälliger Minderjähriger Östrogen verabreicht gekriegt hat. Daraufhin wuchsen ihm Brüste.Bin ja mal gespannt, wann es die ersten Zwangskastrationen von Straftätern gibt. Die können sich nicht wehren, \"haben es verdient\" und haben keine schlagkräftige Lobby, die sie verteidigen würde.Hey, da könnte man doch auch veganes Essen zwangsverabreichen! Weiß doch jeder, dass Fleisch aggressiv macht!1!!Wo wir gerade bei freidrehenden Ideologen sind: Die Proponenten der Cancel Culture sind bei \"collateral damage builds character:Sind in den USA Menschen zu Unrecht entlassen worden, weil sie gegen \"linke Sprachregelungen\" verstoßen haben? Vielleicht. Das muss man mit den einzelnen Unternehmen ausmachen. Es gibt keinen Kampf für Gerechtigkeit, der jeden Augenblick gerecht ausgefochten wird.Ja aber echt mal! Die Unternehmen sind Schuld! Was sind die auch so blöde und lassen sich von Twitter-Lynchmobs mit Boykottaufrufen einschüchtern? Wenn denen Free Speech wirklich was wert wäre, dann hätten sie sich hinter die Lynchmob-Opfer unter ihren Angestellten gestellt anstatt sie vor den einfahrenden Hassmob-Zug zu schubsen!1!!","author":null,"summary":null,"content":null,"url":"http://blog.fefe.de/?ts=a1ee15c6","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/a45c543a02621921bc944d9b62fa26b2a415ff70?base64_url=aHR0cDovL2Jsb2cuZmVmZS5kZS8_dHM9YTFlZTE1YzY=","published":"2020-07-16T22:36:34.562438Z","created_at":"2020-07-16T22:36:34.562416Z"},{"id":2517484415,"feed_id":1389029,"title":"Meet High Priestess Maiev and Treasure Goblin Cho’Gall with new Heroes of the Storm skins","author":"Cory Taylor","summary":"Fate has twisted once again in Heroes of the Storm and we see what the world would look like if Maiev had become a High Priestess of Elune — and Cho’gall came from the Burning Hells.","content":"\u003cdiv style=\"margin: 5px 5% 10px 5%;\"\u003e\u003cimg src=\"https://blizzardwatch.com/wp-content/uploads/2020/07/Priestess.Maiev_.Altered.fate_.heroes.of_.the_.storm_.png\" width=\"675\" height=\"380\" title=\"\" alt=\"\"\u003e\u003c/div\u003e\u003cdiv\u003eFate has twisted once again in Heroes of the Storm and we see what the world would look like if Maiev had become a High Priestess of Elune — and Cho’gall came from the Burning Hells.\u003c/div\u003e","url":"https://blizzardwatch.com/gallery/high-priestess-maiev-chogall-greed-demon-skins-heroes-storm/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/fda5f5566877e7a919c14bbe250e69075635521b?base64_url=aHR0cHM6Ly9ibGl6emFyZHdhdGNoLmNvbS9nYWxsZXJ5L2hpZ2gtcHJpZXN0ZXNzLW1haWV2LWNob2dhbGwtZ3JlZWQtZGVtb24tc2tpbnMtaGVyb2VzLXN0b3JtLw==","published":"2020-07-16T22:54:01.000000Z","created_at":"2020-07-16T22:55:44.837297Z"},{"id":2517514674,"feed_id":1389029,"title":"Blizzard Watch Plays: Deadlands Noir TTRPG is the perfect setting for a classic New Orleans detective story","author":"Joe Perez","summary":"This week our tabletop gaming crew took a break from D\u0026D to try something completely new…. Deadlands Noir! The group of hard-boiled detectives took the the mean streets of New Orleans for a murder mystery set in the 1930’s. The game world is set in the","content":"\u003cdiv style=\"margin: 5px 5% 10px 5%;\"\u003e\u003cimg src=\"https://blizzardwatch.com/wp-content/uploads/2020/07/0e152d5e0b47866a60114112f5626ddd_original.jpg\" width=\"675\" height=\"380\" title=\"\" alt=\"Deadlands Noir\"\u003e\u003c/div\u003e\u003cdiv\u003eThis week our tabletop gaming crew took a break from D\u0026amp;D to try something completely new…. Deadlands Noir! The group of hard-boiled detectives took the the mean streets of New Orleans for a murder mystery set in the 1930’s. The game world is set in the same universe as Deadlands (which is often thought as the real...\u003c/div\u003e","url":"https://blizzardwatch.com/2020/07/16/deadlands-noir-ttrpg-play/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/eae99f60efad9e566daf388ff94994529ba0a98a?base64_url=aHR0cHM6Ly9ibGl6emFyZHdhdGNoLmNvbS8yMDIwLzA3LzE2L2RlYWRsYW5kcy1ub2lyLXR0cnBnLXBsYXkv","published":"2020-07-16T23:25:37.000000Z","created_at":"2020-07-16T23:30:22.669620Z"},{"id":2517538436,"feed_id":1405087,"title":"∞ Apple updates coding terminology removing non-inclusive language","author":"Jim Dalrymple","summary":"Apple on Thursday released a note on its developer website that the company is updating its coding terminology. The changes will be across the company’s ecosystem. […]","content":"\u003cp\u003eApple on Thursday released a note on its developer website that the company is updating its coding terminology. The changes will be across the company’s ecosystem. […]\u003c/p\u003e\n","url":"https://www.loopinsight.com/2020/07/16/apple-updates-coding-terminology-removing-non-inclusive-language/?utm_source=rss\u0026utm_medium=rss\u0026utm_campaign=apple-updates-coding-terminology-removing-non-inclusive-language","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/0f1d2d30ccfa9860739befc180194ebd122d9c40?base64_url=aHR0cHM6Ly93d3cubG9vcGluc2lnaHQuY29tLzIwMjAvMDcvMTYvYXBwbGUtdXBkYXRlcy1jb2RpbmctdGVybWlub2xvZ3ktcmVtb3Zpbmctbm9uLWluY2x1c2l2ZS1sYW5ndWFnZS8_dXRtX3NvdXJjZT1yc3MmdXRtX21lZGl1bT1yc3MmdXRtX2NhbXBhaWduPWFwcGxlLXVwZGF0ZXMtY29kaW5nLXRlcm1pbm9sb2d5LXJlbW92aW5nLW5vbi1pbmNsdXNpdmUtbGFuZ3VhZ2U=","published":"2020-07-16T23:54:47.000000Z","created_at":"2020-07-17T00:01:14.751553Z"},{"id":2517913856,"feed_id":1520516,"title":"Get GNOME 3.38’s New Default Wallpapers Right Now","author":"Joey Sneddon","summary":"Here's your first look at GNOME 3.38's new default wallpaper, plus a link to download it to use on your own desktop ahead of September's GNOME 3.38 release. This post, Get GNOME 3.38’s New Default Wallpapers Right Now is from OMG! Ubuntu!. Do not reproduce","content":"\u003cp\u003e\u003cimg width=\"406\" height=\"232\" src=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-406x232.jpg\" class=\"attachment-post-list size-post-list wp-post-image\" alt=\"gnome shell 3.38\" srcset=\"https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-406x232.jpg 406w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-300x172.jpg 300w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-750x430.jpg 750w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-350x200.jpg 350w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-768x440.jpg 768w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-1536x880.jpg 1536w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38-840x480.jpg 840w , https://149366088.v2.pressablecdn.com/wp-content/uploads/2020/07/gnome-3.38.jpg 1682w \" sizes=\"(max-width: 406px) 100vw, 406px\"\u003eHere's your first look at GNOME 3.38's new default wallpaper, plus a link to download it to use on your own desktop ahead of September's GNOME 3.38 release.\u003c/p\u003e\n\u003cp\u003eThis post, \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk/2020/07/gnome-3-38-wallpaper-adwaita\"\u003eGet GNOME 3.38’s New Default Wallpapers Right Now\u003c/a\u003e is from \u003ca rel=\"nofollow\" href=\"https://www.omgubuntu.co.uk\"\u003eOMG! Ubuntu!\u003c/a\u003e. Do not reproduce elsewhere without permission.\u003c/p\u003e\n","url":"https://www.omgubuntu.co.uk/2020/07/gnome-3-38-wallpaper-adwaita","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/38bd3122b39df6628e7d5763df398cf4da526336?base64_url=aHR0cHM6Ly93d3cub21ndWJ1bnR1LmNvLnVrLzIwMjAvMDcvZ25vbWUtMy0zOC13YWxscGFwZXItYWR3YWl0YQ==","published":"2020-07-17T08:25:00.000000Z","created_at":"2020-07-17T08:32:25.513150Z"},{"id":2517938093,"feed_id":1214779,"title":"Kraftstoffverbrauch von Neuwagen sinkt leicht","author":null,"summary":"Neue Autos verbrauchen im WLTP weniger Kraftstoff als im 1. Halbjahr 2019. Doch der Fortschritt ist klein; bei den Abgasnormen müssen fast alle bald nachbessern","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/hintergrund/Kraftstoffverbrauch-von-Neuwagen-sinkt-leicht-4846297.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/7/9/6/3/zapfsaeulen_nacht-4024bdc9fb508c77.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eNeue Autos verbrauchen im WLTP weniger Kraftstoff als im 1. Halbjahr 2019. Doch der Fortschritt ist klein; bei den Abgasnormen müssen fast alle bald nachbessern\u003c/p\u003e","url":"https://www.heise.de/hintergrund/Kraftstoffverbrauch-von-Neuwagen-sinkt-leicht-4846297.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/9fcacab7bbf1098d0a71469c5ec6ca11f275f749?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvaGludGVyZ3J1bmQvS3JhZnRzdG9mZnZlcmJyYXVjaC12b24tTmV1d2FnZW4tc2lua3QtbGVpY2h0LTQ4NDYyOTcuaHRtbD93dF9tYz1yc3MucmVkLmF1dG9zLmF1dG9zLmF0b20uYmVpdHJhZy5iZWl0cmFn","published":"2020-07-17T08:44:00.000000Z","created_at":"2020-07-17T09:04:14.567224Z"},{"id":2518185771,"feed_id":1405087,"title":"∞ Rene Ritchie: Wrong about the Apple Silicon Mac","author":"Dave Mark","summary":"[VIDEO] This is just a great watch, Rene Ritchie presents a master class on Apple’s history with Arm and what it means today. If you have any interest in the Mac’s path to Apple silicon, this is incredibly informative and not too jargony. Video embedded in","content":"\u003cp\u003e[VIDEO] This is just a great watch, Rene Ritchie presents a master class on Apple’s history with Arm and what it means today. If you have any interest in the Mac’s path to Apple silicon, this is incredibly informative and not too jargony. Video embedded in main Loop post.\u003c/p\u003e\n","url":"https://www.loopinsight.com/2020/07/17/rene-ritchie-wrong-about-the-apple-silicon-mac/?utm_source=rss\u0026utm_medium=rss\u0026utm_campaign=rene-ritchie-wrong-about-the-apple-silicon-mac","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/6c8040e685fed6f71c4767f13b5e621eead264be?base64_url=aHR0cHM6Ly93d3cubG9vcGluc2lnaHQuY29tLzIwMjAvMDcvMTcvcmVuZS1yaXRjaGllLXdyb25nLWFib3V0LXRoZS1hcHBsZS1zaWxpY29uLW1hYy8_dXRtX3NvdXJjZT1yc3MmdXRtX21lZGl1bT1yc3MmdXRtX2NhbXBhaWduPXJlbmUtcml0Y2hpZS13cm9uZy1hYm91dC10aGUtYXBwbGUtc2lsaWNvbi1tYWM=","published":"2020-07-17T12:52:56.000000Z","created_at":"2020-07-17T13:09:23.406598Z"},{"id":2518207328,"feed_id":1405087,"title":"∞ Clever music video made entirely made on a Mac screen recording","author":"Dave Mark","summary":"[VIDEO] The fun starts about 45 seconds in. Obviously, this is edited, not a real-time screen recording, but the conceit is really well done. Full screen. Video embedded in main Loop post.","content":"\u003cp\u003e[VIDEO] The fun starts about 45 seconds in. Obviously, this is edited, not a real-time screen recording, but the conceit is really well done. Full screen. Video embedded in main Loop post.\u003c/p\u003e\n","url":"https://www.loopinsight.com/2020/07/17/clever-music-video-made-entirely-made-on-a-mac-screen-recording/?utm_source=rss\u0026utm_medium=rss\u0026utm_campaign=clever-music-video-made-entirely-made-on-a-mac-screen-recording","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/464474063989d1487184ba3f4c38876d0d06c70a?base64_url=aHR0cHM6Ly93d3cubG9vcGluc2lnaHQuY29tLzIwMjAvMDcvMTcvY2xldmVyLW11c2ljLXZpZGVvLW1hZGUtZW50aXJlbHktbWFkZS1vbi1hLW1hYy1zY3JlZW4tcmVjb3JkaW5nLz91dG1fc291cmNlPXJzcyZ1dG1fbWVkaXVtPXJzcyZ1dG1fY2FtcGFpZ249Y2xldmVyLW11c2ljLXZpZGVvLW1hZGUtZW50aXJlbHktbWFkZS1vbi1hLW1hYy1zY3JlZW4tcmVjb3JkaW5n","published":"2020-07-17T13:18:37.000000Z","created_at":"2020-07-17T13:24:07.861362Z"},{"id":2518207336,"feed_id":1405087,"title":"Who’s behind Wednesday’s epic Twitter hack?","author":"Dave Mark","summary":"This starts with a retelling of the hack story, but that’s just the start. The real juice starts down below that. People within the SIM swapping community are obsessed with hijacking so-called “OG” social media accounts. Short for “original gangster,” OG","content":"\u003cp\u003eThis starts with a retelling of the hack story, but that’s just the start. The real juice starts down below that.\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003ePeople within the SIM swapping community are obsessed with hijacking so-called “OG” social media accounts. Short for “original gangster,” OG accounts typically are those with short profile names (such as @B or @joe). Possession of these OG accounts confers a measure of status and perceived influence and wealth in SIM swapping circles, as such accounts can often fetch thousands of dollars when resold in the underground.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eAnd:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eIn a post on OGusers — a forum dedicated to account hijacking — a user named “Chaewon” advertised they could change email address tied to any Twitter account for $250, and provide direct access to accounts for between $2,000 and $3,000 apiece.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eGreat Dalrymple’s Beard!!! That can’t be real, can it?\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eLucky225 said that just before 2 p.m. EDT on Wednesday, he received a password reset confirmation code via Google Voice for the @6 Twitter account. Lucky said he’d previously disabled SMS notifications as a means of receiving multi-factor codes from Twitter, opting instead to have one-time codes generated by a mobile authentication app.\u003c/p\u003e\n  \n  \u003cp\u003eBut because the attackers were able to change the email address tied to the @6 account and disable multi-factor authentication, the one-time authentication code was sent to both his Google Voice account and to the new email address added by the attackers.\u003c/p\u003e\n  \n  \u003cp\u003e“The way the attack worked was that within Twitter’s admin tools, apparently you can update the email address of any Twitter user, and it does this without sending any kind of notification to the user,” Lucky told KrebsOnSecurity. “So [the attackers] could avoid detection by updating the email address on the account first, and then turning off 2FA.”\u003c/p\u003e\n  \n  \u003cp\u003eLucky said he hasn’t been able to review whether any tweets were sent from his account during the time it was hijacked because he still doesn’t have access to it\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eHere’s a link to a \u003ca href=\"https://medium.com/@lucky225/the-twitter-hack-what-exactly-happened-d8740d33c1c\" rel=\"noopener noreferrer\" target=\"_blank\"\u003edetailed telling of this story\u003c/a\u003e.\u003c/p\u003e\n\n\u003cp\u003eRead the whole Krebs on Security post via the headline link. Fascinating and not a little scary. Amazing to me so little damage was done.\u003c/p\u003e\n\n\u003cp\u003eAs I’ve said before, not convinced that this was the end of this particular misadventure. Would not be surprised if this was just some misdirection to hide a more critical unlocking event that will rear its head in the future.\u003c/p\u003e\n\u003cp\u003e\u003ca href=\"https://www.loopinsight.com/2020/07/17/whos-behind-wednesdays-epic-twitter-hack/\" rel=\"bookmark\" title=\"Permanent link to 'Who’s behind Wednesday’s epic Twitter hack?'\" class=\"glyph\"\u003e∞ Read this on The Loop\u003c/a\u003e\u003c/p\u003e\n","url":"https://krebsonsecurity.com/2020/07/whos-behind-wednesdays-epic-twitter-hack/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/9e0352fed39389343714b87c8f31b1a075282490?base64_url=aHR0cHM6Ly9rcmVic29uc2VjdXJpdHkuY29tLzIwMjAvMDcvd2hvcy1iZWhpbmQtd2VkbmVzZGF5cy1lcGljLXR3aXR0ZXItaGFjay8=","published":"2020-07-17T13:11:47.000000Z","created_at":"2020-07-17T13:24:08.029677Z"},{"id":2518215588,"feed_id":1735,"title":"Deals: Amazon Discounts iPad Mini 5 by $50, Starting at $499 for 256GB Wi-Fi","author":"Mitchel Broussard","summary":"Amazon has discounted two models of Apple's iPad mini 5 today, starting with the 256GB Wi-Fi model at $499.00, down from $549.00. You can get this version of the iPad mini 5 in all three color options, including Gold, Space Gray, and Silver. Note:","content":"Amazon has discounted two models of Apple's iPad mini 5 today, starting with the 256GB Wi-Fi model at \u003ca href=\"https://www.amazon.com/dp/B07PSK4LGQ?tag=macrumors-20\u0026amp;linkCode=ogi\u0026amp;th=1\u0026amp;psc=1\"\u003e\u003cstrong\u003e$499.00\u003c/strong\u003e\u003c/a\u003e, down from $549.00. You can get this version of the iPad mini 5 in all three color options, including Gold, Space Gray, and Silver.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2019/03/ipad-mini-5-apple-pencil.jpg\" alt=\"\" width=\"1200\" height=\"630\" class=\"aligncenter size-full wp-image-684646\"\u003e\u003cem\u003e\u003csmall\u003eNote: MacRumors is an affiliate partner with Amazon. When you click a link and make a purchase, we may receive a small payment, which helps us keep the site running.\u003c/small\u003e\u003c/em\u003e\n\u003cbr\u003e\n\n\u003cbr\u003e\nSecond is the 256GB cellular iPad mini 5 at \u003ca href=\"https://www.amazon.com/dp/B07PQ8VVG8?tag=macrumors-20\u0026amp;linkCode=ogi\u0026amp;th=1\"\u003e\u003cstrong\u003e$629.00\u003c/strong\u003e\u003c/a\u003e, down from $679.00. This model is only available in Silver and Space Gray. Amazon has both models of the iPad mini 5 in stock and ready to ship, with an estimated delivery date early next week for orders in the United States.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ccenter\u003e\u003ca href=\"https://www.amazon.com/dp/B07PSK4LGQ?tag=macrumors-20\u0026amp;linkCode=ogi\u0026amp;th=1\u0026amp;psc=1\" target=\"_blank\"\u003e\u003cdiv class=\"fancybutton\"\u003e\n\u003cdiv class=\"fancybadge\"\u003e$50 OFF\u003c/div\u003eiPad mini 5 for $499.00\u003c/div\u003e\u003c/a\u003e\u003c/center\u003e\n\u003cbr\u003e\n\n\u003cbr\u003e\nAmazon doesn't have any 64GB versions of the iPad mini 5 on sale at this time, but we are still tracking a solid deal on the 64GB cellular iPad mini 5 at Tiger Direct. This model is priced at \u003ca href=\"https://www.anrdoezrs.net/links/462641/type/dlg/https://www.tigerdirect.com/applications/SearchTools/item-details.asp?EdpNo=6337584\u0026amp;CatId=11944\"\u003e\u003cstrong\u003e$449.00\u003c/strong\u003e\u003c/a\u003e, down from $529.00, and is a lowest-ever price for this version of the tablet.\n\u003cbr\u003e\n\n\u003cbr\u003e\nFor even more iPad deals, head to our full \u003ca href=\"https://www.macrumors.com/guide/ipad-deals/\"\u003eBest Deals guide for iPad\u003c/a\u003e. In that guide we track the best discounts online for iPad, iPad mini, iPad Air, and iPad Pro.\u003cdiv class=\"linkback\"\u003eRelated Roundup: \u003ca href=\"https://www.macrumors.com/roundup/best-apple-deals/\"\u003eApple Deals\u003c/a\u003e\n\u003c/div\u003e\u003cbr\u003eThis article, \"\u003ca href=\"https://www.macrumors.com/2020/07/17/deals-ipad-mini-5-by-50/\"\u003eDeals: Amazon Discounts iPad Mini 5 by $50, Starting at $499 for 256GB Wi-Fi\u003c/a\u003e\" first appeared on \u003ca href=\"https://www.macrumors.com\"\u003eMacRumors.com\u003c/a\u003e\u003cbr\u003e\u003cbr\u003e\u003ca href=\"https://forums.macrumors.com/threads/deals-amazon-discounts-ipad-mini-5-by-50-starting-at-499-for-256gb-wi-fi.2246275/\"\u003eDiscuss this article\u003c/a\u003e in our forums\u003cbr\u003e\u003cbr\u003e\u003cdiv class=\"feedflare\"\u003e\n\u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=3RaixHsdtdU:PVvsz9Uixos:6W8y8wAjSf4\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=6W8y8wAjSf4\" border=\"0\"\u003e\u003c/a\u003e \u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=3RaixHsdtdU:PVvsz9Uixos:qj6IDK7rITs\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=qj6IDK7rITs\" border=\"0\"\u003e\u003c/a\u003e\n\u003c/div\u003e\u003cimg src=\"http://feeds.feedburner.com/~r/MacRumors-All/~4/3RaixHsdtdU\" height=\"1\" width=\"1\" alt=\"\"\u003e","url":"https://www.macrumors.com/2020/07/17/deals-ipad-mini-5-by-50/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/0c70dbe9d59bb1dcd9bd8820530a592b07d9a907?base64_url=aHR0cHM6Ly93d3cubWFjcnVtb3JzLmNvbS8yMDIwLzA3LzE3L2RlYWxzLWlwYWQtbWluaS01LWJ5LTUwLw==","published":"2020-07-17T13:32:44.000000Z","created_at":"2020-07-17T13:35:19.647383Z"},{"id":2518257420,"feed_id":1214779,"title":"Renault Clio, Mégane und Captur mit ungewöhnlichem Hybridantrieb","author":null,"summary":"Drei Modelle rüstet Renault mit einem ungewöhnlich konzipierten Hybridantrieb auf. Der bietet einen 1,6-Liter-Benziner und zwei E-Motoren.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/hintergrund/Renault-Clio-Megane-und-Captur-mit-ungewoehnlichem-Hybridantrieb-4846821.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/8/2/6/4/10-7cf6799eeaf7410c.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eDrei Modelle rüstet Renault mit einem ungewöhnlich konzipierten Hybridantrieb auf. Der bietet einen 1,6-Liter-Benziner und zwei E-Motoren.\u003c/p\u003e","url":"https://www.heise.de/hintergrund/Renault-Clio-Megane-und-Captur-mit-ungewoehnlichem-Hybridantrieb-4846821.html?wt_mc=rss.red.autos.autos.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/d8d8c790f7d4fcd9c32ac2385a2259a6c4656c3d?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvaGludGVyZ3J1bmQvUmVuYXVsdC1DbGlvLU1lZ2FuZS11bmQtQ2FwdHVyLW1pdC11bmdld29laG5saWNoZW0tSHlicmlkYW50cmllYi00ODQ2ODIxLmh0bWw_d3RfbWM9cnNzLnJlZC5hdXRvcy5hdXRvcy5hdG9tLmJlaXRyYWcuYmVpdHJhZw==","published":"2020-07-17T14:08:00.000000Z","created_at":"2020-07-17T14:14:43.213210Z"},{"id":2518428095,"feed_id":1389029,"title":"Blizzard Watch Podcast 277: Deep dive into the WoW Shadowlands beta revelations","author":"Dan O'Halloran","summary":"This week, Matt Rossi and Joe Perez dig into the massive amount of WoW Shadowlands beta info recently released to the community. You can listen to the podcast directly from this post or click on download icon in the upper right corner of the player below","content":"\u003cdiv style=\"margin: 5px 5% 10px 5%;\"\u003e\u003cimg src=\"https://blizzardwatch.com/wp-content/uploads/2019/12/BW_Podcast-Matt-Joe-675_logo_header.jpg\" width=\"675\" height=\"380\" title=\"\" alt=\"\"\u003e\u003c/div\u003e\u003cdiv\u003eThis week, Matt Rossi and Joe Perez dig into the massive amount of WoW Shadowlands beta info recently released to the community. You can listen to the podcast directly from this post or click on download icon in the upper right corner of the player below to listen on your own device. Have your own...\u003c/div\u003e","url":"https://blizzardwatch.com/2020/07/17/blizzard-watch-podcast-277-wow-shadowlands-beta/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/06db7a5efedcb6fc62bc1d0d6f94915448a7fa1b?base64_url=aHR0cHM6Ly9ibGl6emFyZHdhdGNoLmNvbS8yMDIwLzA3LzE3L2JsaXp6YXJkLXdhdGNoLXBvZGNhc3QtMjc3LXdvdy1zaGFkb3dsYW5kcy1iZXRhLw==","published":"2020-07-17T16:30:11.000000Z","created_at":"2020-07-17T16:32:46.813498Z"},{"id":2518520337,"feed_id":184804,"title":"Slide Trombone","author":null,"summary":"","content":"\u003cimg src=\"https://imgs.xkcd.com/comics/slide_trombone.png\" title=\"Remember the CPS 2000, the super soaker that was discontinued because it was too powerful? Relatedly, can I borrow your tuba?\" alt=\"Remember the CPS 2000, the super soaker that was discontinued because it was too powerful? Relatedly, can I borrow your tuba?\"\u003e","url":"https://xkcd.com/2334/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/104eaf62d106808bc875365061bee6b9593caa6c?base64_url=aHR0cHM6Ly94a2NkLmNvbS8yMzM0Lw==","published":"2020-07-17T00:00:00.000000Z","created_at":"2020-07-17T17:52:08.273275Z"},{"id":2518532381,"feed_id":1735,"title":"MacRumors Giveaway: Win an Apple-Themed Plush Pillow From Throwboy","author":"Juli Clover","summary":"For this week's giveaway, we've teamed up with Throwboy to offer MacRumors readers a chance to win a plush, squeezable version of a classic Apple product like an original iPhone, Classic Macintosh, iMac G3, original iPod, and more. Throwboy has a","content":"For this week's \u003ca href=\"https://www.macrumors.com/guide/giveaway/\"\u003egiveaway\u003c/a\u003e, we've teamed up with \u003ca href=\"https://throwboy.com/\"\u003eThrowboy\u003c/a\u003e to offer \u003cem\u003eMacRumors\u003c/em\u003e readers a chance to win a plush, squeezable version \u003ca href=\"https://throwboy.com/collections/shop\"\u003eof a classic Apple product\u003c/a\u003e like an original \u003ca href=\"https://www.macrumors.com/guide/iphone/\"\u003eiPhone\u003c/a\u003e, Classic Macintosh, \u003ca href=\"https://www.macrumors.com/roundup/imac/\"\u003eiMac\u003c/a\u003e G3, original iPod, and more.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2018/08/iconicpillowcollection1.jpg\" alt=\"\" width=\"680\" height=\"383\" class=\"aligncenter size-full wp-image-650867\"\u003e\n\u003cbr\u003e\nThrowboy has a collection of different plush pillow options modeled after Apple designs, with each pillow priced at $30 to $40. The main collection of Iconic pillows is meant to look like classic Apple devices including the 1984 original Macintosh, the 1998 ‌iMac‌ G3, the 2001 iPod, and the 2007 original ‌iPhone‌.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2018/08/iconicpillowcollection4.jpg\" alt=\"\" width=\"680\" height=\"382\" class=\"aligncenter size-full wp-image-650868\"\u003e\n\u003cbr\u003e\nThere are also other designs available from the Throwboy website like the classic \u003ca href=\"https://throwboy.com/collections/shop/iconpillow\"\u003eFinder icon\u003c/a\u003e and the rainbow colored \u003ca href=\"https://throwboy.com/collections/shop/spinningwheel\"\u003eSpinning Wheel pillow\u003c/a\u003e modeled after the Mac's infamous wheel of death.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2018/08/iconicpillowcollection2.jpg\" alt=\"\" width=\"680\" height=\"383\" class=\"aligncenter size-full wp-image-650870\"\u003e\n\u003cbr\u003e\nEach Throwboy pillow features intricate embroidery to match the detailing on the Apple product they're meant to look like. The \u003ca href=\"https://throwboy.com/collections/shop/1998pillow\"\u003e1998 pillow\u003c/a\u003e, modeled after the original ‌iMac‌ G3, features bright Bondi Blue fabric with details like the CD slot, power button, and more.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2019/05/throwboyimacg3.jpg\" alt=\"\" width=\"800\" height=\"600\" class=\"aligncenter size-full wp-image-694998\"\u003e\n\u003cbr\u003e\nThe \u003ca href=\"https://throwboy.com/collections/shop/1984pillow\"\u003e1984 pillow\u003c/a\u003e designed to look like the Classic Macintosh has a rainbow logo, floppy disk drive, and iconic square shape, and the \u003ca href=\"https://throwboy.com/collections/shop/1977pillow\"\u003e1977 Apple II pillow\u003c/a\u003e has embroidered keys, a rainbow logo, and other true to life detailing.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2018/08/iconicpillowcollection4.jpg\" alt=\"\" width=\"680\" height=\"382\" class=\"aligncenter size-full wp-image-650868\"\u003e\n\u003cbr\u003e\nThe \u003ca href=\"https://throwboy.com/collections/shop/2001pillow\"\u003e2001 pillow\u003c/a\u003e modeled after Apple's original iPod features \u003ca href=\"https://throwboy.com/collections/shop/2001pillow\"\u003eclick wheel embroidery\u003c/a\u003e, the \u003ca href=\"https://throwboy.com/collections/shop/2007pillow\"\u003e2007 pillow\u003c/a\u003e looks just like the original ‌iPhone‌ with thick black bezels and a white body, and the \u003ca href=\"https://throwboy.com/collections/shop/spinningwheel\"\u003emulti-colored spinning wheel pillow\u003c/a\u003e is a life size version of the Mac icon.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/05/throwboyspinningwheel.jpg\" alt=\"\" width=\"2048\" height=\"2048\" class=\"aligncenter size-full wp-image-740874\"\u003e\n\u003cbr\u003e\nAll of Throwboy's plushes are soft, huggable, and the ideal size to use as a throw pillow on a couch or a chair. The Iconic pillow collection is also a great gift idea that would be perfect for any Apple fan.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2018/08/iconicpillowcollection3.jpg\" alt=\"\" width=\"680\" height=\"383\" class=\"aligncenter size-full wp-image-650869\"\u003e\n\u003cbr\u003e\nWe have 10 of Throwboy's pillows to give away, and each winner will be able to pick their favorite pillow: ‌iMac‌ G3, iPod, Apple II, Classic Macintosh, Finder Icon, and original ‌iPhone‌. To enter to win our ‌giveaway‌, use the Gleam.io widget below and enter an email address. Email addresses will be used solely for contact purposes to reach the winners and send the prizes. You can earn additional entries by subscribing to our \u003ca href=\"https://my.leadpages.net/leadbox/145ef7173f72a2%3A104a9fa76b46dc/5754903989321728/\"\u003eweekly newsletter\u003c/a\u003e, subscribing to our \u003ca href=\"https://www.youtube.com/user/macrumors\"\u003eYouTube channel\u003c/a\u003e, following us on \u003ca href=\"https://twitter.com/macrumors\"\u003eTwitter\u003c/a\u003e, following us on \u003ca href=\"https://www.instagram.com/macrumors/\"\u003eInstagram\u003c/a\u003e, or visiting the \u003cem\u003eMacRumors\u003c/em\u003e \u003ca href=\"https://www.facebook.com/MacRumors\"\u003eFacebook page\u003c/a\u003e.\n\u003cbr\u003e\n\n\u003cbr\u003e\nDue to the complexities of international laws regarding giveaways, only \u003cstrong\u003eU.S. residents who are 18 years or older and Canadian residents (excluding Quebec) who have reached the age of majority in their province or territory are eligible to enter\u003c/strong\u003e. To offer feedback or get more information on the ‌giveaway‌ restrictions, please refer to \u003ca href=\"http://forums.macrumors.com/threads/macrumors-giveaways-why-are-they-only-open-to-u-s-residents.1918337/\"\u003eour Site Feedback section\u003c/a\u003e, as that is where discussion of the rules will be redirected.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ccenter\u003e\n\u003ca class=\"e-widget no-button\" href=\"https://gleam.io/cvoY9/throwboy-giveaway\" rel=\"nofollow\"\u003eThrowboy Giveaway\u003c/a\u003e\n\u003cbr\u003e\n\u003cscript type=\"text/javascript\" src=\"https://widget.gleamjs.io/e.js\" async=\"true\"\u003e\u003c/script\u003e\n\u003c/center\u003e\n\u003cbr\u003e\nThe contest will run from today (July 17) at 11:00 a.m. Pacific Time through 11:00 a.m. Pacific Time on July 24. The winners will be chosen randomly on July 24 and will be contacted by email. The winners will have 48 hours to respond and provide a shipping address before new winners are chosen.\n\u003cbr\u003e\n\n\u003cbr\u003e\nFor those that want to purchase a pillow, Throwboy is offering a 15 percent discount with promo code \"rumors\" when checking out. The deal will be available until Sunday, July 19.\u003cdiv class=\"linkback\"\u003eTag: \u003ca href=\"https://www.macrumors.com/guide/giveaway/\"\u003egiveaway\u003c/a\u003e\n\u003c/div\u003e\u003cbr\u003eThis article, \"\u003ca href=\"https://www.macrumors.com/2020/07/17/macrumors-giveaway-throwboy-apple-pillow/\"\u003eMacRumors Giveaway: Win an Apple-Themed Plush Pillow From Throwboy\u003c/a\u003e\" first appeared on \u003ca href=\"https://www.macrumors.com\"\u003eMacRumors.com\u003c/a\u003e\u003cbr\u003e\u003cbr\u003e\u003ca href=\"https://forums.macrumors.com/threads/macrumors-giveaway-win-an-apple-themed-plush-pillow-from-throwboy.2246312/\"\u003eDiscuss this article\u003c/a\u003e in our forums\u003cbr\u003e\u003cbr\u003e\u003cdiv class=\"feedflare\"\u003e\n\u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=EWSlhQpPvrE:IxhfEFGbyr0:6W8y8wAjSf4\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=6W8y8wAjSf4\" border=\"0\"\u003e\u003c/a\u003e \u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=EWSlhQpPvrE:IxhfEFGbyr0:qj6IDK7rITs\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=qj6IDK7rITs\" border=\"0\"\u003e\u003c/a\u003e\n\u003c/div\u003e\u003cimg src=\"http://feeds.feedburner.com/~r/MacRumors-All/~4/EWSlhQpPvrE\" height=\"1\" width=\"1\" alt=\"\"\u003e","url":"https://www.macrumors.com/2020/07/17/macrumors-giveaway-throwboy-apple-pillow/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/1c9d9f113eca0436e66d72a4e809134c79fe0221?base64_url=aHR0cHM6Ly93d3cubWFjcnVtb3JzLmNvbS8yMDIwLzA3LzE3L21hY3J1bW9ycy1naXZlYXdheS10aHJvd2JveS1hcHBsZS1waWxsb3cv","published":"2020-07-17T18:00:54.000000Z","created_at":"2020-07-17T18:01:42.191601Z"},{"id":2518599488,"feed_id":1466657,"title":"Cameron Diaz Is Launching Vegan Wine Brand Avaline","author":"Audrey Enjoli","summary":"Cameron Diaz just launched her own clean wine brand, Avaline. The line features two varieties of wine—and they’re both vegan. The actor created the mini wine collection with entrepreneur Katherine Power, co-founder of Who What Wear and founder of Versed","content":"\u003cp\u003eCameron Diaz just launched her own clean wine brand, Avaline. The line features two varieties of wine—and they’re both vegan. The actor created the mini wine collection with entrepreneur Katherine Power, co-founder of Who What Wear and founder of Versed Skincare. Made in Spain, the Avaline White is a dry wine with a fresh, crisp […]\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co/cameron-diaz-vegan-wine-brand-avaline/\"\u003eCameron Diaz Is Launching Vegan Wine Brand Avaline\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co\"\u003eLIVEKINDLY\u003c/a\u003e.\u003c/p\u003e\n","url":"https://www.livekindly.co/cameron-diaz-vegan-wine-brand-avaline/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/e3bec9cd7b22f80982136287b8169ccfae6a7751?base64_url=aHR0cHM6Ly93d3cubGl2ZWtpbmRseS5jby9jYW1lcm9uLWRpYXotdmVnYW4td2luZS1icmFuZC1hdmFsaW5lLw==","published":"2020-07-17T18:37:09.000000Z","created_at":"2020-07-17T19:02:58.620933Z"},{"id":2518604837,"feed_id":1735,"title":"iOS 14's Compact Interface: Phone Calls, FaceTime, Siri and More","author":"Juli Clover","summary":"iOS 14 introduces several compact user interface elements that iPhone users have been requesting for years, such as a minimized phone call pop up that doesn't take over the entire screen and a smaller Siri window that doesn't dominate the display. This","content":"\u003ca href=\"https://www.macrumors.com/roundup/ios-14/\"\u003eiOS 14\u003c/a\u003e introduces several compact user interface elements that \u003ca href=\"https://www.macrumors.com/guide/iphone/\"\u003eiPhone\u003c/a\u003e users have been requesting for years, such as a minimized phone call pop up that doesn't take over the entire screen and a smaller \u003ca href=\"https://www.macrumors.com/guide/siri/\"\u003eSiri\u003c/a\u003e window that doesn't dominate the display.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/iOS-14-Compact-3d.jpg\" alt=\"\" width=\"1920\" height=\"1080\" class=\"aligncenter size-full wp-image-750750\"\u003e\n\u003cbr\u003e\nThis guide covers all of the new compact UI elements that Apple has added, but for more details on ‌iOS 14‌ design changes, make sure to \u003ca href=\"https://www.macrumors.com/guide/home-screen/\"\u003echeck out our Home Screen guide\u003c/a\u003e and our \u003ca href=\"https://www.macrumors.com/roundup/ios-14/\"\u003efull iOS 14 roundup\u003c/a\u003e.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2 id=\"phone_calls\"\u003ePhone Calls\u003c/h2\u003e\n\u003cbr\u003e\nWhen you're in the middle of using your ‌iPhone‌ and a phone call comes in, it's frustrating to have the incoming call take over the ‌iPhone‌'s display, especially if it's a spam call that you don't care about.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/ios14incomingcallcollapsed.jpg\" alt=\"\" width=\"2000\" height=\"1946\" class=\"aligncenter size-full wp-image-750622\"\u003e\n\u003cbr\u003e\nThat's no longer a problem in ‌iOS 14‌, because incoming phone calls are shown in a notification banner rather than a full incoming phone call display like in \u003ca href=\"https://www.macrumors.com/roundup/ios-13/\"\u003eiOS 13\u003c/a\u003e.\n\u003cbr\u003e\n\n\u003cbr\u003e\nPhone calls are only shown in banner format \u003cstrong\u003ewhen your phone is unlocked\u003c/strong\u003e and in use, so when your ‌iPhone‌ is locked, the Lock Screen interface where with the slide to unlock bar is still present.\n\u003cbr\u003e\n\n\u003cbr\u003e\nYou can tap on the decline button of incoming phone call banner to decline the call right away. Tapping on the accept button activates the call right in the banner interface, but if you tap outside of the accept/reject buttons the call opens in full screen as normal. If you swipe the banner away, you can continue to use your ‌iPhone‌ as normal while the phone rings in the background with a small icon shown over the time at the top left of the display.\n\u003cbr\u003e\n\n\u003cbr\u003e\nIf you tap a phone call banner and open the full display interface, you'll need to accept or decline the call to get out of the phone call view. If you've swiped the banner away, you can tap on the call icon in the upper left corner of the display to get to the call interface.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/incomingcallcollapsedios14.jpg\" alt=\"\" width=\"1400\" height=\"891\" class=\"aligncenter size-full wp-image-750623\"\u003e\n\u003cbr\u003e\n\u003ch2 id=\"facetime_calls\"\u003e\n\u003ca href=\"https://www.macrumors.com/guide/facetime/\"\u003eFaceTime\u003c/a\u003e Calls\u003c/h2\u003e\n\u003cbr\u003e\nIncoming ‌FaceTime‌ calls in ‌iOS 14‌ look the same as incoming phone calls when the ‌iPhone‌ is in use, showing up as a banner that can be swiped away or tapped to accept or decline a call. Note that ‌FaceTime‌ calls only show up in banner format when the ‌iPhone‌ is unlocked and in use. On the Lock Screen, ‌FaceTime‌ calls continue to take up the whole screen and will activate the camera.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/ios14facetime.jpg\" alt=\"\" width=\"1400\" height=\"885\" class=\"aligncenter size-full wp-image-750624\"\u003e\n\u003cbr\u003e\nTapping on a ‌FaceTime‌ call banner will extend the ‌FaceTime‌ call interface to the entire display as it works in ‌iOS 13‌, where you can either decide to accept or decline the call. Tapping right on the \"X\" button will automatically decline the call from the banner, however.\n\u003cbr\u003e\n\n\u003cbr\u003e\nSwiping the ‌FaceTime‌ banner away allows the ‌FaceTime‌ call to continue ringing in the background without disturbing you. You'll see the ‌FaceTime‌ icon over the time in the upper left corner of the display, which you can tap to get back to the ‌FaceTime‌ interface if you want to accept or decline the call. Otherwise, it rings in the background for a few seconds until it registers that you're not picking up.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/facetimecollapsediconios14.jpg\" alt=\"\" width=\"1400\" height=\"893\" class=\"aligncenter size-full wp-image-750625\"\u003e\n\u003cbr\u003e\nOnce you've tapped on a ‌FaceTime‌ call banner or icon, the ‌FaceTime‌ call expands to the full display and the only way to get out of the interface is to accept or decline the ‌FaceTime‌ call.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2 id=\"third-party_voip_calls\"\u003eThird-Party VOIP Calls\u003c/h2\u003e\n\u003cbr\u003e\nApple created a new API for the collapsed call interface that will let developers incorporate the feature into third-party VOIP apps like Facebook Messenger, WhatsApp, Skype, and tons more.\n\u003cbr\u003e\n\n\u003cbr\u003e\nApp developers will need to implement this API into their apps before their apps will be shown with the updated call interface.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2 id=\"picture_in_picture\"\u003ePicture in Picture\u003c/h2\u003e\n\u003cbr\u003e\nApple considers the new Picture in Picture mode on ‌iPhone‌ to be part of the updated compact user interface in ‌iOS 14‌. With Picture in Picture mode you can watch TV shows, movies, and videos from apps and the web while doing other things on your ‌iPhone‌ with the content shown in a windowed view.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/pictureinpicturesafari2.jpg\" alt=\"\" width=\"2000\" height=\"1946\" class=\"aligncenter size-full wp-image-749834\"\u003e\n\u003cbr\u003e\nThere's a ‌FaceTime‌ Picture in Picture feature that's even more useful because there's now a way to carry on a ‌FaceTime‌ conversation while also using your ‌iPhone‌ as normal. In ‌iOS 13‌, exiting out of the ‌FaceTime‌ app pauses the ‌FaceTime‌ video for the person you're speaking with, but in ‌iOS 14‌, exiting out of ‌FaceTime‌ collapses the video view into a small window so both parties can continue their discussion.\n\u003cbr\u003e\n\n\u003cbr\u003e\nWe have a \u003ca href=\"https://www.macrumors.com/guide/picture-in-picture/\"\u003ePicture in Picture guide\u003c/a\u003e with complete details on how the feature works, so make sure to check that out for more info on the Picture in Picture feature.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2 id=\"siri\"\u003e‌Siri‌\u003c/h2\u003e\n\u003cbr\u003e\n‌Siri‌ requests in ‌iOS 14‌ are also part of the compact interface, and activating ‌Siri‌ no longer takes over the entire ‌iPhone‌. Instead, when you activate ‌Siri‌ through a \"Hey ‌Siri‌\" voice command or through a physical button, a circular, animated ‌Siri‌ icon pops up at the bottom of the ‌iPhone‌'s display.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/siriios14.jpg\" alt=\"\" width=\"2000\" height=\"1946\" class=\"aligncenter size-full wp-image-750626\"\u003e\n\u003cbr\u003e\nThe ‌Siri‌ icon moves as you speak to ‌Siri‌ so you can be sure the voice assistant is listening and picking up what you have to say. Queries are also displayed in a more compact view with information provided at the top in a banner format.\n\u003cbr\u003e\n\n\u003cbr\u003e\nIf, for example, you ask ‌Siri‌ the time or the weather, the information is shown in a banner-style popup at the top of the display.\n\u003cbr\u003e\n\n\u003cbr\u003e\nShorter information is shown in smaller banners, but longer questions may have banners that take up more space. This is especially true of results that involve web searches.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/siriios14full.jpg\" alt=\"\" width=\"800\" height=\"1589\" class=\"aligncenter size-full wp-image-750627\"\u003e\n\u003cbr\u003e\nCertain requests, such as those that show off ‌Siri‌'s personality like jokes or product information requests, show the results right above the ‌Siri‌ icon.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/sirijokesios14.jpg\" alt=\"\" width=\"2000\" height=\"1946\" class=\"aligncenter size-full wp-image-750629\"\u003e\n\u003cbr\u003e\nNote that in ‌iOS 14‌, ‌Siri‌ does not display the text of what you've spoken in order to keep the interface as small as possible, but if you enable the \"Always Show Speech\" feature, what you're speaking will show up in a small popup over the ‌Siri‌ icon. It's useful to see what ‌Siri‌ hears to make sure the voice assistant is properly interpreting commands.\n\u003cbr\u003e\n\n\u003cbr\u003e\nTurn on \"Always Show Speech\" by opening up Settings, going to ‌Siri‌ \u0026amp; Search, tapping on ‌Siri‌ Feedback and then toggling on the speech option. You can also turn on captions, which will show captions for all spoken text.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/siricaptions.jpg\" alt=\"\" width=\"800\" height=\"1611\" class=\"aligncenter size-full wp-image-750628\"\u003e\n\u003cbr\u003e\nIn any ‌Siri‌ result, while the ‌Siri‌ interface leaves the background of the ‌iPhone‌ visible so you can continue to see what you were doing prior to the ‌Siri‌ request, there is no option to interact with other apps while the ‌Siri‌ interface is open. Attempting to tap elsewhere on the display while ‌Siri‌ is active closes the ‌Siri‌ interface.\n\u003cbr\u003e\n\n\u003cbr\u003e\nAfter activating ‌Siri‌ the first time with a voice command or a physical button, the ‌Siri‌ interface stays open until you tap out of it, so you can tap on the ‌Siri‌ icon to initiate another ‌Siri‌ request. With ‌Siri‌ results that are interactive like web search results or translations, you can tap in the ‌Siri‌ window to execute the search, hear the spoken translation, and more. All in all, ‌Siri‌ in ‌iOS 14‌ works much the same as it does in ‌iOS 13‌, but in a way that feels like it's less of an interruption to general ‌iPhone‌ and \u003ca href=\"https://www.macrumors.com/roundup/ipad/\"\u003eiPad\u003c/a\u003e usage.\n\u003cbr\u003e\n\n\u003cbr\u003e\nAlong with the redesign, ‌Siri‌ has a few other new features in ‌iOS 14‌ such as support for sending audio messages, getting cycling directions, and sharing an \u003ca href=\"https://www.macrumors.com/guide/maps/\"\u003eApple Maps\u003c/a\u003e ETA with a contact, with how tos covering these new features available below.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cul\u003e\n\u003cbr\u003e\n\u003cli\u003e\u003ca href=\"https://www.macrumors.com/how-to/send-audio-message-using-siri-ios/\"\u003eiOS 14: How to Send an Audio Message Using Siri\u003c/a\u003e\u003c/li\u003e\n\u003cbr\u003e\n\u003cli\u003e\u003ca href=\"https://www.macrumors.com/how-to/get-siri-transcribe-requests-ios/\"\u003eHow to Get Siri to Transcribe Your Requests in iOS 14\u003c/a\u003e\u003c/li\u003e\n\u003cbr\u003e\n\u003cli\u003e\u003ca href=\"https://www.macrumors.com/how-to/get-cycling-directions-siri-ios/\"\u003eHow to Get Cycling Directions From Siri in iOS 14\u003c/a\u003e\u003c/li\u003e\n\u003cbr\u003e\n\u003cli\u003e\u003ca href=\"https://www.macrumors.com/how-to/ask-siri-share-eta-navigating-apple-maps/\"\u003eHow to Ask Siri to Share Your ETA When Navigating With Apple Maps\u003c/a\u003e\u003c/li\u003e\n\u003cbr\u003e\n\u003c/ul\u003e\u003ch2 id=\"guide_feedback\"\u003eGuide Feedback\u003c/h2\u003e\n\u003cbr\u003e\nHave questions about the interface changes in ‌iOS 14‌, know of a feature we left out, or want to offer feedback on this guide? \u003ca href=\"mailto:juli@macrumors.com\"\u003eSend us an email here\u003c/a\u003e. For more on everything new in ‌iOS 14‌, make sure to \u003ca href=\"https://www.macrumors.com/roundup/ios-14/\"\u003echeck out our iOS 14 roundup\u003c/a\u003e and our \u003ca href=\"https://www.macrumors.com/guide/\"\u003eother iOS 14 guides\u003c/a\u003e. \u003cdiv class=\"linkback\"\u003eRelated Roundups: \u003ca href=\"https://www.macrumors.com/roundup/ios-14/\"\u003eiOS 14\u003c/a\u003e, \u003ca href=\"https://www.macrumors.com/roundup/ipados-14/\"\u003eiPadOS 14\u003c/a\u003e\n\u003c/div\u003e\u003cbr\u003eThis article, \"\u003ca href=\"https://www.macrumors.com/guide/ios-14-compact-interface/\"\u003eiOS 14's Compact Interface: Phone Calls, FaceTime, Siri and More\u003c/a\u003e\" first appeared on \u003ca href=\"https://www.macrumors.com\"\u003eMacRumors.com\u003c/a\u003e\u003cbr\u003e\u003cbr\u003e\u003ca href=\"https://forums.macrumors.com/threads/ios-14s-compact-interface-phone-calls-facetime-siri-and-more.2246323/\"\u003eDiscuss this article\u003c/a\u003e in our forums\u003cbr\u003e\u003cbr\u003e\u003cdiv class=\"feedflare\"\u003e\n\u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=HlSOMBEc6gY:A9mnlfLEXg8:6W8y8wAjSf4\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=6W8y8wAjSf4\" border=\"0\"\u003e\u003c/a\u003e \u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=HlSOMBEc6gY:A9mnlfLEXg8:qj6IDK7rITs\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=qj6IDK7rITs\" border=\"0\"\u003e\u003c/a\u003e\n\u003c/div\u003e\u003cimg src=\"http://feeds.feedburner.com/~r/MacRumors-All/~4/HlSOMBEc6gY\" height=\"1\" width=\"1\" alt=\"\"\u003e","url":"https://www.macrumors.com/guide/ios-14-compact-interface/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/7e5ae5c378abe6e8177da626d7a404c37caf3b4d?base64_url=aHR0cHM6Ly93d3cubWFjcnVtb3JzLmNvbS9ndWlkZS9pb3MtMTQtY29tcGFjdC1pbnRlcmZhY2Uv","published":"2020-07-17T19:06:10.000000Z","created_at":"2020-07-17T19:11:37.735801Z"},{"id":2518615561,"feed_id":1405087,"title":"∞ The Dalrymple Report: Dave’s dad, Twitter, and lyrics","author":"Jim Dalrymple","summary":"Do you remember your childhood phone number? How about saying your name backwards? Dave has all kinds of questions today. We also talk about Apple Music lyrics, Twitter hacking, and Apple News+ audio before Dave tells a wonderful story about iPhone and","content":"\u003cp\u003eDo you remember your childhood phone number? How about saying your name backwards? Dave has all kinds of questions today. We also talk about Apple Music lyrics, Twitter hacking, and Apple News+ audio before Dave tells a wonderful story about iPhone and spending some time with his dad.\u003c/p\u003e\n\n\u003cp\u003e\u003ca href=\"http://loopu.in/1KsW16v\"\u003eSubscribe to this podcast\u003c/a\u003e\u003c/p\u003e\n\n\u003cp\u003e\u003cstrong\u003eBrought to you by:\u003c/strong\u003e\u003c/p\u003e\n\n\u003cp\u003e\u003cstrong\u003eRaycon\u003c/strong\u003e: Now’s the time to get the latest and greatest from Raycon. Get 15-percent off your order at \u003ca href=\"https://buyraycon.com/dalrymple\"\u003ehttps://buyraycon.com/dalrymple\u003c/a\u003e! That’s \u003ca href=\"https://buyraycon.com/dalrymple\"\u003ehttps://buyraycon.com/dalrymple\u003c/a\u003e for 15-percent off Raycon wireless earbuds. Use code Dalrymple15 at checkout. \u003ca href=\"https://buyraycon.com/dalrymple\"\u003ehttps://buyraycon.com/dalrymple\u003c/a\u003e\u003c/p\u003e\n\n\u003ciframe style=\"border: none\" src=\"http://html5-player.libsyn.com/embed/episode/id/15251225/height/45/theme/standard/thumbnail/no/direction/backward/\" height=\"45\" width=\"100%\" scrolling=\"no\" allowfullscreen webkitallowfullscreen mozallowfullscreen oallowfullscreen msallowfullscreen\u003e\u003c/iframe\u003e\n","url":"https://www.loopinsight.com/2020/07/17/the-dalrymple-report-daves-dad-twitter-and-lyrics/?utm_source=rss\u0026utm_medium=rss\u0026utm_campaign=the-dalrymple-report-daves-dad-twitter-and-lyrics","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/fbc3304ef3550a93199ea149fb5c097eb429ca67?base64_url=aHR0cHM6Ly93d3cubG9vcGluc2lnaHQuY29tLzIwMjAvMDcvMTcvdGhlLWRhbHJ5bXBsZS1yZXBvcnQtZGF2ZXMtZGFkLXR3aXR0ZXItYW5kLWx5cmljcy8_dXRtX3NvdXJjZT1yc3MmdXRtX21lZGl1bT1yc3MmdXRtX2NhbXBhaWduPXRoZS1kYWxyeW1wbGUtcmVwb3J0LWRhdmVzLWRhZC10d2l0dGVyLWFuZC1seXJpY3M=","published":"2020-07-17T19:22:29.000000Z","created_at":"2020-07-17T19:25:03.689774Z"},{"id":2518715097,"feed_id":1389029,"title":"Ion Hazzikostas says destructible Conduit system won’t go live, leaving us with more questions","author":"Matthew Rossi","summary":"For some folks, the upcoming design of Conduits as part of World of Warcraft’s next expansion Shadowlands has been a concern, and there’s been some debate about if it makes sense if they could be permanently destroyed when removed, the way gems have worked","content":"\u003cdiv style=\"margin: 5px 5% 10px 5%;\"\u003e\u003cimg src=\"https://blizzardwatch.com/wp-content/uploads/2020/07/bw_soulbind_tree_with_conduits_675_380.png\" width=\"675\" height=\"380\" title=\"\" alt=\"\"\u003e\u003c/div\u003e\u003cdiv\u003eFor some folks, the upcoming design of Conduits as part of World of Warcraft’s next expansion Shadowlands has been a concern, and there’s been some debate about if it makes sense if they could be permanently destroyed when removed, the way gems have worked in gear over the years. It feels like those folks will be relieved...\u003c/div\u003e","url":"https://blizzardwatch.com/2020/07/17/destructible-conduit-system/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/70004c1865de9fecd1832559183c6d54dcb95cf7?base64_url=aHR0cHM6Ly9ibGl6emFyZHdhdGNoLmNvbS8yMDIwLzA3LzE3L2Rlc3RydWN0aWJsZS1jb25kdWl0LXN5c3RlbS8=","published":"2020-07-17T21:00:41.000000Z","created_at":"2020-07-17T21:07:56.423638Z"},{"id":2518766860,"feed_id":904489,"title":"Fun With Charts, Apple Silicon edition","author":"Jason Snell","summary":"I’ve been thinking about this chart since it appeared in Apple’s WWDC keynote last month. On one level, it’s a meaningless bit of marketing fluff. There are no labels. The vast breadth of notebooks and desktop PCs is represented by two perfect rounded","content":"\u003cfigure\u003e\u003cimg src=\"https://i2.wp.com/sixcolors.com/wp-content/uploads/2020/07/apple-silicon-chart.png?ssl=1\" alt=\"\" data-image-w=\"\" data-image-h=\"\" class=\" jetpack-broken-image\" data-recalc-dims=\"1\"\u003e\u003cfigcaption\u003e\u003c/figcaption\u003e\u003c/figure\u003e\u003cp\u003eI’ve been thinking about this chart since it appeared in Apple’s WWDC keynote last month.\u003c/p\u003e\n\u003cp\u003eOn one level, it’s a meaningless bit of marketing fluff. There are no labels. The vast breadth of notebooks and desktop PCs is represented by two perfect rounded rectangles. “Macs with Apple Silicon” are represented by a fuzzy gradient of squares. What does it mean?\u003c/p\u003e\n\u003cp\u003eIt’s a question that won’t be answered until after the Betas of Summer have gone. It’ll be a while before we get a clear picture of what the first Macs of this new era will look like. We don’t have a lot to go on… but there is this chart.\u003c/p\u003e\n\u003cp\u003eSo here’s what it implies:\u003c/p\u003e\n\u003cp\u003e\u003cstrong\u003eMacs with Apple Silicon will largely be faster than existing notebooks.\u003c/strong\u003e This isn’t too far-fetched, given Apple’s claims about the superior performance of the 2018 iPad Pro when it was announced. If the iPad Pro is faster than most PC laptops, wouldn’t it stand to reason that even the slowest of the new generation of Macs would be as fast or faster than the competition?\u003c/p\u003e\n\u003cp\u003eLet’s not forget how Apple’s pride factors in to this transition. If Apple releases a Mac that’s not any faster than the Intel-based competition, that reflects poorly on the prowess of Apple’s processor design. I firmly believe that Apple would never initiate a processor transition like this unless it was confident that its chips would end up looking like world-beaters.\u003c/p\u003e\n\u003cp\u003e\u003cstrong\u003eSome Macs with Apple Silicon will offer lower power consumption than existing notebooks, at the same or greater performance.\u003c/strong\u003e What Apple chooses to do with that power efficiency is an open question, however. It could use it to shave some size and weight off its batteries while maintaining the same quoted battery life totals as current models. It could use it to boost its battery-life claims to new heights. Or it could spend that energy savings on additional performance, so that the next-generation Macs are that much faster than the competition.\u003c/p\u003e\n\u003cp\u003eIt seems unlikely that Apple will go all in on any of these strategies. Instead, I’d expect it to strike a balance. Ultimately, it will want to make lighter and thinner laptops that offer more power and longer battery life than the previous generation.\u003c/p\u003e\n\u003cp\u003e\u003cstrong\u003eMacs with Apple Silicon can offer Desktop-beating performance with dramatically lower power consumption.\u003c/strong\u003e What does a desktop computer (or a high-end laptop) running an Apple-designed ARM processor look like? The iPad can be at least a rough analog for a laptop, but no iPad or iPhone has ever been asked to match up with a high-performance PC before. “No problem,” Apple’s vague chart proclaims. “Our gradient extends above Desktops, while remaining far to the left on power consumption.”\u003c/p\u003e\n\u003cp\u003eWe’ll see what Apple’s got up its sleeve when it comes to more high-end applications. But I will remind you that the A12X and A12Z processors that power the 2018 and 2020 iPad Pro, respectively, are eight-core processors with seven- and eight-core graphics processors. Do you know what Mac models currently offer eight-core processors? The highest-end configuration of the 16-inch MacBook Pro, the iMac Pro, and the Mac Pro.\u003c/p\u003e\n\u003cp\u003eNow, you can’t gauge performance by counting cores any more than you can gauge image quality by counting megapixels. The A12 processors use two different kinds of cores, four geared toward energy savings and four geared toward performance, and the system can use those cores as needed given its current needs. But I do think it’s interesting that Apple has been gradually ramping up its core count on its processors, as it spins off more duties into integrated coprocessors like the Neural Engine (used for machine-learning tasks).\u003c/p\u003e\n\u003cp\u003eIt wouldn’t be surprising, then, for Apple to announce a new Mac processor that’s got a large number of processor cores, some of which would be able to sip power to extend MacBook Pro battery life, and some of which would crank up as fast as possible in moments of true power-user need.\u003c/p\u003e\n\u003cp\u003eAt least, that’s how I read Apple’s chart. We’ll all see how reality matches up later this year.\u003c/p\u003e","url":"https://sixcolors.com/post/2020/07/fun-with-charts-apple-silicon-edition/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/0525a71eaa8e9c32b942071e33f6dff585c7bd0d?base64_url=aHR0cHM6Ly9zaXhjb2xvcnMuY29tL3Bvc3QvMjAyMC8wNy9mdW4td2l0aC1jaGFydHMtYXBwbGUtc2lsaWNvbi1lZGl0aW9uLw==","published":"2020-07-17T21:48:45.000000Z","created_at":"2020-07-17T22:02:35.261258Z"},{"id":2518812117,"feed_id":1848912,"title":"Finally, made it to the weekend, time to breathe, relax, and... Cloudflare's taken down a chunk of the web","author":"Thomas Claburn","summary":"DNS provider goes dark amid bad routing, world+dog goes through nine minutes of terror Updated Global internet glue Cloudflare experienced a brief network outage on Friday that broke multiple apps and websites, including your humble Register.…","content":"\u003ch4\u003eDNS provider goes dark amid bad routing, world+dog goes through nine minutes of terror\u003c/h4\u003e \u003cp\u003e\u003cstrong\u003eUpdated\u003c/strong\u003e  Global internet glue Cloudflare experienced a brief network outage on Friday that broke multiple apps and websites, including your humble \u003ci\u003eRegister\u003c/i\u003e.…\u003c/p\u003e \u003cp\u003e\u003c!--#include virtual='/data_centre/_whitepaper_textlinks_top.html' --\u003e\u003c/p\u003e","url":"https://go.theregister.com/feed/www.theregister.com/2020/07/17/cloudflare_outage_bgp_dns/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/94806bc9c8db71af17d23bdfeff7fefb2787d963?base64_url=aHR0cHM6Ly9nby50aGVyZWdpc3Rlci5jb20vZmVlZC93d3cudGhlcmVnaXN0ZXIuY29tLzIwMjAvMDcvMTcvY2xvdWRmbGFyZV9vdXRhZ2VfYmdwX2Rucy8=","published":"2020-07-17T22:32:48.000000Z","created_at":"2020-07-17T22:58:50.270325Z"},{"id":2518892539,"feed_id":1178507,"title":"watchOS 7: How to enable and use handwashing detection","author":"Filipe Espósito","summary":"One of the new features of watchOS 7 highlighted at WWDC 2020 is the handwashing detection, which helps users wash their hands properly to eliminate germs. This feature is disabled by default, so read on as we explain how to enable and use handwashing","content":"\u003cdiv class=\"feat-image\"\u003e\u003cimg src=\"https://9to5mac.com/wp-content/uploads/sites/6/2020/07/Handwashing-watchOS-7.png?w=1600\"\u003e\u003c/div\u003e\n\u003cp\u003eOne of the new features of \u003ca href=\"https://9to5mac.com/2020/06/22/apple-previews-watchos-7-for-apple-watch-sleep-tracking-face-sharing-new-workouts-including-dance-activity-now-fitness-app/\" target=\"_blank\" rel=\"noopener noreferrer\"\u003ewatchOS 7\u003c/a\u003e highlighted at WWDC 2020 is the \u003cstrong\u003ehandwashing detection\u003c/strong\u003e, which helps users wash their hands properly to eliminate germs. This feature is disabled by default, so read on as we explain how to enable and use handwashing detection in watchOS 7.\u003c/p\u003e\n\u003cp\u003e \u003ca href=\"https://9to5mac.com/2020/07/17/watchos-7-how-to-enable-and-use-handwashing-detection/#more-657049\" class=\"more-link\"\u003emore…\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://9to5mac.com/2020/07/17/watchos-7-how-to-enable-and-use-handwashing-detection/\"\u003ewatchOS 7: How to enable and use handwashing detection\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://9to5mac.com\"\u003e9to5Mac\u003c/a\u003e.\u003c/p\u003e\n","url":"https://9to5mac.com/2020/07/17/watchos-7-how-to-enable-and-use-handwashing-detection/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/cac8d46dfe99b359a6a20b178ed354b4c4081c31?base64_url=aHR0cHM6Ly85dG81bWFjLmNvbS8yMDIwLzA3LzE3L3dhdGNob3MtNy1ob3ctdG8tZW5hYmxlLWFuZC11c2UtaGFuZHdhc2hpbmctZGV0ZWN0aW9uLw==","published":"2020-07-18T00:30:56.000000Z","created_at":"2020-07-18T00:50:20.935271Z"},{"id":2518907271,"feed_id":1848912,"title":"US military whips out credit card for unmanned low-Earth-orbit outpost prototype (aka a repurposed ISS cargo pod)","author":"Katyanna Quach","summary":"Pentagon can't wait to do some 'experimentation and testing' in space The Pentagon wants an unmanned military outpost in orbit one day – and this week hired the Sierra Nevada Corporation (SNC) to build a prototype.…","content":"\u003ch4\u003ePentagon can't wait to do some 'experimentation and testing' in space\u003c/h4\u003e \u003cp\u003eThe Pentagon wants an unmanned military outpost in orbit one day – and this week hired the Sierra Nevada Corporation (SNC) to build a prototype.…\u003c/p\u003e","url":"https://go.theregister.com/feed/www.theregister.com/2020/07/18/sierra_nevada_space_station/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/6ee3f34abe8710bbba8c9b5886801e23fd78c5f2?base64_url=aHR0cHM6Ly9nby50aGVyZWdpc3Rlci5jb20vZmVlZC93d3cudGhlcmVnaXN0ZXIuY29tLzIwMjAvMDcvMTgvc2llcnJhX25ldmFkYV9zcGFjZV9zdGF0aW9uLw==","published":"2020-07-18T00:54:40.000000Z","created_at":"2020-07-18T01:14:04.754646Z"},{"id":2518999874,"feed_id":1296379,"title":"NYT: ‘Hackers Tell the Story of the Twitter Attack From the Inside’","author":"John Gruber","summary":"Extraordinary reporting by Nathaniel Popper and Kate Conger for The New York Times: But four people who participated in the scheme spoke with The Times and shared numerous logs and screen shots of the conversations they had on Tuesday and Wednesday,","content":"\n\u003cp\u003eExtraordinary reporting by Nathaniel Popper and Kate Conger for The New York Times:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eBut four people who participated in the scheme spoke with The\nTimes and shared numerous logs and screen shots of the\nconversations they had on Tuesday and Wednesday, demonstrating\ntheir involvement both before and after the hack became public.\u003c/p\u003e\n\n\u003cp\u003eThe interviews indicate that the attack was not the work of a\nsingle country like Russia or a sophisticated group of\nhackers. Instead, it was done by a group of young people —\none of whom says he lives at home with his mother — who got\nto know one another because of their obsession with owning\nearly or unusual screen names, particularly one letter or\nnumber, like @y or @6. […]\u003c/p\u003e\n\n\u003cp\u003eThe hacker “lol” and another one he worked with, who went by the\nscreen name “ever so anxious,” told The Times that they wanted to\ntalk about their work with Kirk in order to prove that they had\nonly facilitated the purchases and takeovers of lesser-known\nTwitter addresses early in the day. They said they had not\ncontinued to work with Kirk once he began more high-profile\nattacks around 3:30 p.m. Eastern time on Wednesday.\u003c/p\u003e\n\n\u003cp\u003eIn one of the first transactions, “lol” brokered a deal for\nsomeone who was willing to pay $1,500, in Bitcoin, for the Twitter\nuser name @y. The money went to the same Bitcoin wallet that Kirk\nused later in the day when he got payments from hacking the\nTwitter accounts of celebrities, the public ledger of Bitcoin\ntransactions shows.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp style=\"padding-top: 1.5em;\"\u003e\u003cem\u003eLink: \u003cstrong\u003e\u003ca href=\"https://www.nytimes.com/2020/07/17/technology/twitter-hackers-interview.html\"\u003enytimes.com/2020/07/17/technology/twitter-hackers-interview…\u003c/a\u003e\u003c/strong\u003e\u003c/em\u003e\u003c/p\u003e\n","url":"https://daringfireball.net/linked/2020/07/17/nyt-twitter-hack","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/b2909c78bf1f7611ccef3758cc6ccb52eea3365e?base64_url=aHR0cHM6Ly9kYXJpbmdmaXJlYmFsbC5uZXQvbGlua2VkLzIwMjAvMDcvMTcvbnl0LXR3aXR0ZXItaGFjaw==","published":"2020-07-18T03:52:50.000000Z","created_at":"2020-07-18T03:57:00.905859Z"},{"id":2519014472,"feed_id":1296379,"title":"Twitter: ‘An Update on Our Security Incident’","author":"John Gruber","summary":"Twitter: At this time, we believe attackers targeted certain Twitter employees through a social engineering scheme. What does this mean? In this context, social engineering is the intentional manipulation of people into performing certain actions and","content":"\n\u003cp\u003eTwitter:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eAt this time, we believe attackers targeted certain Twitter\nemployees through a social engineering scheme. What does this\nmean? In this context, social engineering is the intentional\nmanipulation of people into performing certain actions and\ndivulging confidential information.\u003c/p\u003e\n\n\u003cp\u003eThe attackers successfully manipulated a small number of employees\nand used their credentials to access Twitter’s internal systems,\nincluding getting through our two-factor protections. As of now,\nwe know that they accessed tools only available to our internal\nsupport teams to target 130 Twitter accounts. For 45 of those\naccounts, the attackers were able to initiate a password reset,\nlogin to the account, and send Tweets. We are continuing our\nforensic review of all of the accounts to confirm all actions that\nmay have been taken. In addition, we believe they may have\nattempted to sell some of the usernames.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003e“May” is a bit of a stretch here given that \u003ca href=\"https://www.nytimes.com/2020/07/17/technology/twitter-hackers-interview.html\"\u003eThe New York Times reported hours ago\u003c/a\u003e that they had been selling usernames, and \u003ca href=\"https://krebsonsecurity.com/2020/07/whos-behind-wednesdays-epic-twitter-hack/\"\u003eBrian Krebs reported it yesterday\u003c/a\u003e.\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eFor up to eight of the Twitter accounts involved, the attackers\ntook the additional step of downloading the account’s information\nthrough our “\u003ca href=\"https://help.twitter.com/en/managing-your-account/accessing-your-twitter-data\"\u003eYour Twitter Data\u003c/a\u003e” tool. This is a tool that is\nmeant to provide an account owner with a summary of their Twitter\naccount details and activity. We are reaching out directly to any\naccount owner where we know this to be true.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eDMs are the first thing that comes to mind with the “Your Twitter Data” tool. \u003ca href=\"https://help.twitter.com/en/managing-your-account/accessing-your-twitter-data\"\u003eThat archive contains pretty much everything\u003c/a\u003e, including your location data.\u003c/p\u003e\n\n\u003cp style=\"padding-top: 1.5em;\"\u003e\u003cem\u003eLink: \u003cstrong\u003e\u003ca href=\"https://blog.twitter.com/en_us/topics/company/2020/an-update-on-our-security-incident.html\"\u003eblog.twitter.com/en_us/topics/company/2020/an-update-on-our…\u003c/a\u003e\u003c/strong\u003e\u003c/em\u003e\u003c/p\u003e\n","url":"https://daringfireball.net/linked/2020/07/17/twitter-update-security-incident","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/ac0cc0b1c9b35757ee869ca226d1db2a23c003e3?base64_url=aHR0cHM6Ly9kYXJpbmdmaXJlYmFsbC5uZXQvbGlua2VkLzIwMjAvMDcvMTcvdHdpdHRlci11cGRhdGUtc2VjdXJpdHktaW5jaWRlbnQ=","published":"2020-07-18T03:56:47.000000Z","created_at":"2020-07-18T04:15:55.113212Z"},{"id":2519181629,"feed_id":1342834,"title":"Internet: Konfigurationsfehler nimmt Cloudflare-Dienste offline","author":"Sebastian Grüner","summary":"Wegen eines Fehlers reduzierte sich der Netzwerkverkehr bei Cloudflare auf nur noch 50 Prozent. Davon waren viele Webseiten betroffen. (Cloudflare, Netzwerk)","content":"\u003cimg src=\"https://www.golem.de/2007/149729-237181-237178_rc.jpg\" width=\"140\" height=\"140\" vspace=\"3\" hspace=\"8\" align=\"left\"\u003eWegen eines Fehlers reduzierte sich der Netzwerkverkehr bei Cloudflare auf nur noch 50 Prozent. Davon waren viele Webseiten betroffen. (\u003ca href=\"https://www.golem.de/specials/cloudflare/\"\u003eCloudflare\u003c/a\u003e, \u003ca href=\"https://www.golem.de/specials/netzwerk/\"\u003eNetzwerk\u003c/a\u003e) \u003cimg src=\"https://cpx.golem.de/cpx.php?class=17\u0026amp;aid=149729\u0026amp;page=1\u0026amp;ts=1595061180\" alt=\"\" width=\"1\" height=\"1\"\u003e","url":"https://www.golem.de/news/internet-konfigurationsfehler-nimmt-cloudflare-dienste-offline-2007-149729-rss.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/14a077c87092be1456d617b1fbccef2531865b28?base64_url=aHR0cHM6Ly93d3cuZ29sZW0uZGUvbmV3cy9pbnRlcm5ldC1rb25maWd1cmF0aW9uc2ZlaGxlci1uaW1tdC1jbG91ZGZsYXJlLWRpZW5zdGUtb2ZmbGluZS0yMDA3LTE0OTcyOS1yc3MuaHRtbA==","published":"2020-07-18T08:33:00.000000Z","created_at":"2020-07-18T08:48:42.029907Z"},{"id":2519190193,"feed_id":1342834,"title":"Microblogging: Twitter bestätigt 45 gehackte Accounts","author":"Sebastian Grüner","summary":"Twitter legt sich nun außerdem auf Social Engineering als Angriffsart fest. Bei acht Zugängen sind Daten heruntergeladen worden. (Twitter, Soziales Netz)","content":"\u003cimg src=\"https://www.golem.de/2007/149731-237189-237186_rc.jpg\" width=\"140\" height=\"140\" vspace=\"3\" hspace=\"8\" align=\"left\"\u003eTwitter legt sich nun außerdem auf Social Engineering als Angriffsart fest. Bei acht Zugängen sind Daten heruntergeladen worden. (\u003ca href=\"https://www.golem.de/specials/twitter/\"\u003eTwitter\u003c/a\u003e, \u003ca href=\"https://www.golem.de/specials/socialnetwork/\"\u003eSoziales Netz\u003c/a\u003e) \u003cimg src=\"https://cpx.golem.de/cpx.php?class=17\u0026amp;aid=149731\u0026amp;page=1\u0026amp;ts=1595063160\" alt=\"\" width=\"1\" height=\"1\"\u003e","url":"https://www.golem.de/news/microblogging-twitter-bestaetigt-45-gehackte-accounts-2007-149731-rss.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/7638640e5b0fffab7d260a7b89f43beb152d2f8b?base64_url=aHR0cHM6Ly93d3cuZ29sZW0uZGUvbmV3cy9taWNyb2Jsb2dnaW5nLXR3aXR0ZXItYmVzdGFldGlndC00NS1nZWhhY2t0ZS1hY2NvdW50cy0yMDA3LTE0OTczMS1yc3MuaHRtbA==","published":"2020-07-18T09:06:00.000000Z","created_at":"2020-07-18T09:09:54.708039Z"},{"id":2519191550,"feed_id":1466657,"title":"Aldi Will Remove 2.2 Billion Plastic Items From Stores By 2025","author":"Audrey Enjoli","summary":"Aldi has committed to removing 74,000 tonnes of plastic packaging from its stores by 2025. This equates to roughly 2.2 billion plastic items. The new commitment is part of the supermarket chain’s plastic reduction strategy, initially published in 2018. The","content":"\u003cp\u003eAldi has committed to removing 74,000 tonnes of plastic packaging from its stores by 2025. This equates to roughly 2.2 billion plastic items. The new commitment is part of the supermarket chain’s plastic reduction strategy, initially published in 2018. The strategy includes a ten-part pledge to reduce its plastic use. As part of the plan, […]\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co/aldi-remove-2-billion-plastic-2025/\"\u003eAldi Will Remove 2.2 Billion Plastic Items From Stores By 2025\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co\"\u003eLIVEKINDLY\u003c/a\u003e.\u003c/p\u003e\n","url":"https://www.livekindly.co/aldi-remove-2-billion-plastic-2025/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/48bb916572844a05f5786f8797e1c18b10539dec?base64_url=aHR0cHM6Ly93d3cubGl2ZWtpbmRseS5jby9hbGRpLXJlbW92ZS0yLWJpbGxpb24tcGxhc3RpYy0yMDI1Lw==","published":"2020-07-18T08:37:59.000000Z","created_at":"2020-07-18T09:11:35.890221Z"},{"id":2519213687,"feed_id":1342834,"title":"Elektroautos: Audi will exklusives Schnelllade-Netz schaffen","author":"Andreas Donath","summary":"Ohne Lade-In­fra­struk­tur lassen sich Elektroautos nur schwer verkaufen, meint Audi, und plant deshalb ein Netz exklusiv für die eigenen Kunden. (Ladesäule, Technologie)","content":"\u003cimg src=\"https://www.golem.de/2007/149730-237185-237182_rc.jpg\" width=\"140\" height=\"140\" vspace=\"3\" hspace=\"8\" align=\"left\"\u003eOhne Lade-In­fra­struk­tur lassen sich Elektroautos nur schwer verkaufen, meint Audi, und plant deshalb ein Netz exklusiv für die eigenen Kunden. (\u003ca href=\"https://www.golem.de/specials/ladesaeule/\"\u003eLadesäule\u003c/a\u003e, \u003ca href=\"https://www.golem.de/specials/technologie/\"\u003eTechnologie\u003c/a\u003e) \u003cimg src=\"https://cpx.golem.de/cpx.php?class=17\u0026amp;aid=149730\u0026amp;page=1\u0026amp;ts=1595064900\" alt=\"\" width=\"1\" height=\"1\"\u003e","url":"https://www.golem.de/news/elektroautos-audi-will-exklusives-schnelllade-netz-schaffen-2007-149730-rss.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/8fa2de4841637c44a21f3e483d667bb59d93ec04?base64_url=aHR0cHM6Ly93d3cuZ29sZW0uZGUvbmV3cy9lbGVrdHJvYXV0b3MtYXVkaS13aWxsLWV4a2x1c2l2ZXMtc2NobmVsbGxhZGUtbmV0ei1zY2hhZmZlbi0yMDA3LTE0OTczMC1yc3MuaHRtbA==","published":"2020-07-18T09:35:00.000000Z","created_at":"2020-07-18T09:44:30.481949Z"},{"id":2519214790,"feed_id":1466657,"title":"Vegan Cauliflower Mashed Potatoes Make the Perfect Side","author":"Vancouver With Love","summary":"Craving the perfect dinner side? Look no further than these vegan cauliflower mashed potatoes. I love nothing more than making cozy, delicious meals. There’s something so soul-nourishing about a simple bowl of mashed potatoes. They require minimal","content":"\u003cp\u003eCraving the perfect dinner side? Look no further than these vegan cauliflower mashed potatoes. I love nothing more than making cozy, delicious meals. There’s something so soul-nourishing about a simple bowl of mashed potatoes. They require minimal preparation and fuss and therefore make the perfect accompaniment to any meal. And, let’s be honest, they make […]\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co/vegan-cauliflower-mashed-potatoes/\"\u003eVegan Cauliflower Mashed Potatoes Make the Perfect Side\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co\"\u003eLIVEKINDLY\u003c/a\u003e.\u003c/p\u003e\n","url":"https://www.livekindly.co/vegan-cauliflower-mashed-potatoes/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/d6a5fd834960f72c3e810e836353e0b0d35e51b5?base64_url=aHR0cHM6Ly93d3cubGl2ZWtpbmRseS5jby92ZWdhbi1jYXVsaWZsb3dlci1tYXNoZWQtcG90YXRvZXMv","published":"2020-07-18T09:20:27.000000Z","created_at":"2020-07-18T09:45:46.429539Z"},{"id":2519241496,"feed_id":1848912,"title":"Here's why your Samsung Blu-ray player bricked itself: It downloaded an XML config file that broke the firmware","author":"Thomas Claburn","summary":"Network-connected gear stuck in boot loop needs replacing Analysis Since the middle of last month, thousands of Samsung customers found their older internet-connected Blu-ray players had stopped working.…","content":"\u003ch4\u003eNetwork-connected gear stuck in boot loop needs replacing\u003c/h4\u003e \u003cp\u003e\u003cstrong\u003eAnalysis\u003c/strong\u003e  Since the middle of last month, thousands of Samsung customers found their older internet-connected Blu-ray players had stopped working.…\u003c/p\u003e","url":"https://go.theregister.com/feed/www.theregister.com/2020/07/18/samsung_bluray_mass_dieoff_explained/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/95382bf4267f38b50ff1c0573c714ea5bef69661?base64_url=aHR0cHM6Ly9nby50aGVyZWdpc3Rlci5jb20vZmVlZC93d3cudGhlcmVnaXN0ZXIuY29tLzIwMjAvMDcvMTgvc2Ftc3VuZ19ibHVyYXlfbWFzc19kaWVvZmZfZXhwbGFpbmVkLw==","published":"2020-07-18T10:06:05.000000Z","created_at":"2020-07-18T10:21:34.910840Z"},{"id":2519288861,"feed_id":29836,"title":"Der Bürgermeister von Portland ruft Trump auf, seine Truppen aus seiner Stadt zu entfernen. Wait, what?Nun, was passiert war: Trump hat aus Wahlkampfzwecken \"der Antifa\" den Krieg erklärt. Aus seiner Sicht sind das Linksradikale, also hat er sich Portland in Oregon genommen, eine Stadt, die so hippie-links ist, dass es eine erfolgreiche Satire-Fernsehshow darüber gibt (Portlandia). Und dann hat er da schwarze Kleinbusse mit Söldnern hingeschickt, angeblich um Bundesgebäude zu beschützen. Die sind dann rumgefahren und haben auf offener Straße Leute eingesackt und verschwinden lassen.Glaubt ihr nicht? Seht selbst:The mayor of Portland demanded Friday that President Donald Trump remove militarized federal agents he deployed to the city after some detained people on streets far from federal property they were sent to protect.[…]Brown’s spokesman, Charles Boyle, said Friday that arresting people without probable cause is “extraordinarily concerning and a violation of their civil liberties and constitutional rights.”Well duh!Hier berichtet ein Betroffener in der Washington Post.","author":null,"summary":null,"content":null,"url":"http://blog.fefe.de/?ts=a1ec1cc4","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/99789aa13fafa90b9a7d3f60729e7afbcb31869a?base64_url=aHR0cDovL2Jsb2cuZmVmZS5kZS8_dHM9YTFlYzFjYzQ=","published":"2020-07-18T11:33:20.084507Z","created_at":"2020-07-18T11:33:20.084489Z"},{"id":2519288863,"feed_id":29836,"title":"Es gab mal wieder einen weltweiten Ausfall bei Cloudflare. Mal wieder weil sie was kaputtkonfiguriert hatten.Was machen die noch gleich beruflich?Oh ja richtig, Router konfigurieren, um die Ausfallsicherheit zu erhöhen!1!!Mein Blog ist übrigens ganz unausfallsicher auf einem Rechner bei einem normale Hoster gehostet. Und wisst ihr, was passiert, wenn ihr das von weit weg lesen wollt? China oder Amerika oder so? Es geht einfach. Ganz ohne verteiltes Caching. Flutscht.","author":null,"summary":null,"content":null,"url":"http://blog.fefe.de/?ts=a1ec1ba9","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/716cd737fca3ea5a1bccde48ca91ea4d955f0ce7?base64_url=aHR0cDovL2Jsb2cuZmVmZS5kZS8_dHM9YTFlYzFiYTk=","published":"2020-07-18T11:33:20.224346Z","created_at":"2020-07-18T11:33:20.224325Z"},{"id":2519346401,"feed_id":1735,"title":"Top Stories: Apple Warns About MacBook Webcam Covers, iOS 13.6 Released, Apple Hacked on Twitter","author":"MacRumors Staff","summary":"While much of our attention has been focused on iOS 14 over the past weeks, Apple was finishing up some final updates for the iOS 13 series of releases, and this week delivered iOS 13.6 and associated updates for other platforms to the public. New iOS","content":"While much of our attention has been focused on iOS 14 over the past weeks, Apple was finishing up some final updates for the iOS 13 series of releases, and this week delivered iOS 13.6 and associated updates for other platforms to the public. New iOS features include audio stories for Apple News+, support for digital car keys, and more.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ccenter\u003e\u003ciframe width=\"560\" height=\"315\" src=\"https://www.youtube.com/embed/PtNbtQLpZVY\" frameborder=\"0\" allow=\"accelerometer; autoplay; encrypted-media; gyroscope; picture-in-picture\" allowfullscreen\u003e\u003c/iframe\u003e\u003c/center\u003e\u003ccenter\u003e\u003cem\u003e\u003ca href=\"https://www.youtube.com/user/macrumors?sub_confirmation=1\"\u003eSubscribe to the MacRumors YouTube channel\u003c/a\u003e for more videos.\u003c/em\u003e\u003c/center\u003e\n\u003cbr\u003e\nOne incredibly popular story (and in fact one of our most popular stories in years) was about a new Apple support document warning users to be careful about installing physical webcam covers on their Mac notebooks. Other top stories included a major Twitter security breach that saw Apple's official account tweet out a Bitcoin scam, the opening of filing for cash payments in the iPhone \"Batterygate\" lawsuit, and more.\n\u003cbr\u003e\n\n\u003cbr\u003e\nCheck out our video above and read on below for more on all of this week's biggest stories!\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2\u003eApple Warns Against Closing MacBooks With a Cover Over the Camera\u003c/h2\u003e\n\u003cbr\u003e\nIn a new support document this month, Apple has \u003ca href=\"https://www.macrumors.com/2020/07/10/apple-macbook-camera-cover-warning/\"\u003eadvised customers not to close the lid of a MacBook, MacBook Air, or MacBook Pro with a cover over the webcam\u003c/a\u003e, noting that this practice may cause damage to the display and might also interfere with the ambient light sensor and prevent features like automatic brightness and True Tone from working.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/16MBPCrackedScreenCameraCoverFeature-2.jpg\" alt=\"\" width=\"1769\" height=\"996\" class=\"aligncenter size-full wp-image-749941\"\u003e\n\u003cbr\u003e\nApple said that customers should instead look for a green indicator light to know when the webcam is active, and decide which apps can use the camera in System Preferences. Apple ensures that the camera is engineered so that it can’t activate without the light also turning on.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2\u003eApple Releases iOS 13.6 With Car Key, Apple News+ Audio, and More\u003c/h2\u003e\n\u003cbr\u003e\nApple this week \u003ca href=\"https://www.macrumors.com/2020/07/15/apple-releases-ios-13-6/\"\u003epublicly released iOS 13.6\u003c/a\u003e with a handful of new features, including support for digital car keys in the Wallet app, \u003ca href=\"https://www.macrumors.com/2020/07/15/apple-launches-audio-stories-apple-news/\"\u003eaudio stories for Apple News+ subscribers\u003c/a\u003e, and a toggle switch for enabling or disabling the automatic downloading of iOS updates.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/06/iOS-13.6-beta-feature.jpg\" alt=\"\" width=\"2870\" height=\"1614\" class=\"aligncenter size-full wp-image-748609\"\u003e\n\u003cbr\u003e\nAt launch, the Car Key feature is limited to select BMW models manufactured after July 1, 2020, including the 1, 2, 3, 4, 5, 6, 8, X5, X6, X7, X5M, X6M and Z4. \n\u003cbr\u003e\n\n\u003cbr\u003e\nApple also publicly released macOS 10.15.6, watchOS 6.2.8, tvOS 13.4.8, and a HomePod update.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2\u003eApple's Twitter Account Hacked by Bitcoin Scammers\u003c/h2\u003e\n\u003cbr\u003e\nApple, Uber, Bill Gates, Elon Musk, Barack Obama, Joe Biden, Kim Kardashian, and Kanye West were just some of the high-profile brands and individuals \u003ca href=\"https://www.macrumors.com/2020/07/15/apple-twitter-account-hacked/\"\u003einvolved in a major Twitter security breach this week\u003c/a\u003e.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/apple_bitcoin_hack.jpg\" alt=\"\" width=\"1178\" height=\"876\" class=\"aligncenter size-full wp-image-750451\"\u003e\n\u003cbr\u003e\nAll of the compromised Twitter accounts tweeted out a similar message that claimed users could send Bitcoin to a certain address and receive double the amount in return. The hackers appear to have received more than $100,000 worth of cryptocurrency before Twitter took action.\n\u003cbr\u003e\n\n\u003cbr\u003e\nTwitter \u003ca href=\"https://www.macrumors.com/2020/07/17/twitter-hackers-access-accounts-internal-tool/\"\u003econtinues to investigate the breach\u003c/a\u003e, which it described as a \"coordinated social engineering attack by people who successfully targeted some of our employees with access to internal systems and tools.\"\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2\u003eiPhone Users Can Now File to Receive Around $25 From Apple Over 'Batterygate'\u003c/h2\u003e\n\u003cbr\u003e\nEarlier this year, Apple \u003ca href=\"https://www.macrumors.com/2020/03/02/apple-agrees-to-up-to-500m-batterygate-settlement/\"\u003eagreed to settle a U.S. class action lawsuit\u003c/a\u003e that accused the company of \"secretly throttling\" older iPhone models, and affected customers can now claim their share of the settlement.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/batterygate-feature.jpg\" alt=\"\" width=\"1920\" height=\"1080\" class=\"aligncenter size-full wp-image-750189\"\u003e\n\u003cbr\u003e\n\u003ca href=\"https://www.macrumors.com/2020/07/13/iphone-slowdown-lawsuit-notification/\"\u003eApple will provide a cash payment of approximately $25 to each eligible iPhone owner\u003c/a\u003e who submits a claim, with its total payout to fall between $310 million and $500 million. The exact amount that each iPhone owner receives could vary slightly based on the number of claims submitted.\n\u003cbr\u003e\n\n\u003cbr\u003e\nThe class includes any U.S. resident who owns or previously owned an iPhone 6, iPhone 6 Plus, iPhone 6s, iPhone 6s Plus, and/or iPhone SE that ran iOS 10.2.1 or later, and/or an iPhone 7 or iPhone 7 Plus that ran iOS 11.2 or later, before December 21, 2017. Class members must have experienced \"diminished performance\" on their devices in order to qualify.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch2\u003eFive Mac Apps Worth Checking Out - July 2020\u003c/h2\u003e\n\u003cbr\u003e\nEach month, we \u003ca href=\"https://www.macrumors.com/2020/07/14/five-mac-apps-july-2020/\"\u003eshare a video highlighting a handful of Mac apps worth checking out\u003c/a\u003e. This month's selections focus on productivity and feature great tools for working from home.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003cimg src=\"https://images.macrumors.com/article-new/2020/07/5-mac-apps-7_15_20-Feature2.jpg\" alt=\"\" width=\"2108\" height=\"1186\" class=\"aligncenter size-full wp-image-750349\"\u003e\n\u003cbr\u003e\nFor more of our Mac app picks, \u003ca href=\"https://www.macrumors.com/guide/essential-mac-apps/\"\u003echeck out our essential Mac apps archive\u003c/a\u003e.\n\u003cbr\u003e\n\n\u003cbr\u003e\n\u003ch3\u003eMacRumors Newsletter\u003c/h3\u003e\n\u003cbr\u003e\nEach week, we publish an email newsletter like this highlighting the top Apple stories, making it a great way to get a bite-sized recap of the week hitting all of the major topics we've covered and tying together related stories for a big-picture view.\n\u003cbr\u003e\n\n\u003cbr\u003e\nSo if you want to have \u003ca href=\"https://mailchi.mp/macrumors/jul-9-2020\"\u003etop stories\u003c/a\u003e like the above recap delivered to your email inbox each week, \u003cb\u003e\u003ca href=\"http://eepurl.com/OHdUX\"\u003esubscribe to our newsletter\u003c/a\u003e\u003c/b\u003e!\u003cdiv class=\"linkback\"\u003eTag: \u003ca href=\"https://www.macrumors.com/guide/top-stories/\"\u003eTop Stories\u003c/a\u003e\n\u003c/div\u003e\u003cbr\u003eThis article, \"\u003ca href=\"https://www.macrumors.com/2020/07/18/top-stories-webcam-covers-ios-13-6/\"\u003eTop Stories: Apple Warns About MacBook Webcam Covers, iOS 13.6 Released, Apple Hacked on Twitter\u003c/a\u003e\" first appeared on \u003ca href=\"https://www.macrumors.com\"\u003eMacRumors.com\u003c/a\u003e\u003cbr\u003e\u003cbr\u003e\u003ca href=\"https://forums.macrumors.com/threads/top-stories-apple-warns-about-macbook-webcam-covers-ios-13-6-released-apple-hacked-on-twitter.2246409/\"\u003eDiscuss this article\u003c/a\u003e in our forums\u003cbr\u003e\u003cbr\u003e\u003cdiv class=\"feedflare\"\u003e\n\u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=YVCZnlCz12o:tle8SQiVLoY:6W8y8wAjSf4\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=6W8y8wAjSf4\" border=\"0\"\u003e\u003c/a\u003e \u003ca href=\"http://feeds.macrumors.com/~ff/MacRumors-All?a=YVCZnlCz12o:tle8SQiVLoY:qj6IDK7rITs\"\u003e\u003cimg src=\"http://feeds.feedburner.com/~ff/MacRumors-All?d=qj6IDK7rITs\" border=\"0\"\u003e\u003c/a\u003e\n\u003c/div\u003e\u003cimg src=\"http://feeds.feedburner.com/~r/MacRumors-All/~4/YVCZnlCz12o\" height=\"1\" width=\"1\" alt=\"\"\u003e","url":"https://www.macrumors.com/2020/07/18/top-stories-webcam-covers-ios-13-6/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/6a48b75fa3fb17b355aa039b16b93130516fcf3b?base64_url=aHR0cHM6Ly93d3cubWFjcnVtb3JzLmNvbS8yMDIwLzA3LzE4L3RvcC1zdG9yaWVzLXdlYmNhbS1jb3ZlcnMtaW9zLTEzLTYv","published":"2020-07-18T13:00:19.000000Z","created_at":"2020-07-18T13:02:05.409297Z"},{"id":2519351296,"feed_id":1178507,"title":"Apple @ Work: What are Apple’s plans for MDM now that it owns Fleetsmith?","author":"Bradley Chambers","summary":"For enterprise customers who use Apple, the week of WWDC brought some rather shocking news. Late in the week, first-party mobile device management (MDM) solution Fleetsmith announced Apple had purchased them for an undisclosed sum. There was a bit of","content":"\u003cdiv class=\"feat-image\"\u003e\u003cimg src=\"https://9to5mac.com/wp-content/uploads/sites/6/2020/06/Fleetsmith.jpg?quality=82\u0026amp;strip=all\u0026amp;w=1600\"\u003e\u003c/div\u003e\n\u003cp\u003eFor enterprise customers who use Apple, the week of \u003ca href=\"https://9to5mac.com/2020/06/28/wwdc-2020-ios-14-wrap-up/\"\u003eWWDC\u003c/a\u003e brought some rather shocking news. Late in the week, first-party mobile device management (MDM) solution Fleetsmith \u003ca href=\"https://9to5mac.com/2020/06/24/apple-acquires-fleetsmith-to-enhance-iphone-ipad-and-mac-enterprise-features/\"\u003eannounced\u003c/a\u003e Apple had purchased them for an undisclosed sum. There was a bit of controversy with the purchase, as Apple immediately discontinued Fleetsmith’s App Catalog without warning. This left a lot of customers in a bind as soon as they woke up that morning.\u003c/p\u003e\n\u003cp\u003eWhile the controversy has died off, I know many customers of Fleetsmith, Jamf, and others are wondering what Apple’s plans are now. Since the announcement, I’ve been thinking about what Apple might do in the future now that they own an MDM. Here are some ideas I came up with on what might happen in the future. \u003ca href=\"https://9to5mac.com/2020/07/18/apple-mdm/#more-655146\" class=\"more-link\"\u003emore…\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://9to5mac.com/2020/07/18/apple-mdm/\"\u003eApple @ Work: What are Apple’s plans for MDM now that it owns Fleetsmith?\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://9to5mac.com\"\u003e9to5Mac\u003c/a\u003e.\u003c/p\u003e\n","url":"https://9to5mac.com/2020/07/18/apple-mdm/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/cb93f3ff2afd28064c501040e12842acd2f28012?base64_url=aHR0cHM6Ly85dG81bWFjLmNvbS8yMDIwLzA3LzE4L2FwcGxlLW1kbS8=","published":"2020-07-18T13:00:05.000000Z","created_at":"2020-07-18T13:09:17.473033Z"},{"id":2519355321,"feed_id":1545349,"title":"CSD Leipzig: \"Online war richtige Entscheidung\"","author":null,"summary":"(Szene - Deutschland) Trotz Corona-Krise ziehen die Veranstalter*innen eine positive Bilanz der CSD-Aktionswoche in Leipzig, die heute zu Ende geht. Überschattet wurde sie von einem Einschüchterungsversuch von Neonazis. lesen","content":"\u003cimg src=\"https://www.queer.de/publisher/pics/du-bist-nicht-allein-social.jpg\" align=\"right\" style=\"margin-left: 5px;\"\u003e (Szene - Deutschland) Trotz Corona-Krise ziehen die Veranstalter*innen eine positive Bilanz der CSD-Aktionswoche in Leipzig, die heute zu Ende geht. Überschattet wurde sie von einem Einschüchterungsversuch von Neonazis. \u003ca href=\"https://www.queer.de/detail.php?article_id=36618\"\u003elesen\u003c/a\u003e","url":"https://www.queer.de/detail.php?article_id=36618","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/19c55c64609f3531e8e2b8e0c685cfbe8b19c658?base64_url=aHR0cHM6Ly93d3cucXVlZXIuZGUvZGV0YWlsLnBocD9hcnRpY2xlX2lkPTM2NjE4","published":"2020-07-18T12:47:36.000000Z","created_at":"2020-07-18T13:13:04.577598Z"},{"id":2519367118,"feed_id":1342834,"title":"Super 35 Sensor: 12K-Videos mit Blackmagic Ursa Mini Pro","author":"Andreas Donath","summary":"Derzeit gibt es kaum 8K-Fernseher, da bringt Blackmagic schon eine 12K-Videokamera namens Ursa Mini Pro auf den Markt. (Camcorder, Canon)","content":"\u003cimg src=\"https://www.golem.de/2007/149734-237201-237198_rc.jpg\" width=\"140\" height=\"140\" vspace=\"3\" hspace=\"8\" align=\"left\"\u003eDerzeit gibt es kaum 8K-Fernseher, da bringt Blackmagic schon eine 12K-Videokamera namens Ursa Mini Pro auf den Markt. (\u003ca href=\"https://www.golem.de/specials/camcorder/\"\u003eCamcorder\u003c/a\u003e, \u003ca href=\"https://www.golem.de/specials/canon/\"\u003eCanon\u003c/a\u003e) \u003cimg src=\"https://cpx.golem.de/cpx.php?class=17\u0026amp;aid=149734\u0026amp;page=1\u0026amp;ts=1595078340\" alt=\"\" width=\"1\" height=\"1\"\u003e","url":"https://www.golem.de/news/super-35-sensor-12k-videos-mit-blackmagic-ursa-mini-pro-2007-149734-rss.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/380c4601d61a7b513f9a017f193b9f3334e6c33c?base64_url=aHR0cHM6Ly93d3cuZ29sZW0uZGUvbmV3cy9zdXBlci0zNS1zZW5zb3ItMTJrLXZpZGVvcy1taXQtYmxhY2ttYWdpYy11cnNhLW1pbmktcHJvLTIwMDctMTQ5NzM0LXJzcy5odG1s","published":"2020-07-18T13:19:00.000000Z","created_at":"2020-07-18T13:25:33.451383Z"},{"id":2519453562,"feed_id":1178507,"title":"How much is your iPhone 8 / 8 Plus worth right now?","author":"Stephen Hall","summary":"So you have an iPhone 8 or 8 Plus laying around, and you’re wondering what it’s worth. Great, you’ve come to the right place! You might be surprised how simple it is to ditch your old iPhone 8 or 8 Plus and get on with updating to the latest iPhone. But","content":"\u003cdiv class=\"feat-image\"\u003e\u003cimg src=\"https://9to5mac.com/wp-content/uploads/sites/6/2019/04/iphone-8-1.png?w=1544\"\u003e\u003c/div\u003e\n\u003cp\u003eSo you have an iPhone 8 or 8 Plus laying around, and you’re wondering what it’s worth. Great, you’ve come to the right place! You might be surprised how simple it is to ditch your old iPhone 8 or 8 Plus and get on with updating to the latest iPhone. But first, let’s find out how much it’s worth and how to get the most value for your trade.\u003c/p\u003e\n\u003cp\u003e \u003ca href=\"https://9to5mac.com/2020/07/18/iphone-8-trade-in-value/#more-598546\" class=\"more-link\"\u003emore…\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://9to5mac.com/2020/07/18/iphone-8-trade-in-value/\"\u003eHow much is your iPhone 8 / 8 Plus worth right now?\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://9to5mac.com\"\u003e9to5Mac\u003c/a\u003e.\u003c/p\u003e\n","url":"https://9to5mac.com/2020/07/18/iphone-8-trade-in-value/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/487f37ef075ab4b41de24f4d3254e35b231e0c20?base64_url=aHR0cHM6Ly85dG81bWFjLmNvbS8yMDIwLzA3LzE4L2lwaG9uZS04LXRyYWRlLWluLXZhbHVlLw==","published":"2020-07-18T07:06:21.000000Z","created_at":"2020-07-18T15:22:36.897795Z"},{"id":2519495250,"feed_id":1178507,"title":"Apple shares new ‘Behind the Mac’ videos with musician James Blake, photographer Tyler Mitchell","author":"Chance Miller","summary":"Apple shared a pair of new videos on its YouTube channel today focusing on how creatives incorporate the Mac into their workflow. One video focuses on musician James Blake, while the other highlights photographer Tyler Mitchell. more… The post Apple shares","content":"\u003cdiv class=\"feat-image\"\u003e\u003cimg src=\"https://9to5mac.com/wp-content/uploads/sites/6/2020/07/maxresdefault-9.jpg?quality=82\u0026amp;strip=all\u0026amp;w=1280\"\u003e\u003c/div\u003e\n\u003cp\u003eApple shared a pair of new videos on its YouTube channel today focusing on how creatives incorporate the Mac into their workflow. One video focuses on musician James Blake, while the other highlights photographer Tyler Mitchell.\u003c/p\u003e\n\u003cp\u003e \u003ca href=\"https://9to5mac.com/2020/07/18/apple-behind-the-mac-james-blake-tyler-mitchell/#more-657071\" class=\"more-link\"\u003emore…\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://9to5mac.com/2020/07/18/apple-behind-the-mac-james-blake-tyler-mitchell/\"\u003eApple shares new ‘Behind the Mac’ videos with musician James Blake, photographer Tyler Mitchell\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://9to5mac.com\"\u003e9to5Mac\u003c/a\u003e.\u003c/p\u003e\n","url":"https://9to5mac.com/2020/07/18/apple-behind-the-mac-james-blake-tyler-mitchell/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/8535c4489bde984244a8769a5bd9c2ccb1647bc0?base64_url=aHR0cHM6Ly85dG81bWFjLmNvbS8yMDIwLzA3LzE4L2FwcGxlLWJlaGluZC10aGUtbWFjLWphbWVzLWJsYWtlLXR5bGVyLW1pdGNoZWxsLw==","published":"2020-07-18T16:14:44.000000Z","created_at":"2020-07-18T16:15:41.504890Z"},{"id":2519495255,"feed_id":1178507,"title":"Twitter testing support for customizable app icons on iOS, new splash screen design","author":"Chance Miller","summary":"Twitter appears to be considering offering users additional choices for the Twitter app icon on iOS home screens. The company has sent a new survey to Twitter Insiders program members to gather feedback on new icon designs. more… The post Twitter testing","content":"\u003cdiv class=\"feat-image\"\u003e\u003cimg src=\"https://9to5mac.com/wp-content/uploads/sites/6/2020/07/twitter-app-icons.jpg?quality=82\u0026amp;strip=all\u0026amp;w=1600\"\u003e\u003c/div\u003e\n\u003cp\u003eTwitter appears to be considering offering users additional choices for the Twitter app icon on iOS home screens. The company has sent a new survey to Twitter Insiders program members to gather feedback on new icon designs.\u003c/p\u003e\n\u003cp\u003e \u003ca href=\"https://9to5mac.com/2020/07/18/twitter-testing-support-for-customizable-app-icons-on-ios-new-splash-screen-design/#more-657066\" class=\"more-link\"\u003emore…\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://9to5mac.com/2020/07/18/twitter-testing-support-for-customizable-app-icons-on-ios-new-splash-screen-design/\"\u003eTwitter testing support for customizable app icons on iOS, new splash screen design\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://9to5mac.com\"\u003e9to5Mac\u003c/a\u003e.\u003c/p\u003e\n","url":"https://9to5mac.com/2020/07/18/twitter-testing-support-for-customizable-app-icons-on-ios-new-splash-screen-design/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/c2e3d565b82b874222f0f100eff2ea55c7eaa119?base64_url=aHR0cHM6Ly85dG81bWFjLmNvbS8yMDIwLzA3LzE4L3R3aXR0ZXItdGVzdGluZy1zdXBwb3J0LWZvci1jdXN0b21pemFibGUtYXBwLWljb25zLW9uLWlvcy1uZXctc3BsYXNoLXNjcmVlbi1kZXNpZ24v","published":"2020-07-18T16:02:30.000000Z","created_at":"2020-07-18T16:15:41.733153Z"},{"id":2519786097,"feed_id":197252,"title":"pics/oreally/kubernetes.png","author":null,"summary":"","content":"\u003cimg src=\"http://fun.drno.de/pics/oreally/kubernetes.png\"\u003e","url":"http://fun.drno.de/pics/oreally/kubernetes.png","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/1ae8ae71a43573886a4aa5cb3ba97d831219b490?base64_url=aHR0cDovL2Z1bi5kcm5vLmRlL3BpY3Mvb3JlYWxseS9rdWJlcm5ldGVzLnBuZw==","published":"2020-07-18T23:42:13.527644Z","created_at":"2020-07-18T23:42:13.527619Z"},{"id":2519821335,"feed_id":1296379,"title":"‘Inside Trump’s Failure: The Rush to Abandon Leadership Role on the Virus’","author":"John Gruber","summary":"Michael D. Shear, Noah Weiland, Eric Lipton, Maggie Haberman, and David E. Sanger, reporting for The New York Times: Each morning at 8 as the coronavirus crisis was raging in April, Mark Meadows, the White House chief of staff, convened a small group of","content":"\n\u003cp\u003eMichael D. Shear, Noah Weiland, Eric Lipton, Maggie Haberman, and David E. Sanger, reporting for The New York Times:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eEach morning at 8 as the coronavirus crisis was raging in April,\nMark Meadows, the White House chief of staff, convened a small\ngroup of aides to steer the administration through what had become\na public health, economic and political disaster.\u003c/p\u003e\n\n\u003cp\u003eSeated around Mr. Meadows’s conference table and on a couch in his\noffice down the hall from the Oval Office, they saw their\nimmediate role as practical problem solvers. Produce more\nventilators. Find more personal protective equipment. Provide more\ntesting.\u003c/p\u003e\n\n\u003cp\u003eBut their ultimate goal was to shift responsibility for leading\nthe fight against the pandemic from the White House to the states.\nThey referred to this as “state authority handoff,” and it was at\nthe heart of what would become at once a catastrophic policy\nblunder and an attempt to escape blame for a crisis that had\nengulfed the country — perhaps one of the greatest failures of\npresidential leadership in generations.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eAbsolutely riveting — and infuriating, and terrifying — account of how the U.S. response went so terribly wrong. Devastating assessment of Jared Kushner (no surprise) and Dr. Deborah Birx (a bit surprising).\u003c/p\u003e\n\n\u003cp style=\"padding-top: 1.5em;\"\u003e\u003cem\u003eLink: \u003cstrong\u003e\u003ca href=\"https://www.nytimes.com/2020/07/18/us/politics/trump-coronavirus-response-failure-leadership.html\"\u003enytimes.com/2020/07/18/us/politics/trump-coronavirus…\u003c/a\u003e\u003c/strong\u003e\u003c/em\u003e\u003c/p\u003e\n","url":"https://daringfireball.net/linked/2020/07/18/trumps-failure","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/e98365497549c5ced6c05e94cb177e124eeb3e72?base64_url=aHR0cHM6Ly9kYXJpbmdmaXJlYmFsbC5uZXQvbGlua2VkLzIwMjAvMDcvMTgvdHJ1bXBzLWZhaWx1cmU=","published":"2020-07-19T00:40:11.000000Z","created_at":"2020-07-19T00:47:07.843735Z"},{"id":2519934940,"feed_id":1545349,"title":"\"Taste History\" in Malta","author":null,"summary":"(Reise - Europaziele) Geschichte für den Gaumen zum Nachkochen: In vier Kurzfilmen, gedreht an historisch bedeutsamen Orten, haben Heritage Malta und VisitMalta authentische Rezepte des Landes rekonstruiert. lesen","content":"\u003cimg src=\"https://www.queer.de/publisher/pics/taste-history-malta-social.jpg\" align=\"right\" style=\"margin-left: 5px;\"\u003e (Reise - Europaziele) Geschichte für den Gaumen zum Nachkochen: In vier Kurzfilmen, gedreht an historisch bedeutsamen Orten, haben Heritage Malta und VisitMalta authentische Rezepte des Landes rekonstruiert. \u003ca href=\"https://www.queer.de/detail.php?article_id=36619\"\u003elesen\u003c/a\u003e","url":"https://www.queer.de/detail.php?article_id=36619","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/c7dce8c5c33d4a160e505e5f3f56f62dd5272976?base64_url=aHR0cHM6Ly93d3cucXVlZXIuZGUvZGV0YWlsLnBocD9hcnRpY2xlX2lkPTM2NjE5","published":"2020-07-19T04:03:49.000000Z","created_at":"2020-07-19T04:30:51.252859Z"},{"id":2519952097,"feed_id":1545349,"title":"\"Find Me\": Was aus Elio und Oliver wird","author":null,"summary":"(Buch - Literatur) André Aciman schreibt die Geschichte von Elio und Oliver aus \"Call Me By Your Name\" weiter. Natürlich wollen alle wissen, wie es mit den beiden weitergeht. Doch viele Fans werden vom Roman enttäuscht sein. lesen","content":"\u003cimg src=\"https://www.queer.de/publisher/pics/call-me-by-your-name-neu-social.jpg\" align=\"right\" style=\"margin-left: 5px;\"\u003e (Buch - Literatur) André Aciman schreibt die Geschichte von Elio und Oliver aus \"Call Me By Your Name\" weiter. Natürlich wollen alle wissen, wie es mit den beiden weitergeht. Doch viele Fans werden vom Roman enttäuscht sein. \u003ca href=\"https://www.queer.de/detail.php?article_id=36620\"\u003elesen\u003c/a\u003e","url":"https://www.queer.de/detail.php?article_id=36620","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/a5cf27f8045ba4b48fb2cb2f940aafbf34ebfb70?base64_url=aHR0cHM6Ly93d3cucXVlZXIuZGUvZGV0YWlsLnBocD9hcnRpY2xlX2lkPTM2NjIw","published":"2020-07-19T04:45:22.000000Z","created_at":"2020-07-19T05:03:14.708982Z"},{"id":2520061986,"feed_id":1545349,"title":"Gewinnspiel: Ellie Goulding","author":null,"summary":"(Gewinnspiel) Wir verlosen fünfmal das neue Album \"Brightest Blue\" der britischen Sängerin. lesen","content":"\u003cimg src=\"https://www.queer.de/publisher/pics/ellie-goulding-blackbear-2020-social.jpg\" align=\"right\" style=\"margin-left: 5px;\"\u003e (Gewinnspiel) Wir verlosen fünfmal das neue Album \"Brightest Blue\" der britischen Sängerin. \u003ca href=\"https://www.queer.de/detail.php?article_id=36621\"\u003elesen\u003c/a\u003e","url":"https://www.queer.de/detail.php?article_id=36621","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/c742796dc8a8e0df837534cac4b17e931ca486e8?base64_url=aHR0cHM6Ly93d3cucXVlZXIuZGUvZGV0YWlsLnBocD9hcnRpY2xlX2lkPTM2NjIx","published":"2020-07-19T07:26:21.000000Z","created_at":"2020-07-19T08:11:47.195420Z"},{"id":2520118118,"feed_id":1848912,"title":"Incredible artifact – or vital component after civilization ends? Rare Nazi Enigma M4 box sells for £350,000","author":"Shaun Nichols","summary":"You're gonna need something to secure those Doordash orders over Morse code A Second World War cryptography artifact – a 1944 Enigma M4 machine – has sold at auction for £347,250 ($436,000).…","content":"\u003ch4\u003eYou're gonna need something to secure those Doordash orders over Morse code\u003c/h4\u003e \u003cp\u003eA Second World War cryptography artifact – a 1944 Enigma M4 machine – has sold at auction for £347,250 ($436,000).…\u003c/p\u003e","url":"https://go.theregister.com/feed/www.theregister.com/2020/07/19/enigma_m4_auction/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/bf5949f9c3bd68b36a6a7be29756357ed79af0f8?base64_url=aHR0cHM6Ly9nby50aGVyZWdpc3Rlci5jb20vZmVlZC93d3cudGhlcmVnaXN0ZXIuY29tLzIwMjAvMDcvMTkvZW5pZ21hX200X2F1Y3Rpb24v","published":"2020-07-19T09:40:13.000000Z","created_at":"2020-07-19T09:45:20.877069Z"},{"id":2520146925,"feed_id":1342834,"title":"Zwangsgeld: Telefónica gibt Fehler beim Netzausbau zu","author":"Achim Sawall","summary":"Nach einer weiteren Fristverlängerung schafft die Telefónica nun wohl endlich die Vorgaben im LTE-Ausbau. Ein Zwangsgeld will Unternehmenschef Markus Haas unbedingt vermeiden. (Telefónica, Long Term Evolution)","content":"\u003cimg src=\"https://www.golem.de/2007/149735-237205-237202_rc.jpg\" width=\"140\" height=\"140\" vspace=\"3\" hspace=\"8\" align=\"left\"\u003eNach einer weiteren Fristverlängerung schafft die Telefónica nun wohl endlich die Vorgaben im LTE-Ausbau. Ein Zwangsgeld will Unternehmenschef Markus Haas unbedingt vermeiden. (\u003ca href=\"https://www.golem.de/specials/telefonica/\"\u003eTelefónica\u003c/a\u003e, \u003ca href=\"https://www.golem.de/specials/lte/\"\u003eLong Term Evolution\u003c/a\u003e) \u003cimg src=\"https://cpx.golem.de/cpx.php?class=17\u0026amp;aid=149735\u0026amp;page=1\u0026amp;ts=1595153460\" alt=\"\" width=\"1\" height=\"1\"\u003e","url":"https://www.golem.de/news/zwangsgeld-telefonica-gibt-fehler-beim-netzausbau-zu-2007-149735-rss.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/0c218dd7f9a7c5351d3d3cf59731db51fc33416e?base64_url=aHR0cHM6Ly93d3cuZ29sZW0uZGUvbmV3cy96d2FuZ3NnZWxkLXRlbGVmb25pY2EtZ2lidC1mZWhsZXItYmVpbS1uZXR6YXVzYmF1LXp1LTIwMDctMTQ5NzM1LXJzcy5odG1s","published":"2020-07-19T10:11:00.000000Z","created_at":"2020-07-19T10:26:47.434024Z"},{"id":2520212449,"feed_id":1545349,"title":"AfD-Politiker hetzt gegen CSD in Marzahn","author":null,"summary":"(Politik - Deutschland) Gunnar Lindemann filmte bei dem Pride, den er als \"antirussischen Rassismus\", \"Propaganda\", \"Umerziehung\" und \"Krawall\" diffamiert. lesen","content":"\u003cimg src=\"https://www.queer.de/publisher/pics/lindemann-afd-marzahn-pride-berlin-anders-social.jpg\" align=\"right\" style=\"margin-left: 5px;\"\u003e (Politik - Deutschland) Gunnar Lindemann filmte bei dem Pride, den er als \"antirussischen Rassismus\", \"Propaganda\", \"Umerziehung\" und \"Krawall\" diffamiert. \u003ca href=\"https://www.queer.de/detail.php?article_id=36623\"\u003elesen\u003c/a\u003e","url":"https://www.queer.de/detail.php?article_id=36623","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/4907718f793b79eb3f1c943703b77de7a710f9f7?base64_url=aHR0cHM6Ly93d3cucXVlZXIuZGUvZGV0YWlsLnBocD9hcnRpY2xlX2lkPTM2NjIz","published":"2020-07-19T11:49:55.000000Z","created_at":"2020-07-19T12:07:59.138279Z"},{"id":2520224016,"feed_id":1530714,"title":"heise+ | Die Sicherheitsfunktionen von Windows 10 im Überblick","author":null,"summary":"Die Sicherheit von Windows genießt keinen guten Ruf. Doch ein Blick auf die Security-Features zeigt, dass es Angreifer schwerer denn je haben.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/ratgeber/Die-Sicherheitsfunktionen-von-Windows-10-im-Ueberblick-4842251.html?wt_mc=rss.red.ho.ho.atom.beitrag_plus.beitrag_plus\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/5/3/7/0/ct1620windows_t2_andreas_martini_110280_axv_a-de0bbc2e39fd7e74.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eDie Sicherheit von Windows genießt keinen guten Ruf. Doch ein Blick auf die Security-Features zeigt, dass es Angreifer schwerer denn je haben. \u003c/p\u003e","url":"https://www.heise.de/ratgeber/Die-Sicherheitsfunktionen-von-Windows-10-im-Ueberblick-4842251.html?wt_mc=rss.red.ho.ho.atom.beitrag_plus.beitrag_plus","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/23448a8ba158233ddb83bad9efb08e86d4221168?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvcmF0Z2ViZXIvRGllLVNpY2hlcmhlaXRzZnVua3Rpb25lbi12b24tV2luZG93cy0xMC1pbS1VZWJlcmJsaWNrLTQ4NDIyNTEuaHRtbD93dF9tYz1yc3MucmVkLmhvLmhvLmF0b20uYmVpdHJhZ19wbHVzLmJlaXRyYWdfcGx1cw==","published":"2020-07-19T12:30:00.000000Z","created_at":"2020-07-19T12:31:07.455223Z"},{"id":2520238468,"feed_id":1545349,"title":"Burhan Qurbani, warum spielt ein cis Mann eine trans Frau?","author":null,"summary":"(Film \u0026 TV - Aktuell) Der Regisseur von \"Berlin Alexanderplatz\" über die Queerness seines Films, warum sein Gangster-Boss Pums schwul ist, wie die LGBTI-Repräsentanz im Kino verbessert werden kann - und eine umstrittene Besetzung. lesen","content":"\u003cimg src=\"https://www.queer.de/publisher/pics/berlin-alexanderplatz-pums-social.jpg\" align=\"right\" style=\"margin-left: 5px;\"\u003e (Film \u0026amp; TV - Aktuell) Der Regisseur von \"Berlin Alexanderplatz\" über die Queerness seines Films, warum sein Gangster-Boss Pums schwul ist, wie die LGBTI-Repräsentanz im Kino verbessert werden kann - und eine umstrittene Besetzung. \u003ca href=\"https://www.queer.de/detail.php?article_id=36624\"\u003elesen\u003c/a\u003e","url":"https://www.queer.de/detail.php?article_id=36624","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/376623f757102aa9c0423eb643364499c4f96171?base64_url=aHR0cHM6Ly93d3cucXVlZXIuZGUvZGV0YWlsLnBocD9hcnRpY2xlX2lkPTM2NjI0","published":"2020-07-19T12:25:06.000000Z","created_at":"2020-07-19T12:47:27.453264Z"},{"id":2520246916,"feed_id":1342834,"title":"Ohne Trump: Großbritannien könnte Huawei-Ausschluss zurücknehmen","author":"Achim Sawall","summary":"Sollte Donald Trump in diesem Jahr nicht wiedergewählt werden, dürfte Huawei in Großbritannien doch die Netze nach 2027 ausrüsten dürfen. (Huawei, Handy)","content":"\u003cimg src=\"https://www.golem.de/2007/149737-237209-237206_rc.jpg\" width=\"140\" height=\"140\" vspace=\"3\" hspace=\"8\" align=\"left\"\u003eSollte Donald Trump in diesem Jahr nicht wiedergewählt werden, dürfte Huawei in Großbritannien doch die Netze nach 2027 ausrüsten dürfen. (\u003ca href=\"https://www.golem.de/specials/huawei/\"\u003eHuawei\u003c/a\u003e, \u003ca href=\"https://www.golem.de/specials/handy/\"\u003eHandy\u003c/a\u003e) \u003cimg src=\"https://cpx.golem.de/cpx.php?class=17\u0026amp;aid=149737\u0026amp;page=1\u0026amp;ts=1595163300\" alt=\"\" width=\"1\" height=\"1\"\u003e","url":"https://www.golem.de/news/ohne-trump-grossbritannien-koennte-huawei-ausschluss-zuruecknehmen-2007-149737-rss.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/02f3692b0d1a33109014ea0154f8594e830b8944?base64_url=aHR0cHM6Ly93d3cuZ29sZW0uZGUvbmV3cy9vaG5lLXRydW1wLWdyb3NzYnJpdGFubmllbi1rb2VubnRlLWh1YXdlaS1hdXNzY2hsdXNzLXp1cnVlY2tuZWhtZW4tMjAwNy0xNDk3MzctcnNzLmh0bWw=","published":"2020-07-19T12:55:00.000000Z","created_at":"2020-07-19T13:03:30.758184Z"},{"id":2520272782,"feed_id":904489,"title":"The fax machine still underpins public health systems ↦","author":"Dan Moren","summary":"Sobering story from Sarah Kliff and Margot Sanger-Katz in the New York Times about one of the impediments to coronavirus response and tracking: the fax machine. Dr. Mark Escott, the interim health authority for the city of Austin and Travis County, says","content":"\u003cp\u003e\u003ca href=\"https://www.nytimes.com/2020/07/13/upshot/coronavirus-response-fax-machines.html?referringSource=articleShare\"\u003eSobering story\u003c/a\u003e from Sarah Kliff and Margot Sanger-Katz in the \u003cem\u003eNew York Times\u003c/em\u003e about one of the impediments to coronavirus response and tracking: the fax machine.\u003c/p\u003e\n\u003cblockquote\u003e\u003cp\u003e\n  Dr. Mark Escott, the interim health authority for the city of Austin and Travis County, says his office is receiving around 1,000 faxes a day, including duplicate results. Some faxes are meant for other jurisdictions, and many are missing crucial information needed for his office to investigate cases. Most such faxes in Austin are being sent to a computer, but they still need to be printed and manually entered into public health databases.\n\u003c/p\u003e\u003c/blockquote\u003e\n\u003cp\u003eMind-boggling how much of healthcare still relies on the “cutting edge” of 1990s tech. While physicians offices and hospitals have gotten pushed to update to electronic medical records, the \u003cem\u003eTimes\u003c/em\u003e story points out that public health infrastructure hasn’t been subject to the same initiatives.\u003c/p\u003e\n\u003cblockquote\u003e\u003cp\u003e\n  In the early 2010s, the federal government spent billions to encourage doctors to replace fax machines with electronic records. That program, known as the HITECH Act, did not include similar funding for public health departments, to help them automatically digitize faxes and other nonstandard results. Nor did it require hospitals and doctor’s offices to build technology that would automatically send relevant test results to local health officials.\n\u003c/p\u003e\u003c/blockquote\u003e\n\u003cp\u003eMoreover, our healthcare system is much more decentralized than many other countries, which makes it an increasing challenge to coordinate the response to a public health crisis like we’re in now.\u003c/p\u003e\u003cp\u003e\u003ca href=\"https://www.nytimes.com/2020/07/13/upshot/coronavirus-response-fax-machines.html?referringSource=articleShare\"\u003eGo to the linked site\u003c/a\u003e.\u003c/p\u003e\u003cp\u003e\u003ca href=\"https://sixcolors.com/link/2020/07/12166/\"\u003eRead on Six Colors\u003c/a\u003e.\u003c/p\u003e","url":"https://sixcolors.com/link/2020/07/12166/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/4d133cc085b0fb72bf4dd5607f3f9ce3b5cbcbcf?base64_url=aHR0cHM6Ly9zaXhjb2xvcnMuY29tL2xpbmsvMjAyMC8wNy8xMjE2Ni8=","published":"2020-07-19T13:08:37.000000Z","created_at":"2020-07-19T13:35:44.401052Z"},{"id":2520285847,"feed_id":1342834,"title":"FTTC: Wieder mehr Vectoring und FTTH im Netz der Telekom verfügbar","author":"Achim Sawall","summary":"Diesmal können 290.000 Haushalte Vectoring, Super Vectoring und zunehmend auch Glasfaser (FTTH) nutzen. Die Telekom erweitert ihr Festnetz. (Vectoring, DSL)","content":"\u003cimg src=\"https://www.golem.de/2007/149738-237213-237210_rc.jpg\" width=\"140\" height=\"140\" vspace=\"3\" hspace=\"8\" align=\"left\"\u003eDiesmal können 290.000 Haushalte Vectoring, Super Vectoring und zunehmend auch Glasfaser (FTTH) nutzen. Die Telekom erweitert ihr Festnetz. (\u003ca href=\"https://www.golem.de/specials/vectoring/\"\u003eVectoring\u003c/a\u003e, \u003ca href=\"https://www.golem.de/specials/dsl/\"\u003eDSL\u003c/a\u003e) \u003cimg src=\"https://cpx.golem.de/cpx.php?class=17\u0026amp;aid=149738\u0026amp;page=1\u0026amp;ts=1595166420\" alt=\"\" width=\"1\" height=\"1\"\u003e","url":"https://www.golem.de/news/fttc-wieder-mehr-vectoring-und-ftth-im-netz-der-telekom-verfuegbar-2007-149738-rss.html","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/5bf231d72d064ec81791e0a2a8ffc252d199eb36?base64_url=aHR0cHM6Ly93d3cuZ29sZW0uZGUvbmV3cy9mdHRjLXdpZWRlci1tZWhyLXZlY3RvcmluZy11bmQtZnR0aC1pbS1uZXR6LWRlci10ZWxla29tLXZlcmZ1ZWdiYXItMjAwNy0xNDk3MzgtcnNzLmh0bWw=","published":"2020-07-19T13:47:00.000000Z","created_at":"2020-07-19T13:48:58.688931Z"},{"id":2520294160,"feed_id":1178507,"title":"This week’s top stories: iOS 13.6 released with Car Key, Apple previews new emoji, more","author":"Chance Miller","summary":"In this week’s top stories: iOS 13.6 released to everyone, Apple faces a steep penalty from Samsung, a brief look at new emoji coming this year, and much more. Read on for all of this week’s biggest news. more… The post This week’s top stories: iOS 13.6","content":"\u003cdiv class=\"feat-image\"\u003e\u003cimg src=\"https://9to5mac.com/wp-content/uploads/sites/6/2019/03/top-stories-header-steeber.jpg?quality=82\u0026amp;strip=all\u0026amp;w=1600\"\u003e\u003c/div\u003e\n\u003cp\u003e\u003cstrong\u003eIn this week’s top stories: \u003c/strong\u003eiOS 13.6 released to everyone, Apple faces a steep penalty from Samsung, a brief look at new emoji coming this year, and much more. Read on for all of this week’s biggest news.\u003cbr\u003e\n \u003ca href=\"https://9to5mac.com/2020/07/19/ios-13-6-features-car-key-emoji-more/#more-657127\" class=\"more-link\"\u003emore…\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://9to5mac.com/2020/07/19/ios-13-6-features-car-key-emoji-more/\"\u003eThis week’s top stories: iOS 13.6 released with Car Key, Apple previews new emoji, more\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://9to5mac.com\"\u003e9to5Mac\u003c/a\u003e.\u003c/p\u003e\n","url":"https://9to5mac.com/2020/07/19/ios-13-6-features-car-key-emoji-more/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/a8b7949554846fe05dc2021d6aa794fdbb35d374?base64_url=aHR0cHM6Ly85dG81bWFjLmNvbS8yMDIwLzA3LzE5L2lvcy0xMy02LWZlYXR1cmVzLWNhci1rZXktZW1vamktbW9yZS8=","published":"2020-07-19T13:51:27.000000Z","created_at":"2020-07-19T14:07:12.506365Z"},{"id":2520298294,"feed_id":1530714,"title":"Leak: IBM-Forscher finden 40 GByte an Hacker-Trainingsmaterial","author":null,"summary":"Sicherheitsexperten einer IBM-Forschungsgruppe konnten sich Zugang zu Videos verschaffen, die tiefe Einblicke in das Vorgehen von iranischen Hackern erlauben.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/news/Leak-IBM-Forscher-finden-40-GByte-an-Hacker-Trainingsmaterial-4847330.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/8/6/2/4/shutterstock_70213381-0c4d6cb19d0ce8e5.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eSicherheitsexperten einer IBM-Forschungsgruppe konnten sich Zugang zu Videos verschaffen, die tiefe Einblicke in das Vorgehen von iranischen Hackern erlauben.\u003c/p\u003e","url":"https://www.heise.de/news/Leak-IBM-Forscher-finden-40-GByte-an-Hacker-Trainingsmaterial-4847330.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/ed57dd4bd089b5cc956cfae0a4ee395be1212ff2?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvbmV3cy9MZWFrLUlCTS1Gb3JzY2hlci1maW5kZW4tNDAtR0J5dGUtYW4tSGFja2VyLVRyYWluaW5nc21hdGVyaWFsLTQ4NDczMzAuaHRtbD93dF9tYz1yc3MucmVkLmhvLmhvLmF0b20uYmVpdHJhZy5iZWl0cmFn","published":"2020-07-19T13:46:00.000000Z","created_at":"2020-07-19T14:12:21.062713Z"},{"id":2520349589,"feed_id":1178507,"title":"Apple dedicates Apple․com homepage to late congressman and civil rights activist John Lewis","author":"Chance Miller","summary":"Apple has updated the homepage for Apple․com this weekend to honor Congressman John Lewis, who passed away on Friday at 80-years-old. Apple’s new homepage foregoes all product promotion, instead showing a picture of Congressman Lewis alongside one of his","content":"\u003cdiv class=\"feat-image\"\u003e\u003cimg src=\"https://9to5mac.com/wp-content/uploads/sites/6/2020/07/Screen-Shot-2020-07-19-at-10.06.30-AM.jpeg?quality=82\u0026amp;strip=all\u0026amp;w=1600\"\u003e\u003c/div\u003e\n\u003cp\u003eApple has updated the \u003ca href=\"http://apple.com\"\u003ehomepage\u003c/a\u003e for Apple․com this weekend to honor Congressman John Lewis, who passed away on Friday at 80-years-old. Apple’s new homepage foregoes all product promotion, instead showing a picture of Congressman Lewis alongside one of his quotes.\u003c/p\u003e\n\u003cp\u003e \u003ca href=\"https://9to5mac.com/2020/07/19/john-lewis-apple-homepage/#more-657132\" class=\"more-link\"\u003emore…\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://9to5mac.com/2020/07/19/john-lewis-apple-homepage/\"\u003eApple dedicates Apple․com homepage to late congressman and civil rights activist John Lewis\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://9to5mac.com\"\u003e9to5Mac\u003c/a\u003e.\u003c/p\u003e\n","url":"https://9to5mac.com/2020/07/19/john-lewis-apple-homepage/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/0f9844a0eb4f38cd346c226f3c4588418012b565?base64_url=aHR0cHM6Ly85dG81bWFjLmNvbS8yMDIwLzA3LzE5L2pvaG4tbGV3aXMtYXBwbGUtaG9tZXBhZ2Uv","published":"2020-07-19T15:08:28.000000Z","created_at":"2020-07-19T15:17:13.524053Z"},{"id":2520352641,"feed_id":1466657,"title":"KFC Is Developing Lab-Grown Chicken Nuggets in Russia","author":"Kat Smith","summary":"KFC is developing lab-grown chicken nuggets made from real animal cells and plant-based ingredients in Russia. The international fast-food chain partnered with Russian biotechnology company, 3D Printing Solutions, which produces bioprinters and materials","content":"\u003cp\u003eKFC is developing lab-grown chicken nuggets made from real animal cells and plant-based ingredients in Russia. The international fast-food chain partnered with Russian biotechnology company, 3D Printing Solutions, which produces bioprinters and materials for 3D bioprinting. The project aims to create the world’s first lab-grown—also called clean, cell-based, or cultured—chicken nuggets. KFC expects to receive […]\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co/kfc-lab-grown-chicken-nuggets-russia/\"\u003eKFC Is Developing Lab-Grown Chicken Nuggets in Russia\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co\"\u003eLIVEKINDLY\u003c/a\u003e.\u003c/p\u003e\n","url":"https://www.livekindly.co/kfc-lab-grown-chicken-nuggets-russia/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/988157fa85457d6c941f1ced2e60a5e38a73815c?base64_url=aHR0cHM6Ly93d3cubGl2ZWtpbmRseS5jby9rZmMtbGFiLWdyb3duLWNoaWNrZW4tbnVnZ2V0cy1ydXNzaWEv","published":"2020-07-19T15:13:06.000000Z","created_at":"2020-07-19T15:20:11.173758Z"},{"id":2520353470,"feed_id":1530714,"title":"Wirecard-Skandal setzt Bundesregierung unter Druck","author":null,"summary":"Die mutmaßlichen Bilanzmanipulationen bei Wirecard blieben jahrelang verborgen. Der Fall ist mehr als ein Wirtschaftsskandal, die Opposition fordert Aufklärung.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/news/Wirecard-Skandal-setzt-Bundesregierung-unter-Druck-4847338.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/8/6/2/8/Wirecard_AG_Hauptsitz_Aschheim2_300dpi_03-8ce62ec9cccce9ee.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eDie mutmaßlichen Bilanzmanipulationen bei Wirecard blieben jahrelang verborgen. Der Fall ist mehr als ein Wirtschaftsskandal, die Opposition fordert Aufklärung.\u003c/p\u003e","url":"https://www.heise.de/news/Wirecard-Skandal-setzt-Bundesregierung-unter-Druck-4847338.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/f4a199d2e3608627591edc1529902f31bef67921?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvbmV3cy9XaXJlY2FyZC1Ta2FuZGFsLXNldHp0LUJ1bmRlc3JlZ2llcnVuZy11bnRlci1EcnVjay00ODQ3MzM4Lmh0bWw_d3RfbWM9cnNzLnJlZC5oby5oby5hdG9tLmJlaXRyYWcuYmVpdHJhZw==","published":"2020-07-19T15:02:00.000000Z","created_at":"2020-07-19T15:20:51.793688Z"},{"id":2520366874,"feed_id":1397625,"title":"Hot Gates","author":null,"summary":"","content":"\u003cdiv style=\"display:block; background-color: #ccc;\"\u003e\n\u003cp\u003e\u003ca href=\"https://www.oglaf.com/hotgates/\"\u003e\u003cimg src=\"https://media.oglaf.com/story/tthotgates.gif\"\u003e\u003c/a\u003e\u003c/p\u003e\n\u003cp\u003e\u003ca href=\"https://www.oglaf.com/hotgates/\"\u003e\u003cimg src=\"https://media.oglaf.com/archive/arc-hotgates.png\" width=\"400\" height=\"100\" border=\"0\" alt=\"https://www.oglaf.com/hotgates/\"\u003e\u003c/a\u003e\u003c/p\u003e\n\u003c/div\u003e\n","url":"https://www.oglaf.com/hotgates/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/df1ae47eac624d253bc5b92e32009aa2395cad26?base64_url=aHR0cHM6Ly93d3cub2dsYWYuY29tL2hvdGdhdGVzLw==","published":"2020-07-19T00:00:00.000000Z","created_at":"2020-07-19T15:34:54.195621Z"},{"id":2520407894,"feed_id":1530714,"title":"Manipuliertes Bewertungsportal – Zypern liefert Hacker an die USA aus","author":null,"summary":"Eine Anklage aus dem Jahr 2017 kommt nun in den USA vor Gericht: Der Angeklagte aus Zypern soll kritische Bewertungen aus einem Bewertungsportal entfernt haben.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/news/Zypern-liefert-Hacker-an-die-USA-aus-4847352.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/8/6/3/6/shutterstock_503797840-009cdf23178b84b9.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eEine Anklage aus dem Jahr 2017 kommt nun in den USA vor Gericht: Der Angeklagte aus Zypern soll kritische Bewertungen aus einem Bewertungsportal entfernt haben.\u003c/p\u003e","url":"https://www.heise.de/news/Zypern-liefert-Hacker-an-die-USA-aus-4847352.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/ccc62020411272ff88ee310e8d81f71e0c91f04a?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvbmV3cy9aeXBlcm4tbGllZmVydC1IYWNrZXItYW4tZGllLVVTQS1hdXMtNDg0NzM1Mi5odG1sP3d0X21jPXJzcy5yZWQuaG8uaG8uYXRvbS5iZWl0cmFnLmJlaXRyYWc=","published":"2020-07-19T16:13:00.000000Z","created_at":"2020-07-19T16:33:17.250509Z"},{"id":2520555978,"feed_id":1296379,"title":"White House Portraits of Bill Clinton and George W. Bush Moved From Prominent Space to a Closet, Like Deck Chairs Being Moved on Titanic","author":"John Gruber","summary":"Jeff Zeleny and Kevin Liptak, reporting for CNN: White House tradition calls for portraits of the most recent American presidents to be given the most prominent placement, in the entrance of the executive mansion, visible to guests during official events.","content":"\n\u003cp\u003eJeff Zeleny and Kevin Liptak, reporting for CNN:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eWhite House tradition calls for portraits of the most recent\nAmerican presidents to be given the most prominent placement, in\nthe entrance of the executive mansion, visible to guests during\nofficial events.\u003c/p\u003e\n\n\u003cp\u003eThat was the case through at least July 8, when President Donald\nTrump welcomed Mexican President Andrés Manuel López Obrador. The\ntwo stood in the Cross Hall of the White House and made remarks,\nwith the portraits of Clinton and Bush essentially looking on as\nthey had been throughout Trump’s first term. But in the days after\nafter that, the Clinton and Bush portraits were moved into the Old\nFamily Dining Room, a small, rarely used room that is not seen by\nmost visitors.\u003c/p\u003e\n\n\u003cp\u003eThat places the paintings well outside of Trump’s vantage point in\nthe White House. In their previous location, the pictures would\nhave been seen daily as Trump descends the staircase from his\nthird floor private residence or when he hosts events on the state\nfloor of the White House. Now, they hang in a space used mainly\nfor storing unused tablecloths and furniture.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eThe story of these portraits, in itself, is not important. But what’s behind this petty insignificant-in-the-grand-scheme-of-events story is the same fundamental truth that is the cause of so many deeply important problems happening right now: Donald Trump has the small mind and emotional maturity of a petulant child.\u003c/p\u003e\n\n\u003cp\u003eThe portrait story is all the more clarifying given that Bush is a two-term Republican. It’s hard to imagine a more \u003cem\u003epolitically\u003c/em\u003e polarizing president than Bush. We do live in a polarized time, and George W. Bush exemplified that polarization on the left-right divide. He was certainly far more conservative than Trump, both in rhetoric and policy. (Clinton was, famously, a moderate, and \u003ca href=\"https://www.presidency.ucsb.edu/statistics/data/final-presidential-job-approval-ratings\"\u003eleft office with the highest approval ratings of any president since World War II\u003c/a\u003e. Why Trump despises him is plainly obvious and has nothing to do with politics.) Trump’s problem with Bush isn’t partisan. It’s about adherence to foundational American ideals such as the rule of law, and the idea that the President of the United States is the leader of all Americans, not just those who support him. Say what you want about Bush’s presidency, when the nation faced a true crisis on 9/11, \u003ca href=\"https://news.gallup.com/poll/116500/presidential-approval-ratings-george-bush.aspx\"\u003ehe brought the nation together\u003c/a\u003e. \u003c/p\u003e\n\n\u003cp\u003eWhen faced with this crisis, Donald Trump, mind-bogglingly, drove the country further apart. His remaining supporters are with him not despite this, but because of it, like pigs wallowing in mud.\u003c/p\u003e\n\n\u003cp style=\"padding-top: 1.5em;\"\u003e\u003cem\u003eLink: \u003cstrong\u003e\u003ca href=\"https://www.cnn.com/2020/07/17/politics/white-house-portraits-clinton-bush-trump/index.html\"\u003ecnn.com/2020/07/17/politics/white-house-portraits-clinton…\u003c/a\u003e\u003c/strong\u003e\u003c/em\u003e\u003c/p\u003e\n","url":"https://daringfireball.net/linked/2020/07/19/white-house-portraits","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/64c4952881613382fb248c2c2cee0cd93b28f34e?base64_url=aHR0cHM6Ly9kYXJpbmdmaXJlYmFsbC5uZXQvbGlua2VkLzIwMjAvMDcvMTkvd2hpdGUtaG91c2UtcG9ydHJhaXRz","published":"2020-07-19T18:09:20.000000Z","created_at":"2020-07-19T19:56:35.217172Z"},{"id":2520594418,"feed_id":1466657,"title":"Loaded Vegan Nachos With Beans, Tomatoes and Guac","author":"Hazel and Cacao","summary":"These loaded vegan nachos with beans, tomatoes, and guac make the perfect dinner or side. I used to make vegetarian nachos all the time when I was younger. Except back then I just used store-bought salsa, no beans, and quite a lot of cheese! When I started","content":"\u003cp\u003eThese loaded vegan nachos with beans, tomatoes, and guac make the perfect dinner or side. I used to make vegetarian nachos all the time when I was younger. Except back then I just used store-bought salsa, no beans, and quite a lot of cheese! When I started on my health journey many years ago, I […]\u003c/p\u003e\n\u003cp\u003eThe post \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co/vegan-nachos-beans-tomatoes-guac/\"\u003eLoaded Vegan Nachos With Beans, Tomatoes and Guac\u003c/a\u003e appeared first on \u003ca rel=\"nofollow\" href=\"https://www.livekindly.co\"\u003eLIVEKINDLY\u003c/a\u003e.\u003c/p\u003e\n","url":"https://www.livekindly.co/vegan-nachos-beans-tomatoes-guac/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/db79d4a5e4a8f5e1db26d51a8f741cf4661d9d89?base64_url=aHR0cHM6Ly93d3cubGl2ZWtpbmRseS5jby92ZWdhbi1uYWNob3MtYmVhbnMtdG9tYXRvZXMtZ3VhYy8=","published":"2020-07-19T20:41:06.000000Z","created_at":"2020-07-19T20:47:17.187362Z"},{"id":2520669014,"feed_id":1296379,"title":"Microbe Economics","author":"John Gruber","summary":"Paul Krugman: Econ 101 has lots of good things to say about free markets (probably too many good things, but that’s a discussion for another time), but no rational discussion of economics says that free markets, left to themselves, can solve the problem of","content":"\n\u003cp\u003ePaul Krugman:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eEcon 101 has lots of good things to say about free markets\n(probably too many good things, but that’s a discussion for\nanother time), but no rational discussion of economics says that\nfree markets, left to themselves, can solve the problem of\n“externalities” — costs that individuals or businesses impose on\nothers who have no say in the matter. Pollution is the classic\nexample of an externality that requires government intervention,\nbut spreading a dangerous virus poses exactly the same issues.\u003c/p\u003e\n\n\u003cp\u003eYet many conservatives seem unable or unwilling to grasp this\nsimple point. And they seem equally unwilling to grasp a related\npoint — that there are some things that must be supplied through\npublic policy rather than individual initiative. And the most\nimportant of these “public goods” is probably scientific\nknowledge.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp\u003eAnd this again exemplifies how this abject failure of Republican leadership is not about any traditional left-right conservative-liberal partisanship. It’s science vs. willful ignorance. And the simple truth is that the Republicans used to be — or at least fancied themselves — the party of facing hard truths even when inconvenient or downright unpleasant. They insisted Democrats were “bleeding heart liberals”, who chose policies based on compassion rather than facts.\u003c/p\u003e\n\n\u003cp style=\"padding-top: 1.5em;\"\u003e\u003cem\u003eLink: \u003cstrong\u003e\u003ca href=\"https://www.nytimes.com/2020/07/18/opinion/republicans-keep-flunking-microbe-economics.html\"\u003enytimes.com/2020/07/18/opinion/republicans-keep-flunking…\u003c/a\u003e\u003c/strong\u003e\u003c/em\u003e\u003c/p\u003e\n","url":"https://daringfireball.net/linked/2020/07/19/microbe-economics","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/8a443db26faaa8b9cdbc296b3a79b52077211b12?base64_url=aHR0cHM6Ly9kYXJpbmdmaXJlYmFsbC5uZXQvbGlua2VkLzIwMjAvMDcvMTkvbWljcm9iZS1lY29ub21pY3M=","published":"2020-07-19T22:42:24.000000Z","created_at":"2020-07-19T22:51:40.391816Z"},{"id":2520730301,"feed_id":1296379,"title":"David Shor’s Unified Theory of American Politics","author":"John Gruber","summary":"While I’m on a political kick, please allow me to direct your attention to this interview with political strategist and data analyst David Shor, by Eric Levitz for New York Magazine. Shor’s insights are extraordinarily smart, fascinating, data-driven, and","content":"\n\u003cp\u003eWhile I’m on a political kick, please allow me to direct your attention to this interview with political strategist and data analyst David Shor, by Eric Levitz for New York Magazine. Shor’s insights are extraordinarily smart, fascinating, data-driven, and in ways large and small, often counter to conventional wisdom (e.g. big-money donors are pushing the Democrats to the \u003cem\u003eleft\u003c/em\u003e). I assure you it is worth your time and full attention.\u003c/p\u003e\n\n\u003cp\u003eA taste, regarding why non-college-educated voters have been drifting right, both in the U.S. and Europe, for decades:\u003c/p\u003e\n\n\u003cblockquote\u003e\n  \u003cp\u003eSo why is this happening? The story that makes the most sense to\nme goes like this: In the postwar era, college-educated\nprofessionals were maybe 4 percent of the electorate. Which meant\nthat basically no voters had remotely cosmopolitan values. But the\nflip side of this is that this educated 4 percent still ran the\nworld. Both parties at this point were run by this highly\neducated, cosmopolitan minority that held a bunch of values that\nundergirded the postwar consensus, around democracy and rule of\nlaw, and all these things.\u003c/p\u003e\n\n\u003cp\u003eObviously, these people were more right wing on a bunch of social\nissues than their contemporary counterparts, but during that era,\nboth parties were run by just about the most cosmopolitan segments\nof society. And there were also really strong gatekeepers. This\nsmall group of highly educated people not only controlled the\ncommanding heights of both the left and the right, but also\ncontrolled the media. There were only a small number of TV\nstations — in other countries, those stations were even run by\nthe government. And both sides knew it wasn’t electorally\nadvantageous to campaign on cosmopolitan values.\u003c/p\u003e\n\n\u003cp\u003eSo, as a result, campaigns centered around this cosmopolitan\nelite’s internal disagreements over economic issues. But over the\npast 60 years, college graduates have gone from being 4 percent of\nthe electorate to being more like 35. Now, it’s actually possible\n— for the first time ever in human history — for political\nparties to openly embrace cosmopolitan values and win elections;\ncertainly primary and municipal elections, maybe even national\nelections if you don’t push things too far or if you have a\nrecession at your back. And so Democratic elites started\ncampaigning on the things they’d always wanted to, but which had\npreviously been too toxic. And so did center-left parties\ninternationally.\u003c/p\u003e\n\u003c/blockquote\u003e\n\n\u003cp style=\"padding-top: 1.5em;\"\u003e\u003cem\u003eLink: \u003cstrong\u003e\u003ca href=\"https://nymag.com/intelligencer/2020/07/david-shor-cancel-culture-2020-election-theory-polls.html\"\u003enymag.com/intelligencer/2020/07/david-shor-cancel-culture…\u003c/a\u003e\u003c/strong\u003e\u003c/em\u003e\u003c/p\u003e\n","url":"https://daringfireball.net/linked/2020/07/19/shor-unified-theory-of-us-politics","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/cdf8b0873268c07d4fd4565e5b7b9b890766080a?base64_url=aHR0cHM6Ly9kYXJpbmdmaXJlYmFsbC5uZXQvbGlua2VkLzIwMjAvMDcvMTkvc2hvci11bmlmaWVkLXRoZW9yeS1vZi11cy1wb2xpdGljcw==","published":"2020-07-20T00:03:37.000000Z","created_at":"2020-07-20T00:11:34.215556Z"},{"id":2520786249,"feed_id":1848912,"title":"Japan plans massive national tech modernisation program","author":"Simon Sharwood","summary":"Land of the bureaucratic stamp wants to be ready if plague and natural disasters coincide Japan has announced a policy to modernise its government with a massive new program of digitisation, because the nation has decided it needs better information","content":"\u003ch4\u003eLand of the bureaucratic stamp wants to be ready if plague and natural disasters coincide\u003c/h4\u003e \u003cp\u003eJapan has announced a policy to modernise its government with a massive new program of digitisation, because the nation has decided it needs better information infrastructure to cope with both the current pandemic and other future challenges.…\u003c/p\u003e \u003cp\u003e\u003c!--#include virtual='/data_centre/_whitepaper_textlinks_top.html' --\u003e\u003c/p\u003e","url":"https://go.theregister.com/feed/www.theregister.com/2020/07/20/japan_most_advanced_digital_nation_plan/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/154890833c166d8cb2439b30883e1fe336f3962d?base64_url=aHR0cHM6Ly9nby50aGVyZWdpc3Rlci5jb20vZmVlZC93d3cudGhlcmVnaXN0ZXIuY29tLzIwMjAvMDcvMjAvamFwYW5fbW9zdF9hZHZhbmNlZF9kaWdpdGFsX25hdGlvbl9wbGFuLw==","published":"2020-07-20T01:29:13.000000Z","created_at":"2020-07-20T01:44:16.111631Z"},{"id":2520801907,"feed_id":1530714,"title":"Erste arabische Marsmission \"Al Amal\" startet erfolgreich","author":null,"summary":"Mit einer japanischen Rakete haben die Vereinigten Arabischen Emirate einen Satelliten namens Hoffnung losgeschickt. Er soll das Marswetter beobachten.","content":"\u003cp\u003e\u003ca href=\"https://www.heise.de/news/Erste-arabische-Marsmission-Al-Amal-startet-erfolgreich-4847455.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag\"\u003e\u003cimg src=\"https://www.heise.de/scale/geometry/450/q80//imgs/18/2/9/3/8/7/0/4/Al-Amal-5260e16211e94887.jpeg\" class=\"webfeedsFeaturedVisual\" alt=\"\"\u003e\u003c/a\u003e\u003c/p\u003e\u003cp\u003eMit einer japanischen Rakete haben die Vereinigten Arabischen Emirate einen Satelliten namens Hoffnung losgeschickt. Er soll das Marswetter beobachten.\u003c/p\u003e","url":"https://www.heise.de/news/Erste-arabische-Marsmission-Al-Amal-startet-erfolgreich-4847455.html?wt_mc=rss.red.ho.ho.atom.beitrag.beitrag","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/708b7c89ec315db449d7e838d9da587f6253352b?base64_url=aHR0cHM6Ly93d3cuaGVpc2UuZGUvbmV3cy9FcnN0ZS1hcmFiaXNjaGUtTWFyc21pc3Npb24tQWwtQW1hbC1zdGFydGV0LWVyZm9sZ3JlaWNoLTQ4NDc0NTUuaHRtbD93dF9tYz1yc3MucmVkLmhvLmhvLmF0b20uYmVpdHJhZy5iZWl0cmFn","published":"2020-07-20T01:56:00.000000Z","created_at":"2020-07-20T02:17:35.874306Z"},{"id":2520891036,"feed_id":1848912,"title":"United Arab Emirates’ Mars probe successfully launched and phones home","author":"Simon Sharwood","summary":"‘Hope’ mission is first interplanetary effort from Arab world, aims to sniff Martian atmosphere The United Arab Emirates has successfully launched a Mars probe.…","content":"\u003ch4\u003e‘Hope’ mission is first interplanetary effort from Arab world, aims to sniff Martian atmosphere\u003c/h4\u003e \u003cp\u003eThe United Arab Emirates has successfully launched a Mars probe.…\u003c/p\u003e","url":"https://go.theregister.com/feed/www.theregister.com/2020/07/20/hope_emirates_mars_mission_launch/","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/a02a5ffd7008c05cb9d27c07a22a52007485a4bb?base64_url=aHR0cHM6Ly9nby50aGVyZWdpc3Rlci5jb20vZmVlZC93d3cudGhlcmVnaXN0ZXIuY29tLzIwMjAvMDcvMjAvaG9wZV9lbWlyYXRlc19tYXJzX21pc3Npb25fbGF1bmNoLw==","published":"2020-07-20T04:30:09.000000Z","created_at":"2020-07-20T04:35:51.484882Z"},{"id":2520898115,"feed_id":1545349,"title":"Gewinnspiel: The Pretenders","author":null,"summary":"(Gewinnspiel) Wir verlosen fünfmal das neue Album \"Hate For Sale\" der britischen Rockband. lesen","content":"\u003cimg src=\"https://www.queer.de/publisher/pics/the-pretenders-band-2020-social.jpg\" align=\"right\" style=\"margin-left: 5px;\"\u003e (Gewinnspiel) Wir verlosen fünfmal das neue Album \"Hate For Sale\" der britischen Rockband. \u003ca href=\"https://www.queer.de/detail.php?article_id=36626\"\u003elesen\u003c/a\u003e","url":"https://www.queer.de/detail.php?article_id=36626","extracted_content_url":"https://extract.feedbin.com/parser/feedbin/c1cf5a4b4e219e84d2628457352ce5d89c805cb9?base64_url=aHR0cHM6Ly93d3cucXVlZXIuZGUvZGV0YWlsLnBocD9hcnRpY2xlX2lkPTM2NjI2","published":"2020-07-20T04:11:36.000000Z","created_at":"2020-07-20T04:46:03.218742Z"}]"#;
        let _deserialized: Vec<Entry> = serde_json::from_str(json).unwrap();
    }
}
