//! Query a program for its features.
//!
//! The [`obtain_features`] function, when passed
//! a [`Config`][crate::defs::Config] object, will run a program with
//! the appropriate command-line options, analyze its output, and
//! build up a list of the supported features.
//!
//! See the crate-level documentation for sample usage.

/*
 * Copyright (c) 2021  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
use std::collections::HashMap;
use std::error;
use std::process;

use crate::defs;

fn decode_output(output: Vec<u8>) -> Result<String, Box<dyn error::Error>> {
    // FIXME: handle non-UTF-8 stuff
    Ok(String::from_utf8(output)?)
}

/// Run the specified program, analyze its output.
///
/// See the crate-level documentation for sample usage.
pub fn obtain_features(config: &defs::Config) -> Result<defs::Obtained, Box<dyn error::Error>> {
    match process::Command::new(&config.program)
        .args(&[&config.option_name])
        .output()
    {
        Ok(output) => match output.status.success() {
            true => {
                match decode_output(output.stdout)?
                    .lines()
                    .find(|line| line.starts_with(&config.prefix))
                {
                    Some(line) => {
                        let pfx_len = config.prefix.len();
                        let res: HashMap<String, String> = line[pfx_len..]
                            .split(' ')
                            .map(|pair| match pair.find('=') {
                                Some(pos) => (pair[..pos].to_string(), pair[pos + 1..].to_string()),
                                None => (pair.to_string(), "1.0".to_string()),
                            })
                            .collect();
                        Ok(defs::Obtained::Features(res))
                    }
                    None => Ok(defs::Obtained::NotSupported),
                }
            }
            false => Ok(defs::Obtained::NotSupported),
        },
        Err(err) => Ok(defs::Obtained::Failed(Box::new(err))),
    }
}
