#![no_std]
#![no_main]
#![allow(clippy::bool_comparison)]

use bsp::ehal;
use bsp::hal;
use feather_m4 as bsp;

#[cfg(not(feature = "use_semihosting"))]
use panic_halt as _;
#[cfg(feature = "use_semihosting")]
use panic_semihosting as _;

use bsp::entry;
use ehal::digital::v2::ToggleableOutputPin;
use hal::clock::GenericClockController;
use hal::pac::{interrupt, CorePeripherals, Peripherals};
use hal::{pukcc::*, usb::UsbBus};

use usb_device::bus::UsbBusAllocator;
use usb_device::prelude::*;
use usbd_serial::{SerialPort, USB_CLASS_CDC};

use cortex_m::asm::delay as cycle_delay;
use cortex_m::peripheral::NVIC;

#[entry]
fn main() -> ! {
    let mut peripherals = Peripherals::take().unwrap();
    let mut core = CorePeripherals::take().unwrap();
    let mut clocks = GenericClockController::with_external_32kosc(
        peripherals.GCLK,
        &mut peripherals.MCLK,
        &mut peripherals.OSC32KCTRL,
        &mut peripherals.OSCCTRL,
        &mut peripherals.NVMCTRL,
    );
    let pins = bsp::Pins::new(peripherals.PORT);
    let mut red_led = pins.d13.into_push_pull_output();

    let bus_allocator = unsafe {
        USB_ALLOCATOR = Some(bsp::usb_allocator(
            pins.usb_dm,
            pins.usb_dp,
            peripherals.USB,
            &mut clocks,
            &mut peripherals.MCLK,
        ));
        USB_ALLOCATOR.as_ref().unwrap()
    };

    unsafe {
        USB_SERIAL = Some(SerialPort::new(bus_allocator));
        USB_BUS = Some(
            UsbDeviceBuilder::new(bus_allocator, UsbVidPid(0x16c0, 0x27dd))
                .manufacturer("Fake company")
                .product("Serial port")
                .serial_number("TEST")
                .device_class(USB_CLASS_CDC)
                .build(),
        );
    }

    unsafe {
        core.NVIC.set_priority(interrupt::USB_OTHER, 1);
        core.NVIC.set_priority(interrupt::USB_TRCPT0, 1);
        core.NVIC.set_priority(interrupt::USB_TRCPT1, 1);
        NVIC::unmask(interrupt::USB_OTHER);
        NVIC::unmask(interrupt::USB_TRCPT0);
        NVIC::unmask(interrupt::USB_TRCPT1);
    }

    let pukcc = Pukcc::enable(&mut peripherals.MCLK).unwrap();

    loop {
        serial_writeln!("ECDSA Test");
        serial_writeln!("Column 1: Is generated signature identical to a reference signature?");
        serial_writeln!("Column 2: Is a signature valid according to PUKCC");
        serial_writeln!("Column 3: Is a broken signature invalid according to PUKCC");
        serial_writeln!("Test vector: {} samples", ecdsa::K_SIGNATURE_PAIRS.len());
        for (i, (k, reference_signature)) in ecdsa::K_SIGNATURE_PAIRS.iter().enumerate() {
            let i = i + 1;
            let mut generated_signature = [0_u8; 64];
            let are_signatures_same = match unsafe {
                pukcc.zp_ecdsa_sign_with_raw_k::<curves::Nist256p>(
                    &mut generated_signature,
                    &HASH,
                    &ecdsa::PRIVATE_KEY,
                    k,
                )
            } {
                Ok(_) => generated_signature
                    .iter()
                    .zip(reference_signature.iter())
                    .map(|(&left, &right)| left == right)
                    .all(|r| r == true),
                Err(e) => {
                    serial_writeln!("Error during signature generation: {:?}", e);
                    false
                }
            };
            let is_signature_valid = pukcc
                .zp_ecdsa_verify_signature::<curves::Nist256p>(
                    &generated_signature,
                    &HASH,
                    &ecdsa::PUBLIC_KEY,
                )
                .is_ok();

            // Break signature
            generated_signature[14] = generated_signature[14].wrapping_sub(1);

            let is_broken_signature_invalid = pukcc
                .zp_ecdsa_verify_signature::<curves::Nist256p>(
                    &generated_signature,
                    &HASH,
                    &ecdsa::PUBLIC_KEY,
                )
                .is_err();

            serial_writeln!(
                "{:>2}: {:<5} | {:<5} | {:<5}",
                i,
                are_signatures_same,
                is_signature_valid,
                is_broken_signature_invalid,
            );
        }
        serial_writeln!("ExpMod Test (RSA)");
        serial_writeln!("Column 1: Is generated signature identical to a reference signature?");
        serial_writeln!("Column 2: Is a signature valid");
        serial_writeln!("Column 3: Is a broken signature invalid");
        serial_writeln!("Test vector: {} samples", exp_mod::RSA_VECTOR.len());
        for (i, (modulus, private_exponent, reference_signature)) in
            exp_mod::RSA_VECTOR.iter().enumerate()
        {
            let mut buffer = [0_u8; 512 + 5];
            let are_signatures_same = match pukcc.modular_exponentiation(
                &HASH,
                private_exponent,
                modulus,
                ExpModMode::Regular,
                ExpModWindowSize::One,
                &mut buffer,
            ) {
                Ok(generated_signature) => generated_signature
                    .iter()
                    .cloned()
                    .rev()
                    .zip(reference_signature.iter().cloned().rev())
                    .map(|(left, right)| left == right)
                    .all(|r| r == true),
                Err(e) => {
                    serial_writeln!("Error during signture generation: {:?}", e);
                    false
                }
            };
            let is_signature_valid = match pukcc.modular_exponentiation(
                reference_signature,
                exp_mod::PUBLIC_EXPONENT,
                modulus,
                ExpModMode::Regular,
                ExpModWindowSize::One,
                &mut buffer,
            ) {
                Ok(retrieved_hash) => retrieved_hash
                    .iter()
                    .cloned()
                    .rev()
                    .zip(HASH.iter().cloned().rev())
                    .map(|(left, right)| left == right)
                    .all(|r| r == true),
                Err(e) => {
                    serial_writeln!("Error during signture generation: {:?}", e);
                    false
                }
            };
            let mut broken_reference_signature = [0_u8; 512];
            broken_reference_signature
                .iter_mut()
                .rev()
                .zip(reference_signature.iter().rev())
                .for_each(|(target_iter, source_iter)| *target_iter = *source_iter);
            broken_reference_signature[broken_reference_signature.len() - 1] =
                broken_reference_signature[broken_reference_signature.len() - 1].wrapping_sub(1);
            let is_broken_signature_invalid = match pukcc.modular_exponentiation(
                &broken_reference_signature[broken_reference_signature.len() - modulus.len()..],
                exp_mod::PUBLIC_EXPONENT,
                modulus,
                ExpModMode::Regular,
                ExpModWindowSize::One,
                &mut buffer,
            ) {
                Ok(retrieved_hash) => !retrieved_hash
                    .iter()
                    .cloned()
                    .rev()
                    .zip(HASH.iter().cloned().rev())
                    .map(|(left, right)| left == right)
                    .all(|r| r == true),
                Err(e) => {
                    serial_writeln!("Error during signture generation: {:?}", e);
                    false
                }
            };
            serial_writeln!(
                "{:>2}: {:<5} | {:<5} | {:<5} (RSA{})",
                i,
                are_signatures_same,
                is_signature_valid,
                is_broken_signature_invalid,
                modulus.len() * 8,
            );
        }

        cycle_delay(5 * 1024 * 1024);
        red_led.toggle().ok();
    }
}

static mut USB_ALLOCATOR: Option<UsbBusAllocator<UsbBus>> = None;
static mut USB_BUS: Option<UsbDevice<UsbBus>> = None;
static mut USB_SERIAL: Option<SerialPort<UsbBus>> = None;

/// Borrows the global singleton `UsbSerial` for a brief period with interrupts
/// disabled
///
/// # Arguments
/// `borrower`: The closure that gets run borrowing the global `UsbSerial`
///
/// # Safety
/// the global singleton `UsbSerial` can be safely borrowed because we disable
/// interrupts while it is being borrowed, guaranteeing that interrupt handlers
/// like `USB` cannot mutate `UsbSerial` while we are as well.
///
/// # Panic
/// If `init` has not been called and we haven't initialized our global
/// singleton `UsbSerial`, we will panic.
fn usbserial_get<T, R>(borrower: T) -> R
where
    T: Fn(&mut SerialPort<UsbBus>) -> R,
{
    usb_free(|_| unsafe {
        let usb_serial = USB_SERIAL.as_mut().expect("UsbSerial not initialized");
        borrower(usb_serial)
    })
}

/// Execute closure `f` in an interrupt-free context.
///
/// This as also known as a "critical section".
#[inline]
fn usb_free<F, R>(f: F) -> R
where
    F: FnOnce(&cortex_m::interrupt::CriticalSection) -> R,
{
    NVIC::mask(interrupt::USB_OTHER);
    NVIC::mask(interrupt::USB_TRCPT0);
    NVIC::mask(interrupt::USB_TRCPT1);

    let r = f(unsafe { &cortex_m::interrupt::CriticalSection::new() });

    unsafe {
        NVIC::unmask(interrupt::USB_OTHER);
        NVIC::unmask(interrupt::USB_TRCPT0);
        NVIC::unmask(interrupt::USB_TRCPT1);
    };

    r
}

/// Writes the given message out over USB serial.
///
/// # Arguments
/// * println args: variable arguments passed along to `core::write!`
///
/// # Warning
/// as this function deals with a static mut, and it is also accessed in the
/// USB interrupt handler, we both have unsafe code for unwrapping a static mut
/// as well as disabling of interrupts while we do so.
///
/// # Safety
/// the only time the static mut is used, we have interrupts disabled so we know
/// we have sole access
#[macro_export]
macro_rules! serial_writeln {
    ($($tt:tt)+) => {{
        use core::fmt::Write;

        let mut s: heapless::String<256> = heapless::String::new();
        core::write!(&mut s, $($tt)*).unwrap();
        usbserial_get(|usbserial| {
            usbserial.write(s.as_bytes()).ok();
            usbserial.write("\r\n".as_bytes()).ok();
        });
    }};
}

fn poll_usb() {
    unsafe {
        if let Some(usb_dev) = USB_BUS.as_mut() {
            if let Some(serial) = USB_SERIAL.as_mut() {
                usb_dev.poll(&mut [serial]);
                let mut buf = [0u8; 64];

                if let Ok(count) = serial.read(&mut buf) {
                    for (i, c) in buf.iter().enumerate() {
                        if i >= count {
                            break;
                        }
                        serial.write(&[*c]).unwrap();
                    }
                };
            };
        };
    };
}

#[interrupt]
fn USB_OTHER() {
    poll_usb();
}

#[interrupt]
fn USB_TRCPT0() {
    poll_usb();
}

#[interrupt]
fn USB_TRCPT1() {
    poll_usb();
}

pub const HASH: [u8; 32] = [
    0xc7, 0x9a, 0x27, 0x4d, 0x91, 0xbb, 0x92, 0x9f, 0x29, 0x16, 0xf8, 0x9c, 0xb2, 0xa6, 0xec, 0x66,
    0xa0, 0xcd, 0xb4, 0x4a, 0x14, 0x97, 0x63, 0x65, 0x3f, 0x28, 0x08, 0x52, 0xbb, 0xa5, 0x3b, 0x0e,
];

mod exp_mod {
    pub const PUBLIC_EXPONENT: &[u8] = &[0x01, 0x00, 0x01];

    /// Vector of tuples of (modulus, private_exponent, expected_signature)
    pub const RSA_VECTOR: [(&[u8], &[u8], &[u8]); 8] = [
        // RSA512
        (
            &[
                0xb4, 0x91, 0x91, 0x01, 0x0e, 0x56, 0x37, 0x0b, 0x63, 0x74, 0xd2, 0xf6, 0x5f, 0xee,
                0x2b, 0x06, 0x29, 0x4d, 0x34, 0x05, 0x97, 0xc0, 0xdd, 0xc5, 0xd4, 0xba, 0x16, 0xbe,
                0x74, 0x7c, 0xf2, 0x5f, 0xbd, 0xdb, 0xbb, 0x64, 0x29, 0x33, 0x32, 0x09, 0x18, 0x45,
                0xde, 0xd6, 0x1c, 0x5b, 0xc3, 0xbc, 0x5c, 0x32, 0xd7, 0xf7, 0xd9, 0xd2, 0xa5, 0x7e,
                0xf3, 0x4e, 0x1b, 0x7b, 0xd6, 0x87, 0x7d, 0xf9,
            ],
            &[
                0xb1, 0x3f, 0xa3, 0x23, 0x72, 0x3b, 0x57, 0x55, 0x2a, 0x8f, 0xe6, 0xf6, 0x4a, 0x3c,
                0xb4, 0xa7, 0x1b, 0xab, 0xd9, 0x41, 0x14, 0x32, 0x12, 0x80, 0xbb, 0xcc, 0xdf, 0xbe,
                0x9d, 0x02, 0x79, 0xb7, 0xb3, 0xbb, 0xbd, 0x39, 0xa1, 0x53, 0x4c, 0x9a, 0xdd, 0xfb,
                0x4b, 0x4b, 0x47, 0x78, 0xfe, 0x76, 0x90, 0xa7, 0x29, 0xdf, 0x25, 0x67, 0x4d, 0x2f,
                0x82, 0x92, 0x8a, 0x45, 0xb6, 0xec, 0x74, 0xb1,
            ],
            &[
                0x09, 0x8d, 0x27, 0x93, 0xd9, 0xcd, 0x67, 0xcb, 0xb4, 0x85, 0xbd, 0x43, 0xec, 0x43,
                0x12, 0x89, 0xf6, 0x3a, 0x40, 0xd8, 0x5c, 0x07, 0xa9, 0x48, 0x0d, 0x5a, 0xa9, 0x7f,
                0xf6, 0x12, 0x7b, 0x54, 0xe2, 0xad, 0xc6, 0x6d, 0x8d, 0x0a, 0x9f, 0x4b, 0x3f, 0x1c,
                0xf3, 0x00, 0x44, 0xa1, 0xa9, 0x78, 0x9d, 0x9b, 0xa7, 0xf4, 0x17, 0x4f, 0xbd, 0xa5,
                0xa3, 0xe3, 0xa4, 0x5d, 0x6e, 0xf4, 0x6a, 0x4e,
            ],
        ),
        (
            &[
                0xc5, 0x98, 0xfd, 0x58, 0x86, 0x43, 0x71, 0x16, 0x62, 0xa1, 0xe1, 0x05, 0xf8, 0xe1,
                0x33, 0x7d, 0x88, 0xe3, 0xca, 0xd0, 0x4b, 0x8e, 0xc7, 0x24, 0x49, 0x41, 0x36, 0xeb,
                0x2d, 0x87, 0xab, 0xe9, 0x90, 0x9e, 0x6c, 0x3a, 0x5b, 0xda, 0x27, 0x4c, 0x6c, 0xb2,
                0x58, 0xc2, 0x93, 0x82, 0xe9, 0xc4, 0x04, 0x45, 0x6a, 0x45, 0x26, 0x33, 0x52, 0xc0,
                0xd5, 0x54, 0xfa, 0x76, 0x10, 0x23, 0x59, 0x39,
            ],
            &[
                0xb6, 0x23, 0xb1, 0xd3, 0xf9, 0xdf, 0x0c, 0xcc, 0xef, 0x99, 0xfc, 0x0c, 0x3f, 0x3f,
                0x86, 0xf0, 0xfe, 0x4f, 0xcb, 0x51, 0x79, 0x74, 0x60, 0xc8, 0x67, 0xe5, 0xff, 0x33,
                0x85, 0x42, 0x7c, 0x09, 0x2f, 0x2b, 0x1c, 0xf9, 0xd7, 0xec, 0x0b, 0x6a, 0x38, 0x20,
                0xb1, 0xa5, 0x96, 0x09, 0x9b, 0xd8, 0xd0, 0xcb, 0x8b, 0xc8, 0xfb, 0xa4, 0xb0, 0xf8,
                0x6b, 0xdc, 0xf7, 0xa0, 0xf2, 0x94, 0x85, 0x49,
            ],
            &[
                0x35, 0xae, 0x2b, 0x10, 0x49, 0x1d, 0xcf, 0x53, 0x51, 0xf2, 0xf4, 0xab, 0xa8, 0x4c,
                0x6c, 0x38, 0xe8, 0xd8, 0xa5, 0x40, 0x65, 0x16, 0x62, 0x2a, 0xe2, 0xdd, 0x93, 0x71,
                0x1e, 0xd0, 0xa7, 0x2d, 0xcd, 0xd7, 0x61, 0x22, 0x83, 0x85, 0x76, 0x1f, 0xb2, 0x96,
                0xca, 0xea, 0x30, 0x7d, 0x05, 0xc8, 0x9f, 0x9c, 0x71, 0xfd, 0x51, 0x0e, 0xe5, 0x2a,
                0xf7, 0xb1, 0xd5, 0x26, 0x9e, 0x98, 0x27, 0x90,
            ],
        ),
        // RSA1024
        (
            &[
                0xbd, 0x36, 0x49, 0x74, 0xde, 0x21, 0x36, 0x0e, 0x34, 0xaf, 0xc8, 0x97, 0x49, 0xb0,
                0xe6, 0x3d, 0x67, 0x1c, 0x83, 0x0c, 0x43, 0x17, 0x39, 0x53, 0x54, 0xe8, 0xcd, 0x52,
                0xc2, 0xc1, 0x78, 0x03, 0x20, 0xb5, 0xbd, 0x93, 0xc2, 0x52, 0xf6, 0xb1, 0xb9, 0xab,
                0x9c, 0xcb, 0xe0, 0x2f, 0x26, 0xf1, 0x02, 0x7e, 0xe6, 0xfb, 0xc8, 0x61, 0xe3, 0x9f,
                0xd3, 0xa7, 0x43, 0x7e, 0x6e, 0x4f, 0xfa, 0x35, 0x3f, 0x64, 0xe6, 0x59, 0x11, 0xc6,
                0xe4, 0x18, 0xed, 0xf1, 0x0a, 0x78, 0x9b, 0x86, 0xa2, 0x09, 0x46, 0x7f, 0x65, 0x91,
                0x03, 0xf4, 0xc9, 0x31, 0xaa, 0x42, 0x50, 0xe9, 0xdf, 0x26, 0x1e, 0x10, 0x95, 0x62,
                0x40, 0x0e, 0xa5, 0x02, 0x13, 0xe3, 0xcc, 0xc9, 0x94, 0x2f, 0xf8, 0x8a, 0xf2, 0x03,
                0x02, 0x70, 0xb5, 0x62, 0xc4, 0x9c, 0xc9, 0xcf, 0x85, 0x56, 0xf6, 0x3f, 0x64, 0xbe,
                0xd0, 0x77,
            ],
            &[
                0x1f, 0xd4, 0x8f, 0x1c, 0xf2, 0xf5, 0x61, 0x53, 0x98, 0x77, 0x6d, 0xe6, 0x4d, 0x86,
                0x4d, 0xe5, 0x4c, 0x80, 0x56, 0x67, 0x6c, 0xad, 0xee, 0x7d, 0xdf, 0x4d, 0xde, 0xa4,
                0xaa, 0x90, 0xc3, 0x70, 0xbb, 0x42, 0xf7, 0xa6, 0x70, 0xcd, 0x66, 0x24, 0xd5, 0xd8,
                0x51, 0xe3, 0x56, 0x4e, 0x78, 0x5d, 0x99, 0x0e, 0xe2, 0x2a, 0xbf, 0x36, 0x00, 0x85,
                0xf5, 0xa4, 0x30, 0xcd, 0x87, 0x1f, 0x3b, 0x37, 0x09, 0xe4, 0x37, 0xf6, 0x2a, 0xa6,
                0x2d, 0x73, 0x82, 0x0c, 0x72, 0xac, 0xc4, 0xe5, 0x2d, 0x37, 0x0e, 0xe0, 0x57, 0x31,
                0xe0, 0x13, 0xdd, 0x35, 0x26, 0x9f, 0x8f, 0x4a, 0x53, 0x61, 0x2b, 0x08, 0xd0, 0xfe,
                0xe8, 0x87, 0xc9, 0xa3, 0x12, 0xad, 0xe7, 0x08, 0x3f, 0xc9, 0xd7, 0xc1, 0xb7, 0x98,
                0x0d, 0xae, 0x32, 0xc2, 0xc7, 0x0b, 0x14, 0x2c, 0xc5, 0xdb, 0xed, 0x0d, 0x22, 0x4a,
                0xc6, 0xf9,
            ],
            &[
                0x25, 0x3f, 0x02, 0x7c, 0xa2, 0x6f, 0x25, 0x18, 0x51, 0x2b, 0xb6, 0x47, 0x57, 0xf1,
                0x47, 0xbe, 0xb6, 0xa5, 0x8c, 0xa7, 0x3b, 0x12, 0x3b, 0x16, 0x52, 0xea, 0xe2, 0x73,
                0xe1, 0xda, 0x93, 0x65, 0xd2, 0x1b, 0xe8, 0x18, 0xaa, 0xbb, 0x35, 0x67, 0xbc, 0xa8,
                0xe5, 0x0d, 0xd6, 0x10, 0x50, 0xf8, 0x92, 0x8e, 0xa0, 0x62, 0x39, 0x8c, 0xe1, 0xb7,
                0x00, 0xf9, 0x47, 0x34, 0xa5, 0x6b, 0x37, 0x1e, 0xd3, 0xf5, 0xaa, 0x65, 0x9a, 0x10,
                0x10, 0xfa, 0x26, 0xe1, 0x36, 0x17, 0xe1, 0x87, 0xab, 0xc4, 0xa4, 0x68, 0x69, 0xc1,
                0xe0, 0x32, 0x2c, 0x0b, 0xba, 0x6a, 0x9a, 0x17, 0x09, 0xa8, 0x50, 0xee, 0x5f, 0x13,
                0x30, 0x02, 0x3c, 0xf5, 0x49, 0xd1, 0x58, 0xc7, 0xb6, 0x27, 0x3f, 0x74, 0x77, 0xcd,
                0xb4, 0x08, 0x2f, 0xe5, 0x02, 0x14, 0x1c, 0xd2, 0xdc, 0x85, 0x09, 0x32, 0x40, 0xfb,
                0xd9, 0x7e,
            ],
        ),
        (
            &[
                0xb5, 0xb7, 0x34, 0x9b, 0x99, 0xc5, 0x39, 0xfc, 0xdd, 0x02, 0x6f, 0x7e, 0x6e, 0xb4,
                0xa6, 0x35, 0xfe, 0xeb, 0x29, 0xe2, 0xba, 0x2f, 0x4d, 0xb2, 0x94, 0x49, 0xe7, 0x57,
                0xb2, 0xcc, 0xce, 0xb0, 0xb7, 0x30, 0xb0, 0x2f, 0x64, 0xe0, 0xf0, 0xd9, 0x68, 0x75,
                0xd0, 0x62, 0x6f, 0x6c, 0x9b, 0xfc, 0x89, 0x42, 0xf3, 0xad, 0x2e, 0x67, 0xef, 0xeb,
                0x77, 0xcc, 0x61, 0xeb, 0xd6, 0x09, 0x32, 0xcf, 0xfc, 0xaa, 0xd0, 0x3c, 0x56, 0xb4,
                0x3d, 0xbe, 0x84, 0x33, 0x64, 0x78, 0xd3, 0x31, 0x47, 0x6f, 0x23, 0x02, 0x81, 0xe3,
                0xce, 0x6f, 0xdd, 0x77, 0x36, 0x8b, 0xce, 0x60, 0xa5, 0xec, 0x63, 0x3f, 0x6b, 0x6a,
                0xb0, 0x4a, 0x0c, 0xed, 0x9b, 0x95, 0x83, 0x6b, 0x5e, 0xe7, 0x74, 0x23, 0x43, 0xa3,
                0x6e, 0xcc, 0x0b, 0x92, 0xf6, 0x09, 0x07, 0x6e, 0x40, 0xa1, 0x2b, 0x97, 0xe6, 0x49,
                0x94, 0x43,
            ],
            &[
                0xa8, 0x51, 0x60, 0x00, 0x75, 0x69, 0xf3, 0xb1, 0x9e, 0x82, 0x20, 0x06, 0x4b, 0xc3,
                0x37, 0x66, 0x32, 0x8f, 0x5f, 0x87, 0xed, 0x0f, 0xdd, 0xf7, 0x79, 0x56, 0x0f, 0x5c,
                0xf3, 0x78, 0xb4, 0x47, 0x8a, 0x18, 0x26, 0x4a, 0x70, 0x35, 0xcf, 0xc2, 0x81, 0xf9,
                0x07, 0x21, 0xf6, 0xb5, 0xf2, 0xb2, 0xf3, 0xed, 0xb9, 0x4b, 0x03, 0xfe, 0x30, 0x84,
                0xba, 0xbd, 0xed, 0x42, 0x07, 0x4b, 0x13, 0xed, 0x74, 0xd1, 0x0d, 0x42, 0x70, 0x9a,
                0xf8, 0xff, 0x39, 0x1c, 0xab, 0x6c, 0xe7, 0xbc, 0x26, 0x55, 0x64, 0x69, 0x5c, 0x48,
                0x72, 0x43, 0x2d, 0x04, 0x99, 0x50, 0x35, 0x03, 0x3b, 0x3e, 0xfd, 0xb3, 0x86, 0xfd,
                0x57, 0x2b, 0x03, 0xc3, 0x40, 0x9f, 0xe7, 0x9d, 0x39, 0xf2, 0x68, 0x48, 0x6f, 0xda,
                0x88, 0x56, 0x82, 0x74, 0x55, 0xea, 0x0b, 0xa7, 0x50, 0x4b, 0xff, 0x14, 0xf5, 0xee,
                0xce, 0xb9,
            ],
            &[
                0x67, 0x51, 0xc4, 0x41, 0x89, 0x48, 0x06, 0x33, 0xd7, 0x86, 0x1b, 0x1c, 0x35, 0x12,
                0x6c, 0x10, 0x37, 0xad, 0x78, 0xc6, 0x47, 0xbe, 0x86, 0xea, 0x3e, 0x09, 0x17, 0x04,
                0xe0, 0x13, 0x9b, 0xac, 0x02, 0xcb, 0x6d, 0x4c, 0x56, 0xc2, 0xcc, 0xfc, 0x88, 0x98,
                0x96, 0x2c, 0xd1, 0x7f, 0xcd, 0x36, 0xe3, 0x26, 0x1f, 0x63, 0xe2, 0xc1, 0x82, 0x4d,
                0x84, 0x14, 0x11, 0x00, 0x1c, 0x02, 0x00, 0x58, 0xa1, 0xb1, 0x42, 0x60, 0x56, 0x4f,
                0x0e, 0x7b, 0xf2, 0xd2, 0x89, 0xd9, 0xb0, 0xce, 0x1d, 0xd9, 0x19, 0x84, 0x18, 0x35,
                0xbe, 0x18, 0x9d, 0xed, 0x96, 0x26, 0x15, 0x7e, 0x96, 0x54, 0xbf, 0x10, 0x4d, 0x25,
                0x22, 0x4f, 0xeb, 0x40, 0x91, 0x3b, 0x1f, 0xb6, 0x54, 0x9f, 0x1e, 0x52, 0x2d, 0x39,
                0x9c, 0x91, 0x54, 0xac, 0x41, 0xd1, 0xb6, 0x88, 0x52, 0xe1, 0x89, 0x02, 0xd7, 0x56,
                0x01, 0xd5,
            ],
        ),
        // RSA2048
        (
            &[
                0xb5, 0x85, 0x27, 0x1a, 0xda, 0xf4, 0x0d, 0x95, 0x13, 0xcc, 0x6a, 0x46, 0x88, 0x35,
                0x99, 0xc9, 0x83, 0x57, 0xc8, 0xc5, 0xa3, 0x12, 0x1c, 0x68, 0xae, 0x00, 0x80, 0xe2,
                0x3e, 0x8c, 0x2e, 0x4e, 0x9f, 0xae, 0x21, 0x58, 0x31, 0x54, 0x82, 0x59, 0xd6, 0x5d,
                0x5e, 0x96, 0x3a, 0x0e, 0x17, 0xf5, 0x03, 0xdb, 0x10, 0x6a, 0xdd, 0xea, 0x85, 0x2c,
                0x60, 0x1d, 0xde, 0x06, 0xf1, 0x6d, 0x53, 0xc2, 0xaa, 0x2d, 0x5e, 0x3e, 0xa0, 0x31,
                0xb9, 0x90, 0xa7, 0x2d, 0xeb, 0x7c, 0x84, 0xde, 0xdf, 0xe8, 0xe9, 0xcb, 0x19, 0xcc,
                0xea, 0xd2, 0x62, 0xf6, 0x99, 0xfe, 0x1a, 0xf0, 0xa8, 0x7c, 0x4b, 0x56, 0x75, 0x31,
                0x66, 0x86, 0x0a, 0x23, 0x0b, 0xb5, 0xf9, 0xa2, 0xe8, 0xe3, 0x09, 0xa3, 0xee, 0x5e,
                0xfc, 0x87, 0x65, 0xe8, 0x86, 0x46, 0xd8, 0xa4, 0xd2, 0x6a, 0xd7, 0xbe, 0x3d, 0x3b,
                0xeb, 0xd8, 0xee, 0x25, 0xd5, 0x3a, 0xe4, 0xcf, 0xca, 0xad, 0x76, 0x0a, 0x2e, 0x51,
                0x67, 0x5a, 0x6e, 0xb9, 0xb9, 0x4c, 0xee, 0x14, 0xa0, 0x0a, 0x45, 0x18, 0x8f, 0x7c,
                0xcd, 0x6d, 0x71, 0x30, 0x3c, 0x91, 0x67, 0x1a, 0x77, 0x2f, 0x11, 0x3a, 0x18, 0x6c,
                0x87, 0xe3, 0x22, 0xb9, 0x8a, 0x6a, 0xc2, 0x91, 0x26, 0xe7, 0x6b, 0x0e, 0x2e, 0xe0,
                0x8b, 0x89, 0x07, 0x5c, 0x5b, 0x47, 0x6c, 0x63, 0x2d, 0x9d, 0x1a, 0x94, 0xa1, 0xea,
                0xc2, 0x4d, 0xd2, 0xb6, 0x9d, 0x04, 0xaa, 0xd7, 0x61, 0xb4, 0x8f, 0x45, 0xba, 0xbf,
                0xb9, 0x60, 0xaf, 0xac, 0x3a, 0x89, 0xb9, 0x32, 0x20, 0xe8, 0xa0, 0xc0, 0x01, 0x4c,
                0xda, 0xe4, 0x2c, 0x62, 0x7e, 0x58, 0xfd, 0xd1, 0x31, 0xf0, 0x43, 0x88, 0x19, 0x1c,
                0x4b, 0x53, 0x3e, 0x3c, 0x85, 0x7c, 0xbd, 0xfc, 0x64, 0x56, 0xc6, 0x4d, 0x7c, 0x35,
                0x97, 0xd6, 0x36, 0x61,
            ],
            &[
                0x47, 0xe3, 0x73, 0x08, 0x44, 0xbc, 0xb1, 0x00, 0x60, 0x75, 0xed, 0x84, 0xff, 0x7e,
                0xd2, 0xe8, 0x26, 0xd7, 0x46, 0x51, 0x57, 0x72, 0xdd, 0xc3, 0x6b, 0x5e, 0x11, 0xad,
                0x08, 0x7e, 0x75, 0xfc, 0x77, 0x6a, 0xfc, 0x13, 0xb4, 0x7d, 0xb6, 0x9e, 0x23, 0xb2,
                0x98, 0xba, 0x40, 0x45, 0xc2, 0xa1, 0x2b, 0xa4, 0xbf, 0x8c, 0xc3, 0x54, 0x94, 0xe7,
                0x6d, 0x2d, 0x86, 0xf8, 0x12, 0xf7, 0x6c, 0x5b, 0xc5, 0x0f, 0xf0, 0xaa, 0x36, 0xc1,
                0x5a, 0xaf, 0x7a, 0x36, 0x4a, 0x73, 0xe7, 0x1f, 0x69, 0x68, 0x11, 0xe7, 0x78, 0xd1,
                0x5a, 0x12, 0x76, 0x55, 0x19, 0xc9, 0xb4, 0x1b, 0xa9, 0x6e, 0x88, 0x5b, 0xb6, 0x50,
                0x19, 0x3d, 0x6e, 0x98, 0x50, 0x94, 0x02, 0x48, 0xcd, 0x98, 0xd5, 0x01, 0x92, 0x6f,
                0x15, 0xed, 0xfd, 0xa3, 0x28, 0x42, 0xb8, 0x9c, 0x16, 0x25, 0x70, 0x4a, 0x0c, 0x70,
                0x45, 0xc7, 0xda, 0x1e, 0x03, 0xed, 0x36, 0x98, 0x23, 0x1d, 0xbb, 0xa2, 0x51, 0x2f,
                0x37, 0x61, 0x55, 0xbf, 0x54, 0x1e, 0xf7, 0xd5, 0xb9, 0xfa, 0x33, 0x9f, 0x0d, 0x0d,
                0xe2, 0x12, 0x41, 0x26, 0x46, 0x8d, 0xee, 0x9d, 0x46, 0x06, 0xda, 0x17, 0xea, 0xec,
                0x59, 0x55, 0x8b, 0x4f, 0xa9, 0x59, 0x7e, 0xdd, 0x1b, 0xed, 0x45, 0x85, 0x06, 0xc3,
                0x1b, 0xed, 0x9f, 0x21, 0xd6, 0x13, 0xf8, 0x25, 0x0d, 0x98, 0x30, 0xaf, 0x95, 0xd2,
                0x4c, 0x74, 0xd2, 0x38, 0x30, 0xc6, 0x73, 0x37, 0xf5, 0x6e, 0x6c, 0x9c, 0x9c, 0x80,
                0x32, 0x68, 0x8d, 0x18, 0x34, 0xe4, 0x5b, 0xc2, 0x09, 0xbd, 0x5e, 0x19, 0xe4, 0x6c,
                0x32, 0x55, 0xbc, 0x63, 0x05, 0xcd, 0x61, 0xca, 0xab, 0xcf, 0x05, 0x99, 0xfb, 0xf1,
                0xc3, 0x07, 0xef, 0xb4, 0x34, 0xb3, 0xfb, 0x9c, 0xd3, 0x43, 0x6d, 0xab, 0x71, 0xf6,
                0x5f, 0xf0, 0xf6, 0x3d,
            ],
            &[
                0x85, 0x7a, 0xb1, 0xfe, 0x08, 0x90, 0x74, 0x33, 0x2a, 0xfa, 0x8a, 0x4c, 0x61, 0xd4,
                0x86, 0xed, 0x04, 0x30, 0xcf, 0x8d, 0x92, 0xa8, 0xff, 0x5f, 0xd6, 0xdb, 0x0c, 0x4b,
                0x8f, 0x92, 0xcf, 0x0c, 0xc4, 0xd9, 0x5d, 0xa9, 0x71, 0x9f, 0x69, 0xef, 0x9d, 0x1e,
                0x81, 0xad, 0x25, 0x3a, 0x20, 0x8a, 0x0e, 0x0a, 0x71, 0x2f, 0x16, 0x4f, 0x20, 0xa2,
                0xbe, 0x03, 0x28, 0xaa, 0x6e, 0x64, 0xde, 0x18, 0x44, 0x41, 0x81, 0xde, 0xbb, 0xd7,
                0x1c, 0x8a, 0x20, 0x7f, 0x1e, 0xed, 0xde, 0xab, 0x85, 0x99, 0xac, 0xa6, 0x87, 0xa1,
                0xf0, 0xa1, 0xe4, 0x4c, 0x15, 0x46, 0xba, 0xc7, 0x75, 0xc8, 0x7c, 0x2e, 0x2b, 0xdc,
                0x80, 0x43, 0x5e, 0x81, 0xe1, 0xe2, 0xb6, 0x72, 0xf5, 0x37, 0xe3, 0xe5, 0x6a, 0xe6,
                0x58, 0x07, 0xa0, 0x77, 0xe0, 0x7a, 0x33, 0x2e, 0x02, 0xe4, 0xbd, 0xc2, 0x39, 0x2f,
                0x07, 0xc2, 0x99, 0xcb, 0x2f, 0xa6, 0x89, 0x1a, 0xc4, 0xaf, 0xf4, 0x9f, 0xc1, 0x93,
                0x7a, 0x6d, 0xd2, 0x62, 0x9c, 0x53, 0x25, 0x3b, 0xac, 0x4d, 0x55, 0x89, 0x3c, 0x81,
                0x5e, 0xea, 0x4b, 0x1f, 0x63, 0x74, 0x86, 0x0f, 0xc5, 0x05, 0xfa, 0xa8, 0x58, 0x8f,
                0xe7, 0x3e, 0x53, 0xd2, 0x55, 0xeb, 0x90, 0x57, 0x39, 0xf9, 0x10, 0x94, 0xe3, 0x1c,
                0xab, 0xcf, 0x15, 0x9e, 0x2e, 0x09, 0xd6, 0x53, 0xbc, 0xd4, 0x9e, 0xe4, 0xfb, 0x5d,
                0x9b, 0x8f, 0xc0, 0x16, 0x4b, 0x95, 0x9f, 0x1d, 0x66, 0xb4, 0x83, 0xc3, 0xc5, 0xa9,
                0x15, 0x04, 0xc8, 0xb0, 0x8e, 0xf9, 0x46, 0x13, 0x52, 0x49, 0x39, 0x69, 0x3d, 0xc1,
                0x4c, 0xb8, 0x73, 0x62, 0x51, 0x94, 0xd3, 0x76, 0x86, 0xb7, 0x35, 0xa3, 0x9a, 0x96,
                0x27, 0x3d, 0x4c, 0x70, 0x73, 0xd7, 0x57, 0x4d, 0x1f, 0x40, 0x8c, 0xb2, 0x4a, 0x45,
                0x57, 0xb6, 0x15, 0xc5,
            ],
        ),
        (
            &[
                0xb9, 0xb4, 0x5e, 0x3d, 0x40, 0x4a, 0x62, 0xc7, 0xa3, 0x10, 0xc2, 0x0d, 0x8d, 0xd8,
                0x90, 0xa3, 0x4a, 0x20, 0x83, 0x8f, 0x74, 0x0e, 0x3c, 0xb9, 0x39, 0x05, 0x61, 0x56,
                0xa6, 0x37, 0x6f, 0x1b, 0xfa, 0x5f, 0xe7, 0xbd, 0x4d, 0x36, 0x93, 0x80, 0x35, 0x88,
                0xa1, 0xee, 0xf3, 0xbc, 0x37, 0x33, 0x89, 0xea, 0x18, 0x66, 0x65, 0x69, 0x36, 0x35,
                0xcc, 0x0a, 0x0b, 0x1d, 0x49, 0xbc, 0x5a, 0x39, 0x67, 0x9a, 0x74, 0xa1, 0x8f, 0xdd,
                0x3f, 0xa6, 0x3b, 0xac, 0xa1, 0x21, 0x81, 0xa0, 0xc7, 0xb9, 0x30, 0xf0, 0x11, 0x10,
                0xb5, 0xac, 0xf7, 0x20, 0x92, 0x97, 0x49, 0xa7, 0x37, 0x15, 0xc9, 0xd9, 0x8c, 0xae,
                0x8d, 0x7a, 0xa4, 0x9e, 0xaf, 0x73, 0xcd, 0x76, 0xd4, 0x57, 0xe4, 0x59, 0xac, 0x7a,
                0x9b, 0x8c, 0xf8, 0xfa, 0x3f, 0xa4, 0x30, 0x60, 0xa1, 0x13, 0xbc, 0xd6, 0x32, 0x72,
                0x50, 0xbc, 0x02, 0x7c, 0x2a, 0x5a, 0xe1, 0x50, 0x2e, 0x66, 0xd4, 0x91, 0xc8, 0x3a,
                0x28, 0x38, 0x41, 0xd4, 0x8a, 0xef, 0x04, 0xd5, 0x58, 0xb9, 0x02, 0xf9, 0x6d, 0x68,
                0xc1, 0x80, 0xc7, 0x96, 0x0f, 0xd0, 0x42, 0x22, 0x13, 0x8e, 0xa8, 0xf1, 0x7a, 0x4e,
                0xc5, 0xe3, 0xb8, 0x10, 0x81, 0x9f, 0x76, 0x7e, 0xb0, 0xb4, 0xfb, 0xc7, 0xd2, 0x1f,
                0xf1, 0x5e, 0x2c, 0x51, 0xbb, 0xf4, 0xe7, 0xe9, 0xd4, 0x1e, 0x23, 0x97, 0x14, 0xe6,
                0x07, 0x53, 0x25, 0x7a, 0xd7, 0xde, 0xb8, 0xab, 0x62, 0x23, 0x3e, 0x83, 0x16, 0xbf,
                0x89, 0xb1, 0x33, 0x0c, 0x78, 0x49, 0xa7, 0xd7, 0x51, 0xae, 0xd1, 0x05, 0x89, 0x9e,
                0xe5, 0x0f, 0x8a, 0x3c, 0x73, 0x18, 0x9d, 0xa5, 0xf1, 0xf3, 0xbb, 0x6a, 0xa4, 0x9f,
                0x9e, 0xc1, 0x2c, 0x0f, 0xb1, 0xfc, 0xe4, 0xe5, 0x13, 0x86, 0x19, 0x86, 0xb4, 0x39,
                0x6a, 0x70, 0xc6, 0xc1,
            ],
            &[
                0xab, 0x16, 0x0b, 0x04, 0x6f, 0x28, 0x98, 0xdc, 0xc7, 0xd4, 0x76, 0x93, 0x3a, 0x2d,
                0x5d, 0x03, 0xb3, 0x15, 0x45, 0x5f, 0x72, 0x52, 0x73, 0x8b, 0x49, 0x87, 0x35, 0x68,
                0x38, 0xf6, 0x35, 0x3d, 0x17, 0x6c, 0x27, 0xf9, 0xf5, 0x1a, 0xe4, 0xc5, 0x67, 0x8c,
                0x9b, 0x73, 0xa3, 0xc5, 0xb1, 0x2d, 0xa0, 0x4f, 0xb5, 0x6f, 0x10, 0xda, 0xdf, 0x80,
                0xac, 0x9c, 0x4c, 0x25, 0x0d, 0x7b, 0xa3, 0xbb, 0xe3, 0x41, 0x1f, 0x56, 0x81, 0x4e,
                0x1a, 0x87, 0xb1, 0xce, 0x97, 0x1c, 0x61, 0x6a, 0x98, 0xd6, 0x7a, 0xc9, 0x91, 0x4f,
                0x4d, 0xb1, 0x2e, 0x74, 0x29, 0xd9, 0x8b, 0x97, 0xac, 0x5e, 0x3c, 0x7a, 0x5a, 0xeb,
                0xad, 0x98, 0x61, 0xf5, 0x78, 0x3b, 0x3d, 0xfd, 0xce, 0x1f, 0xb3, 0x57, 0x12, 0x5a,
                0x5a, 0xd8, 0x83, 0xc1, 0x39, 0xc4, 0xb0, 0x75, 0x35, 0xb1, 0x13, 0x76, 0x5b, 0x3f,
                0x8a, 0x34, 0x39, 0xef, 0xf9, 0x1c, 0xa2, 0x4c, 0x7d, 0x93, 0x99, 0xbc, 0x7d, 0xbb,
                0xff, 0xf1, 0xf3, 0xd5, 0xe6, 0x87, 0x23, 0x48, 0x02, 0xf8, 0xee, 0x54, 0xab, 0x20,
                0x3d, 0xcc, 0x7e, 0x05, 0x22, 0x83, 0x4b, 0x1d, 0x55, 0x93, 0xcd, 0x4a, 0xe3, 0x51,
                0xb6, 0xcf, 0x58, 0x92, 0x26, 0x20, 0xd5, 0xa3, 0xde, 0x57, 0xe4, 0x89, 0x67, 0x4f,
                0xe0, 0x2a, 0x8e, 0x9c, 0xaf, 0xf6, 0x1a, 0x24, 0x65, 0x8b, 0x7c, 0x24, 0xfa, 0x51,
                0xcb, 0x0e, 0xd3, 0xe0, 0x3e, 0x83, 0x3e, 0xce, 0x93, 0x14, 0x7c, 0xac, 0x12, 0xd4,
                0x2e, 0x20, 0x0c, 0x21, 0xe4, 0x46, 0x3a, 0xab, 0xf0, 0xc6, 0x55, 0x77, 0xf4, 0xaf,
                0x81, 0x0e, 0x9c, 0xd8, 0xda, 0x87, 0xa7, 0xfa, 0xaf, 0x76, 0x5d, 0x43, 0xaf, 0x95,
                0xa5, 0x80, 0x21, 0x70, 0xee, 0xe5, 0x3b, 0x02, 0x7c, 0x67, 0x61, 0xde, 0x02, 0x5f,
                0x93, 0x07, 0x7e, 0x01,
            ],
            &[
                0x9d, 0x55, 0x61, 0x46, 0x1b, 0x3e, 0xb9, 0x34, 0x13, 0xdd, 0xec, 0x7f, 0x32, 0xff,
                0xe5, 0x5b, 0x9c, 0x37, 0xda, 0xe7, 0x35, 0x89, 0xf6, 0x72, 0x92, 0x0c, 0x27, 0xce,
                0xa4, 0x2f, 0x80, 0x9d, 0x48, 0xde, 0xfd, 0x91, 0x6d, 0x09, 0xba, 0x28, 0xd8, 0x3a,
                0x86, 0x2b, 0xa5, 0x8e, 0xa1, 0x1d, 0x74, 0x36, 0xd9, 0x94, 0x87, 0x1a, 0xef, 0xb4,
                0x6d, 0x56, 0xf4, 0x0e, 0x97, 0xc9, 0x3d, 0x73, 0x22, 0x3e, 0x0c, 0xcb, 0x83, 0xdb,
                0x2d, 0x90, 0x2d, 0x0a, 0x50, 0xf2, 0x97, 0x48, 0x36, 0xa5, 0x5f, 0xe2, 0x96, 0xdf,
                0x0d, 0x09, 0x70, 0x3e, 0x7e, 0x28, 0x21, 0xb4, 0xf4, 0x74, 0xa7, 0x5c, 0x76, 0x3e,
                0x7c, 0x43, 0x0b, 0x89, 0x1e, 0x2f, 0x29, 0x3a, 0x4a, 0x01, 0xdf, 0x15, 0x3f, 0x39,
                0xc5, 0x57, 0xf9, 0x8a, 0x93, 0x17, 0xd9, 0x39, 0xea, 0x4e, 0x8c, 0xfd, 0x94, 0xfa,
                0xae, 0x31, 0x38, 0xae, 0x07, 0xe4, 0x2a, 0x8d, 0xc9, 0xf6, 0xb0, 0x2f, 0x42, 0x40,
                0x67, 0x82, 0x30, 0xdc, 0x69, 0x7c, 0x16, 0xa6, 0x57, 0x08, 0x5c, 0xbd, 0x69, 0xfb,
                0xf0, 0x69, 0xfa, 0x06, 0x69, 0x77, 0x32, 0x95, 0x24, 0x0a, 0xf0, 0x9a, 0xb4, 0xa3,
                0xce, 0x84, 0xc6, 0x6d, 0xdf, 0x75, 0xb3, 0x01, 0x5d, 0x50, 0x10, 0x74, 0x62, 0xdd,
                0x5b, 0x19, 0x71, 0xcb, 0xf7, 0xbd, 0xa8, 0x6f, 0xd3, 0xb5, 0x60, 0xd2, 0x90, 0x62,
                0x1c, 0x0c, 0xe9, 0xf9, 0xe6, 0x77, 0x72, 0x9a, 0x0f, 0xeb, 0x3a, 0xaf, 0xca, 0x13,
                0xb6, 0xc1, 0xf0, 0xb3, 0xcd, 0x30, 0x3b, 0x34, 0xad, 0xcc, 0x19, 0x87, 0x8a, 0xaf,
                0xf2, 0xa8, 0x96, 0x80, 0xea, 0xda, 0xbd, 0xb8, 0xf6, 0x2f, 0xe5, 0xe0, 0x85, 0xf5,
                0x59, 0x80, 0x26, 0xa6, 0x11, 0xfb, 0x43, 0xe8, 0xe6, 0xcc, 0xdd, 0x86, 0x77, 0x32,
                0x10, 0x09, 0x26, 0x56,
            ],
        ),
        // RSA4096
        (
            &[
                0xac, 0x1b, 0x1c, 0x45, 0x5e, 0x91, 0x59, 0x14, 0x06, 0x35, 0xd5, 0xba, 0xf0, 0xde,
                0x7c, 0x8d, 0x4d, 0x67, 0xc0, 0x0e, 0xf5, 0x30, 0x62, 0x6c, 0xef, 0xc0, 0xa9, 0xc0,
                0x1d, 0xb0, 0x5c, 0xcb, 0x1d, 0xff, 0x9b, 0xe8, 0x23, 0x3e, 0xf6, 0xec, 0xfe, 0xb7,
                0x88, 0x0b, 0x5a, 0x43, 0x98, 0xdc, 0x52, 0x32, 0x14, 0x0c, 0xea, 0x7e, 0x2e, 0xbe,
                0xe2, 0xb2, 0xea, 0x48, 0xe1, 0xca, 0x04, 0xa6, 0x40, 0x13, 0x23, 0x88, 0xda, 0xbf,
                0x9b, 0x51, 0x77, 0x67, 0x15, 0xef, 0x55, 0x09, 0xaf, 0xc1, 0xf6, 0x85, 0x68, 0x6a,
                0xeb, 0x1b, 0x95, 0x62, 0x38, 0x95, 0x91, 0xd2, 0x19, 0x23, 0xa2, 0x18, 0xd4, 0xd1,
                0x7f, 0x9b, 0x53, 0xab, 0x2f, 0x56, 0x3e, 0xd8, 0x0e, 0x9f, 0x7e, 0xe3, 0xd0, 0x2e,
                0xbf, 0x9a, 0x6f, 0xf9, 0xc3, 0xa4, 0x68, 0x83, 0x43, 0x02, 0x3d, 0xc9, 0xf5, 0x2b,
                0xb5, 0x17, 0xe7, 0xe4, 0xcb, 0x43, 0x83, 0x1a, 0xe3, 0xad, 0xe3, 0x64, 0xf5, 0x33,
                0xd6, 0xb3, 0x53, 0x43, 0xc6, 0x38, 0x13, 0xb7, 0xa6, 0x60, 0xf0, 0x2f, 0x85, 0x94,
                0xcf, 0xe2, 0x47, 0x9b, 0x2e, 0x6b, 0x44, 0x62, 0x0a, 0x7e, 0xec, 0xdf, 0x09, 0x78,
                0x83, 0x29, 0xcd, 0x00, 0x6d, 0xbe, 0x1a, 0x1b, 0xe7, 0xbb, 0x72, 0x3f, 0xb3, 0x3f,
                0xcc, 0x76, 0xa1, 0x1d, 0x18, 0x13, 0x60, 0xbc, 0x10, 0xd1, 0x46, 0x23, 0x0c, 0xb3,
                0xf0, 0x00, 0x2d, 0xf4, 0x1b, 0x31, 0xcc, 0x95, 0x3b, 0x3c, 0x8b, 0x01, 0x0d, 0xca,
                0x1a, 0x81, 0xc5, 0x04, 0xda, 0xad, 0xab, 0x14, 0x3f, 0xd2, 0x73, 0x39, 0xd3, 0x4b,
                0xa0, 0xfd, 0xf9, 0xf5, 0x5c, 0x5b, 0x3e, 0xaa, 0xd8, 0xc5, 0x23, 0x6c, 0xfc, 0xff,
                0x71, 0xcc, 0x3b, 0xe8, 0x74, 0x33, 0xed, 0xcf, 0x3b, 0x4a, 0xab, 0xe5, 0x0b, 0x69,
                0x3a, 0xce, 0x03, 0x2e, 0xda, 0x6a, 0x5c, 0x46, 0x36, 0xb2, 0x6f, 0xcf, 0x89, 0x66,
                0xcb, 0x7b, 0x83, 0xe5, 0x78, 0x11, 0xcd, 0x1d, 0xc6, 0xa2, 0x21, 0xf1, 0x81, 0x5b,
                0xf4, 0xfe, 0x59, 0xea, 0x88, 0xc8, 0x3e, 0x11, 0xd4, 0x9d, 0xf8, 0x70, 0xe8, 0x39,
                0x61, 0x6d, 0x17, 0xe8, 0xbc, 0x19, 0x1f, 0xc2, 0xd6, 0x85, 0x5d, 0x80, 0x1d, 0x37,
                0xe4, 0xea, 0x95, 0xfe, 0xa5, 0x21, 0x81, 0xdc, 0xeb, 0x1f, 0x80, 0xe1, 0xba, 0x7c,
                0xad, 0xb2, 0x89, 0x9a, 0x2a, 0x6e, 0x72, 0xab, 0x44, 0x3c, 0x5a, 0x99, 0x5f, 0x5b,
                0xd1, 0x7f, 0x56, 0xbc, 0x39, 0x99, 0x28, 0x32, 0x13, 0xb0, 0x3e, 0x17, 0x61, 0x3d,
                0xc5, 0x1b, 0x6a, 0x32, 0x88, 0x3c, 0x06, 0x07, 0x0c, 0x61, 0x6e, 0x90, 0x67, 0x61,
                0xf3, 0x7d, 0x21, 0x5e, 0x2b, 0x00, 0x1a, 0x1f, 0xd5, 0x16, 0x90, 0xce, 0x1a, 0x12,
                0x0f, 0x3a, 0x0c, 0x82, 0xce, 0xe0, 0x7b, 0x46, 0x63, 0xf4, 0xb9, 0xae, 0xcc, 0xa5,
                0x65, 0xcb, 0xa0, 0x38, 0xf7, 0x93, 0x31, 0x74, 0x1b, 0x3c, 0x85, 0xf9, 0x2b, 0x98,
                0xef, 0xee, 0x9a, 0x74, 0x6c, 0x34, 0x41, 0x7e, 0x38, 0x86, 0xbe, 0x25, 0xf9, 0x41,
                0xbd, 0x9e, 0x34, 0x3a, 0xe9, 0x24, 0x7c, 0x1a, 0x27, 0xe7, 0x36, 0x9f, 0x03, 0x0b,
                0x64, 0x76, 0xf5, 0xb0, 0xf3, 0xb1, 0xdb, 0x61, 0x30, 0x6b, 0xa3, 0x3a, 0x8e, 0x4b,
                0x01, 0xa6, 0x48, 0x08, 0xdb, 0x66, 0xb4, 0x3d, 0xbb, 0x28, 0x80, 0x90, 0xcb, 0x1b,
                0x20, 0x20, 0x99, 0xce, 0xb5, 0x85, 0x11, 0xd5, 0x51, 0x81, 0xa1, 0x97, 0x76, 0x71,
                0xbe, 0x1c, 0xc4, 0x51, 0x82, 0xbe, 0xbd, 0x8f, 0xe6, 0xd9, 0x94, 0xee, 0x59, 0xb6,
                0x29, 0xd1, 0x12, 0xa6, 0x03, 0x30, 0xf9, 0x91, 0x67, 0xea, 0x95, 0xc7, 0x34, 0xa5,
                0x42, 0xbd, 0x6a, 0x97, 0x62, 0xf0, 0xc6, 0xb9,
            ],
            &[
                0x85, 0xb9, 0x16, 0xd1, 0x4b, 0x76, 0x31, 0xb9, 0x5e, 0x4d, 0xec, 0x00, 0x31, 0x71,
                0x1d, 0x63, 0x89, 0x16, 0x28, 0xe3, 0x36, 0x5d, 0x5e, 0xcc, 0x77, 0xc8, 0xc1, 0xdc,
                0x54, 0xf5, 0x18, 0x54, 0x75, 0xbd, 0x8a, 0x7c, 0xe7, 0x0d, 0xe0, 0x3c, 0x2a, 0x79,
                0x9d, 0xc9, 0xfc, 0x5b, 0x73, 0x65, 0x14, 0xb4, 0x76, 0x61, 0xc6, 0xbd, 0x3e, 0x42,
                0xf0, 0xcf, 0xc5, 0x3b, 0xd5, 0xbb, 0xea, 0xba, 0xe6, 0x24, 0x38, 0xc2, 0xf7, 0xfc,
                0x52, 0x89, 0x0c, 0xf6, 0x5a, 0xd3, 0xb7, 0xc6, 0x2b, 0xfa, 0xd0, 0x39, 0xbd, 0xf4,
                0xfd, 0x32, 0x54, 0x72, 0x99, 0xb6, 0x95, 0x33, 0xa2, 0x76, 0xce, 0x56, 0xee, 0xdc,
                0xcc, 0x82, 0x7a, 0x93, 0x12, 0xd7, 0xb7, 0x42, 0x96, 0xb2, 0x14, 0x9c, 0x9b, 0xc0,
                0x06, 0xfd, 0xcf, 0x2d, 0x48, 0x76, 0xf1, 0x01, 0xb4, 0x4a, 0x04, 0x8a, 0x6b, 0xe5,
                0x86, 0xf2, 0xc9, 0x2a, 0x3b, 0x48, 0xfc, 0x90, 0x23, 0x01, 0x94, 0x22, 0x78, 0x66,
                0xcb, 0xd6, 0x4f, 0xc6, 0xe4, 0x37, 0xe4, 0x12, 0x18, 0xc3, 0x4f, 0x3c, 0x0d, 0x55,
                0x9f, 0xe4, 0x98, 0x70, 0x99, 0xb3, 0x9c, 0xd1, 0x74, 0x88, 0x44, 0x31, 0xd4, 0x21,
                0x00, 0x00, 0xb5, 0x99, 0x34, 0xab, 0xb5, 0x8b, 0xa3, 0x15, 0x40, 0xe2, 0xff, 0xba,
                0x7d, 0x7a, 0x5b, 0x1a, 0xc8, 0xff, 0x1d, 0x75, 0x62, 0xb7, 0xc8, 0x00, 0x29, 0xb9,
                0x91, 0xac, 0x02, 0x08, 0x9e, 0x2c, 0xa6, 0x61, 0xf9, 0x41, 0xc4, 0x5b, 0x90, 0x44,
                0x4d, 0x2e, 0x31, 0x4f, 0xe1, 0x3d, 0x79, 0x24, 0xe5, 0xa5, 0xf6, 0x03, 0xa3, 0x0a,
                0x2b, 0x4f, 0xcb, 0x2c, 0x7a, 0x93, 0x26, 0xf1, 0x06, 0x36, 0xbe, 0x9c, 0xcb, 0x43,
                0x0b, 0x3f, 0x7d, 0xf8, 0xd2, 0x79, 0x1a, 0xb8, 0x9e, 0xc0, 0x19, 0x29, 0x81, 0x21,
                0x45, 0x20, 0x96, 0xd5, 0x45, 0x50, 0xf1, 0x2a, 0xb3, 0xe9, 0x98, 0x0c, 0x86, 0x27,
                0xde, 0x0a, 0x7a, 0x2d, 0xdd, 0xf6, 0x42, 0xb9, 0xda, 0x8a, 0x22, 0x17, 0xbd, 0x6c,
                0x72, 0xda, 0xf7, 0xea, 0x67, 0x78, 0x48, 0xe6, 0x33, 0xd0, 0x00, 0xd7, 0x72, 0xe4,
                0xb8, 0x81, 0x66, 0xea, 0x1a, 0x9f, 0x4b, 0x45, 0xbe, 0x92, 0xa1, 0xea, 0x77, 0xd7,
                0xc6, 0xba, 0x7b, 0x27, 0x0c, 0x52, 0x24, 0xad, 0xd0, 0x14, 0xab, 0xf7, 0xe5, 0x42,
                0xc3, 0x2e, 0xe5, 0x3d, 0x23, 0x2c, 0x11, 0xdc, 0x89, 0x15, 0xd7, 0x61, 0xce, 0x6c,
                0xce, 0x16, 0x44, 0xc4, 0xf7, 0xe4, 0x6c, 0x3c, 0xea, 0x63, 0xf7, 0x43, 0x72, 0x78,
                0xcd, 0xab, 0xf7, 0x2e, 0xbe, 0xa5, 0x6a, 0x72, 0x9d, 0x91, 0x2b, 0x2a, 0xe3, 0xdd,
                0x5c, 0x55, 0xe0, 0xee, 0x8b, 0x05, 0x27, 0xfb, 0x16, 0xde, 0x72, 0x9e, 0x40, 0xec,
                0x67, 0xa1, 0x0a, 0xed, 0x9b, 0xde, 0x79, 0x48, 0x8d, 0x20, 0x75, 0x5b, 0x10, 0xd7,
                0xe6, 0xd4, 0x38, 0x1a, 0x85, 0xc0, 0xdf, 0xd2, 0x31, 0x64, 0x88, 0x50, 0xeb, 0xa2,
                0x63, 0x33, 0x92, 0xb7, 0xbf, 0x98, 0xb1, 0x7d, 0x16, 0xe5, 0x1e, 0x86, 0x33, 0xf5,
                0x57, 0x48, 0x49, 0x08, 0xb0, 0xb8, 0x6b, 0xf8, 0xc5, 0x55, 0xb2, 0xf4, 0x46, 0x1e,
                0x0c, 0xad, 0xe8, 0x55, 0x5d, 0xee, 0x67, 0xea, 0xbf, 0x3a, 0x2e, 0xb3, 0xf3, 0x15,
                0xf8, 0x9f, 0xe6, 0x47, 0xb6, 0x82, 0x2f, 0x60, 0x90, 0xf6, 0x30, 0x6b, 0x8d, 0xee,
                0x39, 0x20, 0x39, 0xf0, 0x64, 0xfd, 0x7e, 0xfb, 0x05, 0x92, 0xb4, 0xbe, 0x39, 0x13,
                0xeb, 0x0d, 0x8d, 0x5e, 0xee, 0xf0, 0xf1, 0x19, 0x34, 0x0f, 0xde, 0x8d, 0x68, 0x7b,
                0x49, 0x9d, 0xb8, 0xe2, 0xa7, 0x75, 0x65, 0xc7, 0xe2, 0x1a, 0xc1, 0x9a, 0x5f, 0x73,
                0x51, 0xc4, 0x55, 0xa3, 0x5f, 0x92, 0x63, 0x95,
            ],
            &[
                0x5f, 0xcb, 0xe7, 0x42, 0x4c, 0xe1, 0x44, 0x0e, 0x5c, 0xe8, 0xfa, 0x52, 0x65, 0xff,
                0xea, 0x14, 0x57, 0x79, 0xe3, 0x6a, 0x2f, 0x89, 0xfa, 0xdc, 0x06, 0x48, 0xbe, 0x79,
                0xf8, 0x54, 0x25, 0x52, 0x2c, 0xd6, 0x0b, 0x69, 0xb6, 0x83, 0x63, 0xc5, 0xd0, 0x63,
                0x61, 0xe2, 0x31, 0xb0, 0xc2, 0x50, 0x8f, 0xce, 0x27, 0x1f, 0xab, 0x50, 0xef, 0xc2,
                0x1c, 0x63, 0x44, 0x07, 0x86, 0x7b, 0xb8, 0x67, 0x34, 0x90, 0x7e, 0x60, 0xd2, 0xe5,
                0x1e, 0x9b, 0xa5, 0x44, 0x4d, 0x0d, 0x32, 0x83, 0x9f, 0x18, 0xb4, 0x80, 0x7e, 0xa6,
                0xd5, 0x2d, 0xff, 0x8c, 0xb9, 0x0d, 0x36, 0xee, 0xc4, 0x5d, 0xd1, 0x55, 0xec, 0x64,
                0xeb, 0x46, 0xa7, 0x67, 0x83, 0x6c, 0x85, 0x39, 0x06, 0xbc, 0xf2, 0x00, 0x0f, 0xd4,
                0xb6, 0x03, 0xdc, 0xb2, 0xe6, 0x72, 0xa0, 0x33, 0x0c, 0x24, 0x67, 0x61, 0xa2, 0x76,
                0x54, 0x5e, 0x20, 0xd1, 0xfc, 0x82, 0x7d, 0x64, 0x54, 0xbd, 0x22, 0xdf, 0x5b, 0xf0,
                0x1d, 0x4c, 0x15, 0x4a, 0x11, 0x33, 0xac, 0x28, 0x33, 0x57, 0x08, 0x47, 0xf3, 0x1b,
                0x67, 0xac, 0x80, 0x92, 0xc7, 0x1d, 0x3b, 0xea, 0xb3, 0x1f, 0xef, 0x45, 0x23, 0x0e,
                0x10, 0xb0, 0xba, 0x0a, 0x7b, 0x44, 0x10, 0xee, 0xfa, 0xe4, 0xc4, 0xf6, 0xef, 0x53,
                0x78, 0x41, 0x0d, 0x29, 0xd4, 0x39, 0x8e, 0x2d, 0xbf, 0xc3, 0xd7, 0x71, 0xf1, 0x80,
                0xfb, 0xa9, 0xbd, 0x53, 0x59, 0xd4, 0x9c, 0x06, 0x2e, 0x66, 0x67, 0xa6, 0x1a, 0x17,
                0x51, 0x1b, 0xc7, 0x6f, 0xdb, 0xe7, 0xa1, 0xf3, 0x07, 0x4f, 0x07, 0x2b, 0x0e, 0x73,
                0x98, 0x0f, 0x1b, 0xf0, 0xe5, 0x5e, 0x78, 0xe7, 0xcf, 0x4e, 0x09, 0xfe, 0xcd, 0xf8,
                0xba, 0x64, 0x33, 0x25, 0x24, 0xec, 0xd7, 0xa8, 0xc6, 0x20, 0xb3, 0x37, 0x34, 0x48,
                0x84, 0xd5, 0x40, 0xcd, 0xa0, 0x37, 0x6c, 0x07, 0xa4, 0x1d, 0x99, 0xaf, 0x27, 0x54,
                0xfa, 0xb8, 0x77, 0x84, 0x5f, 0x27, 0xe7, 0x6c, 0x8e, 0x10, 0x68, 0x75, 0x52, 0x43,
                0xde, 0x81, 0xbe, 0x33, 0xad, 0x3f, 0x34, 0x16, 0x8b, 0x53, 0x46, 0xe9, 0xa3, 0x00,
                0x0f, 0x81, 0x02, 0xed, 0x30, 0xdd, 0x84, 0x53, 0x22, 0xd6, 0xdc, 0xf7, 0x1f, 0x11,
                0xf8, 0xd5, 0x23, 0xcb, 0x92, 0xe9, 0x70, 0xc9, 0xde, 0xd2, 0x3b, 0x7e, 0xd2, 0x3b,
                0x35, 0xd4, 0xff, 0xa5, 0x81, 0xbc, 0x7c, 0x73, 0x0d, 0xf6, 0xad, 0x4c, 0xc0, 0x27,
                0x75, 0x2f, 0xf4, 0xb1, 0x03, 0x52, 0x55, 0x45, 0x3b, 0xfb, 0x50, 0x6b, 0x3d, 0x18,
                0x5a, 0x0b, 0xc3, 0x16, 0x47, 0xb5, 0x29, 0x99, 0x7a, 0xed, 0xe1, 0x7a, 0xc8, 0x2f,
                0xaa, 0xee, 0x4c, 0xfa, 0xca, 0x44, 0x53, 0x5c, 0x1f, 0x7a, 0xb8, 0x27, 0x7c, 0xf4,
                0xd2, 0xc7, 0xd8, 0xd5, 0xae, 0x56, 0x45, 0x52, 0xd6, 0x7e, 0x8b, 0x36, 0x05, 0xd4,
                0xa3, 0x70, 0xe2, 0xa1, 0x04, 0x3d, 0xde, 0x0c, 0x2e, 0x49, 0x73, 0xe5, 0xbc, 0x7b,
                0x1b, 0xc3, 0xba, 0xb9, 0x37, 0xb9, 0x35, 0x13, 0x88, 0xce, 0x1f, 0x86, 0xa1, 0xed,
                0x8e, 0x16, 0x8b, 0xeb, 0x8f, 0x01, 0x2a, 0x54, 0x0a, 0x30, 0xe4, 0x7f, 0x5e, 0x13,
                0x4e, 0xe9, 0x0c, 0x60, 0xb4, 0xcc, 0xec, 0x00, 0x5e, 0x96, 0x47, 0x48, 0xe5, 0x03,
                0xf6, 0xba, 0x1b, 0xac, 0x9f, 0x45, 0xcd, 0x19, 0x8c, 0x9e, 0x21, 0x1b, 0x85, 0xec,
                0x56, 0x0c, 0xe9, 0xad, 0xb6, 0xd0, 0xfa, 0x0a, 0xd3, 0x64, 0x60, 0xc8, 0xa1, 0x16,
                0x58, 0x7b, 0xfc, 0x48, 0x2c, 0xaf, 0x48, 0xfe, 0xa9, 0x2d, 0x44, 0xa7, 0xc6, 0x59,
                0x11, 0xd3, 0xc9, 0xb7, 0x52, 0xbb, 0x00, 0x58, 0xef, 0xe2, 0x47, 0x0b, 0x7e, 0x68,
                0x37, 0xe6, 0x71, 0xfa, 0xe5, 0x93, 0x59, 0x21,
            ],
        ),
        (
            &[
                0xbf, 0xc9, 0xf7, 0x2a, 0xf7, 0x03, 0x9e, 0x40, 0x52, 0xe7, 0x52, 0x78, 0x46, 0x98,
                0x82, 0x27, 0x76, 0x2b, 0x11, 0x83, 0xea, 0xbf, 0xec, 0x82, 0x98, 0xf7, 0xd5, 0xdf,
                0x9d, 0xbc, 0x87, 0xdd, 0xc6, 0x08, 0xa1, 0x8f, 0x3d, 0x72, 0x45, 0x3e, 0xe4, 0x93,
                0x8f, 0x10, 0x15, 0x7f, 0x80, 0xdb, 0xd2, 0x65, 0x5d, 0x96, 0x33, 0x10, 0x60, 0x76,
                0xeb, 0x90, 0x9a, 0xf2, 0x7a, 0xed, 0x7d, 0x49, 0xc3, 0x02, 0x41, 0x34, 0xde, 0xf1,
                0xee, 0xe0, 0x7d, 0x36, 0xf1, 0x71, 0x8d, 0x15, 0xeb, 0xfd, 0xd5, 0x94, 0xac, 0xfb,
                0x95, 0x19, 0x83, 0x4c, 0xbd, 0xc7, 0xbe, 0x4b, 0x4a, 0x41, 0xe8, 0xdc, 0x0d, 0x72,
                0x00, 0x46, 0xa2, 0x5f, 0x57, 0x67, 0xb2, 0x36, 0xf5, 0x0d, 0xf8, 0x5c, 0xe4, 0xe3,
                0xe5, 0xea, 0xc1, 0x38, 0x46, 0x36, 0xa5, 0x7e, 0xba, 0xb2, 0xda, 0xcc, 0xb6, 0xf2,
                0xf1, 0xee, 0xb9, 0x57, 0x54, 0x07, 0x6a, 0xf7, 0xb2, 0xcb, 0x71, 0xd8, 0xcd, 0x86,
                0x29, 0x96, 0x75, 0x5e, 0xb1, 0x90, 0xe2, 0xc0, 0x14, 0xdd, 0x8b, 0x5d, 0x5b, 0xe2,
                0x32, 0x51, 0xb5, 0xce, 0x9e, 0xd8, 0xd2, 0x71, 0xba, 0x0f, 0x91, 0x6b, 0xb0, 0xe1,
                0x7e, 0xdc, 0x52, 0x5f, 0xf7, 0x64, 0x28, 0x39, 0xa5, 0x1a, 0xfd, 0x14, 0x89, 0xfe,
                0x05, 0x1c, 0x1d, 0x6a, 0x84, 0x82, 0x5d, 0xb4, 0xaa, 0x49, 0x50, 0x9f, 0xbf, 0x2a,
                0xed, 0xb6, 0x57, 0x99, 0x6c, 0x01, 0x3e, 0x9f, 0xa3, 0xe1, 0x18, 0x5d, 0x86, 0x85,
                0x64, 0xca, 0xe1, 0xd1, 0x73, 0x18, 0x9e, 0xd7, 0xfa, 0x40, 0x2a, 0x1e, 0xf8, 0x75,
                0x0b, 0xe7, 0xb8, 0xed, 0xd7, 0x06, 0x8e, 0x56, 0x0c, 0xa1, 0xeb, 0xc4, 0xb9, 0x5b,
                0x9b, 0x24, 0x5d, 0xe4, 0x8c, 0x66, 0xb7, 0x4c, 0xd0, 0x6f, 0xdb, 0xd8, 0x40, 0x77,
                0xd7, 0xbb, 0xc9, 0x51, 0x33, 0x82, 0x53, 0x8d, 0xb4, 0xde, 0xd1, 0xe5, 0x03, 0xda,
                0x62, 0x50, 0x66, 0x2b, 0x03, 0x7e, 0x07, 0x48, 0xc5, 0x20, 0x82, 0x7e, 0x93, 0xe7,
                0x2c, 0xc2, 0x1e, 0xf3, 0xdb, 0x94, 0x8a, 0x98, 0x4c, 0x32, 0x36, 0x56, 0x74, 0xed,
                0xda, 0x9e, 0xfd, 0xa7, 0xbf, 0x5f, 0x71, 0x8c, 0x8c, 0x64, 0x6e, 0xee, 0xc7, 0xd1,
                0x97, 0x68, 0xe3, 0xce, 0x23, 0x18, 0xfc, 0x90, 0xaa, 0x53, 0xc2, 0x63, 0x98, 0x64,
                0x80, 0x02, 0x70, 0xcf, 0xe4, 0x10, 0x79, 0x8f, 0x52, 0xb5, 0x04, 0xfd, 0x07, 0xdb,
                0xa3, 0xef, 0xd4, 0x71, 0x13, 0x50, 0x59, 0x16, 0x75, 0xa4, 0xf7, 0xe8, 0x34, 0x3a,
                0xa9, 0x31, 0x97, 0x85, 0x81, 0x1b, 0x7b, 0x23, 0x01, 0x05, 0x57, 0xf5, 0x68, 0xcc,
                0x44, 0x86, 0x5d, 0xff, 0xe3, 0xc6, 0x8f, 0xd1, 0x4a, 0x85, 0x15, 0xeb, 0x42, 0x41,
                0x14, 0x86, 0x90, 0x73, 0xec, 0x7b, 0x1c, 0xcc, 0x40, 0x8b, 0xee, 0x05, 0x1a, 0xd0,
                0x91, 0xfc, 0xc7, 0x86, 0xfd, 0xc4, 0xb2, 0x13, 0xb1, 0x85, 0xb1, 0x47, 0x1a, 0x7f,
                0xe0, 0x1e, 0x0c, 0xdd, 0x46, 0xdd, 0x1d, 0x30, 0x8e, 0x45, 0x45, 0xf0, 0x6b, 0x07,
                0x28, 0x2e, 0x58, 0x58, 0xf7, 0xb7, 0xb4, 0x13, 0xe4, 0xfc, 0xbf, 0x44, 0x52, 0x85,
                0x20, 0x31, 0x31, 0x5e, 0xf9, 0x3c, 0x65, 0xec, 0x98, 0x6b, 0x9b, 0x56, 0x34, 0xab,
                0xb7, 0x2f, 0xa6, 0x7a, 0xc3, 0x03, 0x04, 0x94, 0xed, 0x12, 0xb9, 0xe3, 0xac, 0xb5,
                0xc8, 0x6f, 0x96, 0x5c, 0x97, 0x63, 0xaf, 0x6f, 0x6a, 0x91, 0x1a, 0x2f, 0xd0, 0xc6,
                0x52, 0x55, 0x8c, 0x04, 0x6c, 0xf9, 0xf1, 0x72, 0x0f, 0xf8, 0x94, 0xfc, 0x6c, 0xbe,
                0x28, 0xa1, 0x0c, 0xda, 0x74, 0x00, 0xf6, 0xf3, 0x08, 0xa4, 0xdf, 0x53, 0x76, 0x89,
                0x52, 0xc4, 0x02, 0x84, 0x8f, 0xe3, 0xb6, 0x03,
            ],
            &[
                0x23, 0x54, 0xa8, 0x64, 0xd0, 0xd6, 0x68, 0xcb, 0xbe, 0xba, 0x00, 0x76, 0x49, 0xc3,
                0x04, 0x8f, 0x12, 0x74, 0xc2, 0xa8, 0x43, 0x91, 0x91, 0x97, 0x49, 0x68, 0xb6, 0x8c,
                0x98, 0x39, 0x47, 0xea, 0x31, 0xf6, 0x1b, 0x15, 0x11, 0x23, 0xc0, 0xdf, 0xe2, 0x29,
                0xd0, 0xbc, 0x0c, 0xc9, 0xcd, 0x4a, 0x31, 0x8b, 0x1c, 0xdf, 0x73, 0x8e, 0xbb, 0xc6,
                0x8c, 0x84, 0xba, 0x16, 0x9b, 0x50, 0xae, 0xb8, 0xec, 0xe4, 0xb8, 0x70, 0x6d, 0xf5,
                0xb1, 0xa4, 0xc7, 0x4c, 0x5c, 0xd4, 0x27, 0x42, 0x77, 0x93, 0xee, 0x49, 0x92, 0x48,
                0x52, 0x62, 0x3d, 0xce, 0xe0, 0x53, 0x30, 0x9a, 0x1c, 0x16, 0xe2, 0x37, 0xcf, 0x7e,
                0x45, 0xd0, 0xbd, 0x4e, 0xc5, 0x02, 0x44, 0x51, 0x5d, 0x79, 0x72, 0x5c, 0x62, 0x8a,
                0x1d, 0x2b, 0xce, 0xe6, 0x78, 0x00, 0xcf, 0x21, 0xf6, 0x70, 0xc6, 0x5f, 0xda, 0x00,
                0x0d, 0x53, 0x85, 0xef, 0x31, 0x7a, 0xa0, 0x58, 0xfb, 0x26, 0x01, 0x56, 0x08, 0x1e,
                0x84, 0x00, 0xc4, 0xa4, 0x6a, 0x1f, 0x9f, 0xb5, 0xf4, 0xe2, 0x0f, 0x2f, 0x66, 0xa2,
                0xd7, 0xd4, 0x37, 0xa2, 0xd5, 0x9e, 0x69, 0xbe, 0x2b, 0xa1, 0x7c, 0x8f, 0x93, 0x29,
                0x27, 0x3e, 0x9d, 0x2a, 0x32, 0x9f, 0xcf, 0xcd, 0x36, 0xbe, 0x2f, 0x0b, 0x1e, 0x94,
                0x9e, 0x0a, 0x5c, 0xdc, 0xe7, 0x86, 0x40, 0x8e, 0xec, 0xa3, 0xce, 0xe7, 0x6e, 0xc7,
                0x10, 0xbd, 0x7b, 0x8b, 0xb6, 0xda, 0xcf, 0xd7, 0x86, 0xd0, 0x0b, 0xb6, 0x06, 0xf7,
                0x01, 0xe7, 0x62, 0x0c, 0x3c, 0xa0, 0xb7, 0x7a, 0x60, 0x0e, 0x7b, 0xf3, 0xf2, 0x9f,
                0x55, 0x4f, 0x1d, 0xc1, 0x2a, 0xd0, 0x79, 0x5e, 0x1e, 0xbb, 0xa1, 0x7c, 0x3f, 0x0d,
                0x42, 0x1a, 0x43, 0xf5, 0xbb, 0x6b, 0x9c, 0xae, 0xd8, 0xe3, 0x12, 0x63, 0xd7, 0x14,
                0x7c, 0xb8, 0x8a, 0x50, 0x4a, 0x02, 0x99, 0x4f, 0xcf, 0x9c, 0x8c, 0x73, 0x69, 0xd5,
                0x82, 0x52, 0x3e, 0xeb, 0x40, 0xc8, 0x55, 0xb6, 0xde, 0xaa, 0x98, 0x79, 0x07, 0x01,
                0x4b, 0x0d, 0x37, 0x6e, 0x5a, 0xbc, 0xb1, 0x70, 0x53, 0x64, 0x6d, 0x9a, 0xfa, 0x22,
                0x06, 0x94, 0x1b, 0x11, 0x14, 0x6d, 0xec, 0x81, 0x25, 0x24, 0x82, 0x4d, 0x8f, 0x4d,
                0x41, 0x45, 0xd5, 0xd5, 0x3c, 0x26, 0xb4, 0x32, 0xbf, 0x59, 0x7a, 0x9c, 0x05, 0x15,
                0x10, 0x81, 0x7a, 0xc6, 0x56, 0x93, 0x5d, 0xf0, 0xfa, 0xed, 0xd4, 0x6b, 0x4b, 0xa2,
                0xef, 0xe1, 0x4f, 0x78, 0xe2, 0xab, 0x09, 0x7c, 0xe1, 0x4b, 0xbc, 0x5b, 0xfb, 0x04,
                0x0a, 0xd2, 0xf8, 0x18, 0x52, 0x3d, 0xa1, 0xff, 0x78, 0x2e, 0x7f, 0xb1, 0x77, 0xbb,
                0x1f, 0x1b, 0xfe, 0x34, 0x26, 0x14, 0x36, 0xf3, 0xab, 0xfe, 0x5f, 0x6d, 0xa1, 0x2f,
                0x0d, 0x70, 0xe1, 0xf4, 0xce, 0xe3, 0xe2, 0x93, 0xc2, 0xef, 0x41, 0xb9, 0x8d, 0x26,
                0x34, 0xf3, 0xf4, 0x4d, 0xeb, 0x19, 0x8f, 0x1c, 0x2f, 0x17, 0x67, 0x78, 0x98, 0xba,
                0xe0, 0x6d, 0x8b, 0xff, 0x02, 0x64, 0x6c, 0x6e, 0x4c, 0x6b, 0xed, 0xee, 0x00, 0x01,
                0xd7, 0x39, 0x87, 0xf8, 0x98, 0xe1, 0x64, 0x1a, 0xde, 0x1c, 0xdd, 0x12, 0xe2, 0x27,
                0x78, 0xac, 0x8d, 0x89, 0x2e, 0x8c, 0x23, 0x88, 0x59, 0xb6, 0x3e, 0xe5, 0x92, 0x65,
                0xe3, 0x16, 0x31, 0x0f, 0x71, 0x39, 0x18, 0x6e, 0x62, 0xa1, 0xe2, 0x2f, 0xe2, 0xe2,
                0x3a, 0x3b, 0x4a, 0x70, 0x1b, 0x00, 0x5d, 0xd2, 0x00, 0xa5, 0x7f, 0x53, 0x9d, 0xe4,
                0x9e, 0x2e, 0x0c, 0x9b, 0xad, 0x83, 0xc0, 0x5f, 0x93, 0x2c, 0x94, 0x8d, 0x4f, 0x81,
                0xce, 0x15, 0x95, 0x7b, 0xc1, 0x0b, 0xcd, 0x58, 0x32, 0x0c, 0xb3, 0x2e, 0x0c, 0xba,
                0x8f, 0x86, 0xcb, 0xba, 0xfb, 0x11, 0x89, 0xb1,
            ],
            &[
                0x92, 0x15, 0x78, 0x80, 0x3f, 0x0f, 0x22, 0x90, 0xe7, 0x8e, 0x88, 0x81, 0x3c, 0x70,
                0x3d, 0x21, 0x1d, 0xea, 0x26, 0x52, 0x7d, 0xe3, 0x20, 0x9e, 0x86, 0x5e, 0x77, 0x8b,
                0x32, 0x5e, 0x80, 0x27, 0x9f, 0x6f, 0xa3, 0xf0, 0x76, 0xc1, 0x3e, 0x04, 0xe2, 0x19,
                0xd6, 0xc8, 0x11, 0x86, 0x11, 0x82, 0xb6, 0x96, 0x50, 0x37, 0x05, 0xbb, 0xbe, 0x44,
                0xe8, 0x70, 0x39, 0xcf, 0x5c, 0x75, 0xc3, 0xf3, 0x96, 0x58, 0x42, 0x17, 0x24, 0xb2,
                0x61, 0x76, 0x79, 0xf3, 0x75, 0x54, 0xff, 0xed, 0x1b, 0xf4, 0x33, 0x75, 0x15, 0xc5,
                0x72, 0x51, 0x62, 0xbb, 0xf5, 0xa8, 0xfe, 0x05, 0xa4, 0x5c, 0x35, 0xc7, 0x51, 0xa2,
                0xb8, 0x42, 0x2e, 0x4b, 0x15, 0xd1, 0x6a, 0x5f, 0x5e, 0xd4, 0xef, 0xdd, 0x2f, 0x31,
                0xc2, 0x0c, 0x01, 0xba, 0x38, 0x29, 0x0b, 0xc0, 0xd6, 0xa7, 0x90, 0x3d, 0xb1, 0x3a,
                0xcc, 0x0b, 0xd1, 0x8f, 0xb1, 0xf3, 0x49, 0xae, 0x80, 0x59, 0x10, 0x3c, 0x2b, 0x47,
                0xc9, 0x78, 0x07, 0xfb, 0x48, 0x8b, 0xac, 0x76, 0x48, 0x04, 0xf5, 0x74, 0x2a, 0xf6,
                0xd5, 0xfe, 0xa7, 0x8c, 0x6c, 0x52, 0x46, 0x5d, 0xa7, 0x1e, 0x80, 0x74, 0xb0, 0xcf,
                0x73, 0xaa, 0x28, 0xec, 0x36, 0x68, 0x33, 0x78, 0xa5, 0xf3, 0x4b, 0x02, 0xe9, 0x02,
                0x90, 0x5e, 0xbe, 0x2d, 0x9e, 0x8d, 0x12, 0xfa, 0xcf, 0x5b, 0x24, 0x96, 0x76, 0xb6,
                0x0c, 0xa5, 0x33, 0xe9, 0xea, 0x17, 0x57, 0xe7, 0x74, 0x3a, 0xde, 0x0f, 0xea, 0x97,
                0x23, 0x62, 0x8a, 0x7e, 0xb8, 0x4d, 0x4a, 0x65, 0x57, 0xec, 0x0c, 0x03, 0xe5, 0xe5,
                0xfb, 0xde, 0xc5, 0xc2, 0x1d, 0x6c, 0x5b, 0x95, 0x54, 0xd9, 0x6f, 0x89, 0xfe, 0x11,
                0x31, 0xf8, 0xad, 0x13, 0xec, 0x38, 0xc6, 0x16, 0x5b, 0xed, 0x4c, 0x2c, 0xc0, 0xa3,
                0xd6, 0xbf, 0x93, 0x83, 0xa1, 0xae, 0x07, 0x29, 0x5e, 0xad, 0x4b, 0xd5, 0x92, 0x38,
                0x31, 0xdd, 0x54, 0x03, 0x7d, 0xeb, 0xba, 0x2a, 0x05, 0x0e, 0xcc, 0xb6, 0xb1, 0x22,
                0xb8, 0xd6, 0xac, 0xd3, 0xe1, 0x31, 0x5c, 0x6e, 0xff, 0x26, 0xf3, 0x9b, 0xd1, 0xc8,
                0x33, 0x92, 0xa1, 0xc2, 0xe9, 0xf0, 0x9d, 0x1d, 0x04, 0x39, 0x56, 0xb1, 0xea, 0x03,
                0x6b, 0x7f, 0xbc, 0xa0, 0xd4, 0x11, 0x04, 0x02, 0xed, 0x2e, 0x18, 0x8c, 0x5a, 0xb6,
                0x6d, 0xfd, 0xd1, 0x78, 0x0e, 0x82, 0xaf, 0x58, 0x2d, 0xae, 0xed, 0x07, 0x44, 0x6d,
                0x7a, 0xf0, 0x02, 0xdd, 0xaa, 0x35, 0x84, 0xfd, 0xd9, 0x61, 0x0e, 0x74, 0x7d, 0x19,
                0x82, 0x98, 0xa2, 0xb9, 0xbb, 0x7a, 0x31, 0x6d, 0xf0, 0x38, 0xf4, 0x2f, 0x87, 0x07,
                0x0f, 0x1b, 0xc4, 0x8f, 0xad, 0x45, 0x52, 0x4c, 0xf3, 0x6b, 0x03, 0xbb, 0x3f, 0x15,
                0x39, 0x2e, 0xc2, 0x9f, 0xfe, 0x6d, 0xc5, 0xe3, 0xaf, 0xe3, 0x11, 0x3a, 0x93, 0x0f,
                0x22, 0x99, 0x68, 0xbc, 0x1c, 0x13, 0xdc, 0x21, 0x3b, 0x5a, 0x5f, 0xa4, 0x85, 0x6a,
                0x76, 0xc9, 0x03, 0x25, 0x59, 0xa9, 0xf8, 0x3b, 0x49, 0x2a, 0x70, 0x1e, 0x0f, 0x7a,
                0x10, 0x8b, 0x9d, 0x93, 0x78, 0x34, 0xd7, 0x4c, 0xcc, 0x5f, 0xd4, 0x75, 0x99, 0x93,
                0x11, 0xaa, 0x31, 0xe3, 0xa6, 0x85, 0xbf, 0x22, 0x71, 0xa6, 0xa9, 0xb2, 0x8f, 0xcb,
                0x25, 0x24, 0xa7, 0xf0, 0xc7, 0x67, 0x2e, 0xfa, 0x17, 0x16, 0xe2, 0x9b, 0xbe, 0x17,
                0xf6, 0x34, 0x9e, 0xc2, 0xc3, 0x94, 0x0f, 0x9e, 0xae, 0x87, 0x12, 0x8c, 0x70, 0xab,
                0x41, 0x74, 0x65, 0xfd, 0x09, 0x71, 0xeb, 0x50, 0x12, 0xcd, 0x6d, 0x36, 0xd8, 0x9f,
                0xc4, 0x0f, 0xb5, 0xff, 0xde, 0x8d, 0xd0, 0xf7, 0xeb, 0x82, 0xbb, 0xda, 0xe1, 0xd4,
                0x94, 0x35, 0x9e, 0x3e, 0x4a, 0xa1, 0xf3, 0x51,
            ],
        ),
    ];
}

mod ecdsa {
    pub const PRIVATE_KEY: [u8; 32] = [
        0x30, 0x8d, 0x6c, 0x77, 0xcc, 0x43, 0xf7, 0xb8, 0x4f, 0x44, 0x74, 0xdc, 0x2f, 0x99, 0xf6,
        0x33, 0x3e, 0x26, 0x8a, 0x0c, 0x94, 0x4c, 0xde, 0x56, 0xff, 0xb5, 0x27, 0xb7, 0x7f, 0xa6,
        0x11, 0x0c,
    ];

    pub const PUBLIC_KEY: [u8; 64] = [
        0x16, 0xa6, 0xbd, 0x9a, 0x66, 0x66, 0x36, 0xd0, 0x72, 0x86, 0xde, 0x78, 0xb9, 0xa1, 0xe7,
        0xf6, 0xdd, 0x67, 0x75, 0xb2, 0xc6, 0xf4, 0x2c, 0xcf, 0x83, 0x2d, 0xe4, 0x5e, 0x1e, 0x22,
        0x9d, 0x84, 0x0a, 0xca, 0x0d, 0xdd, 0xe8, 0xf5, 0xc8, 0x2f, 0x84, 0x10, 0xb5, 0x62, 0xc2,
        0x3a, 0x46, 0xde, 0xcd, 0xcb, 0x59, 0x6e, 0x40, 0x02, 0xcb, 0x10, 0xc6, 0x2f, 0x5b, 0x5e,
        0xb5, 0xf2, 0xa7, 0xd7,
    ];

    pub const K_SIGNATURE_PAIRS: [([u8; 32], [u8; 64]); 10] = [
        (
            [
                0x48, 0x4a, 0x19, 0x66, 0x02, 0x50, 0x0e, 0xf2, 0xd0, 0xbe, 0x90, 0x84, 0x23, 0x8e,
                0x45, 0x09, 0x6c, 0x23, 0x8b, 0x1b, 0x74, 0xa8, 0x6b, 0x17, 0x46, 0x62, 0x75, 0xd2,
                0xfa, 0x27, 0x7e, 0x1b,
            ],
            [
                0x21, 0xea, 0x0f, 0xfd, 0x35, 0x43, 0xdf, 0x7a, 0xdb, 0xf5, 0x4f, 0x88, 0x0e, 0x9d,
                0xd2, 0xa7, 0x26, 0x4f, 0x2f, 0x96, 0xe9, 0x85, 0x5f, 0x67, 0xa9, 0x82, 0x46, 0xfe,
                0x46, 0xef, 0x92, 0x9d, 0x3c, 0x59, 0x7c, 0x22, 0x4b, 0x69, 0x80, 0xf7, 0x01, 0x46,
                0x09, 0xce, 0x13, 0x59, 0xfd, 0x21, 0xd1, 0x45, 0x65, 0xfb, 0xb0, 0x82, 0x1b, 0x91,
                0xce, 0x1e, 0x87, 0xf5, 0xe5, 0xc8, 0xdc, 0x9c,
            ],
        ),
        (
            [
                0xea, 0x40, 0xe8, 0x9d, 0xf6, 0x63, 0xf4, 0x3e, 0x71, 0xf2, 0x6b, 0x7f, 0xcd, 0xa0,
                0x15, 0x59, 0x13, 0x4f, 0xa9, 0x17, 0xbd, 0x5f, 0xbc, 0xf3, 0x36, 0xfb, 0x48, 0x14,
                0x8f, 0x59, 0x99, 0x1d,
            ],
            [
                0x9a, 0x84, 0x64, 0x3b, 0xd1, 0xb8, 0xe2, 0xa6, 0xe3, 0xc7, 0x96, 0x9b, 0xfa, 0x00,
                0xac, 0x65, 0x19, 0xa8, 0x3e, 0x22, 0x2e, 0x40, 0x7d, 0x90, 0x98, 0x92, 0xce, 0x3b,
                0x77, 0x4e, 0x8c, 0x41, 0xe7, 0xa1, 0xcd, 0xb1, 0xc4, 0x0a, 0xc0, 0x73, 0xfa, 0x87,
                0x5f, 0xa5, 0xae, 0xcf, 0x27, 0x14, 0x06, 0x38, 0x9f, 0x4c, 0x7f, 0xaa, 0xf9, 0x76,
                0x6e, 0x49, 0x03, 0x0c, 0xc8, 0x33, 0x26, 0x03,
            ],
        ),
        (
            [
                0x99, 0xde, 0xf2, 0x6b, 0xa6, 0xfe, 0x92, 0x0f, 0xd6, 0x33, 0x3a, 0x1b, 0x21, 0x2c,
                0xcb, 0xd2, 0x50, 0x81, 0x57, 0xad, 0x26, 0x31, 0xea, 0x56, 0x23, 0x94, 0x69, 0x3b,
                0xc3, 0xe7, 0x96, 0xd7,
            ],
            [
                0x47, 0x1a, 0x16, 0x6b, 0xde, 0x2e, 0x34, 0xb3, 0xc6, 0x80, 0xa2, 0x18, 0xed, 0xa7,
                0xfa, 0xc6, 0x7f, 0xfc, 0x77, 0xae, 0x80, 0xce, 0x18, 0x90, 0x51, 0x1f, 0x4d, 0x23,
                0x8a, 0x96, 0x62, 0x25, 0xa7, 0x5a, 0xc7, 0x47, 0x68, 0xa2, 0xf0, 0x76, 0x5e, 0x01,
                0x6b, 0x29, 0xb2, 0x9d, 0xba, 0x3b, 0x71, 0x8a, 0x7c, 0xfd, 0xaa, 0x49, 0x53, 0xe0,
                0x90, 0x62, 0xce, 0x06, 0x95, 0x55, 0xd4, 0xc4,
            ],
        ),
        (
            [
                0x91, 0xda, 0x2c, 0xea, 0x22, 0xc3, 0x08, 0x44, 0x5c, 0x01, 0x0e, 0x2b, 0x00, 0x74,
                0x44, 0x05, 0x14, 0x50, 0x25, 0x92, 0xb3, 0xde, 0xe9, 0xcd, 0xb0, 0x67, 0x25, 0x10,
                0x26, 0x8a, 0x66, 0xb6,
            ],
            [
                0x89, 0xaa, 0x32, 0x68, 0x08, 0xbf, 0x3f, 0xd8, 0xbb, 0x13, 0xc5, 0x51, 0xa6, 0x0e,
                0x13, 0x3f, 0xb5, 0x6f, 0x96, 0xcd, 0x7d, 0x9f, 0xe7, 0xd4, 0x17, 0xef, 0xad, 0x93,
                0x14, 0xed, 0x4f, 0x0f, 0xdb, 0x34, 0xc1, 0xc3, 0xf4, 0xc9, 0x11, 0x9e, 0xd7, 0xe7,
                0x23, 0xbc, 0xd3, 0x5c, 0x73, 0x57, 0xd5, 0x74, 0x75, 0x90, 0xaf, 0x4e, 0x60, 0x47,
                0x57, 0xe0, 0x16, 0xc2, 0x0d, 0x9e, 0xce, 0x44,
            ],
        ),
        (
            [
                0x3d, 0x3d, 0x65, 0x81, 0x9d, 0xc3, 0xd1, 0x23, 0xde, 0x2d, 0xe0, 0x92, 0x99, 0x7d,
                0x0b, 0xb5, 0xab, 0x93, 0x02, 0x0a, 0x8b, 0x0d, 0x37, 0xe3, 0xe0, 0x0f, 0xf7, 0x91,
                0x60, 0x39, 0xf4, 0x97,
            ],
            [
                0x56, 0x99, 0xc2, 0x70, 0x77, 0x34, 0x71, 0x9a, 0xdb, 0xcf, 0xb3, 0xc1, 0x0a, 0x5d,
                0x2a, 0x18, 0xbd, 0x35, 0xcc, 0x46, 0x6c, 0xfb, 0x87, 0x0a, 0xe2, 0xc2, 0x6f, 0xdf,
                0x23, 0x70, 0x2c, 0x49, 0xc0, 0xd7, 0x2a, 0x54, 0xf6, 0xd6, 0x46, 0x5f, 0xb0, 0x59,
                0xe0, 0x70, 0x58, 0xae, 0x64, 0x9c, 0x3f, 0x2d, 0x48, 0xad, 0xf6, 0x66, 0xe9, 0x03,
                0x88, 0xf7, 0x0a, 0x0e, 0x2a, 0xec, 0xba, 0x12,
            ],
        ),
        (
            [
                0x4e, 0xa9, 0xce, 0xda, 0xce, 0xe2, 0xe9, 0x58, 0x43, 0xcd, 0x90, 0x70, 0x75, 0xc6,
                0xe8, 0x58, 0x19, 0x74, 0x09, 0x0a, 0x75, 0xa3, 0xfb, 0xbd, 0x38, 0x97, 0xba, 0x92,
                0xb3, 0x87, 0x81, 0x88,
            ],
            [
                0x4a, 0x20, 0xe6, 0xf3, 0xf0, 0x96, 0xc4, 0xad, 0x6b, 0xe4, 0x95, 0xae, 0xeb, 0xee,
                0xa9, 0xb8, 0x90, 0x45, 0x87, 0xfb, 0x32, 0x8e, 0x30, 0xce, 0x49, 0xaa, 0x11, 0x7f,
                0x11, 0x2a, 0xba, 0xa1, 0x54, 0xe0, 0xb3, 0x68, 0x25, 0x76, 0x5c, 0xf9, 0x0b, 0x46,
                0xdf, 0x8d, 0x8b, 0x99, 0x1b, 0x9d, 0x2d, 0x9f, 0xfb, 0x52, 0xcc, 0x32, 0xb2, 0x4c,
                0x2a, 0x93, 0xff, 0x23, 0xe5, 0xf7, 0x88, 0x9f,
            ],
        ),
        (
            [
                0x8d, 0xbc, 0xea, 0x73, 0x54, 0x1b, 0x93, 0x29, 0xc6, 0xb6, 0x82, 0xbc, 0xd2, 0xa6,
                0xeb, 0x68, 0x09, 0x9c, 0x64, 0x97, 0x34, 0xc9, 0x3c, 0xe8, 0x56, 0xd7, 0x3a, 0xdb,
                0x32, 0x78, 0xb6, 0x0a,
            ],
            [
                0x91, 0x1a, 0x54, 0x4e, 0xc3, 0x77, 0x3b, 0x37, 0x48, 0x84, 0xbc, 0x84, 0xc2, 0x6d,
                0x32, 0x9b, 0xc9, 0x5f, 0x24, 0x7c, 0x4d, 0x29, 0xc7, 0xd6, 0xd5, 0x23, 0xf5, 0x25,
                0x49, 0x6f, 0xf0, 0xd3, 0xa3, 0x0f, 0xf7, 0x2a, 0xa9, 0x86, 0xb1, 0xd1, 0xf0, 0x31,
                0xa5, 0x71, 0x40, 0x8c, 0xc4, 0x0f, 0x56, 0xa0, 0x8c, 0x4b, 0xfc, 0x7d, 0xe5, 0x98,
                0x90, 0xdb, 0xcd, 0x68, 0xe9, 0x4b, 0x4f, 0x9c,
            ],
        ),
        (
            [
                0x2b, 0xe8, 0x71, 0x47, 0x76, 0x0c, 0x8e, 0x96, 0x7c, 0xcf, 0x2b, 0x78, 0xc2, 0x89,
                0xbd, 0xef, 0x8d, 0x2f, 0x7a, 0xe7, 0xa0, 0xab, 0x8b, 0x84, 0xa8, 0x43, 0xe6, 0x33,
                0x36, 0x67, 0xcd, 0x08,
            ],
            [
                0x6d, 0xa1, 0x3e, 0xf9, 0xf0, 0x53, 0x89, 0x67, 0xb0, 0xf4, 0xe3, 0x86, 0xb3, 0x56,
                0x7a, 0x9a, 0xcf, 0xba, 0x94, 0xb8, 0xba, 0xbf, 0xb6, 0xa0, 0x7f, 0xaa, 0xc4, 0xd8,
                0xcb, 0x2c, 0x3a, 0xf4, 0x11, 0xc4, 0x3a, 0x17, 0x52, 0x10, 0x5d, 0xde, 0x72, 0x5d,
                0x5a, 0xc1, 0xd9, 0x3a, 0x5f, 0x56, 0xb8, 0x79, 0x78, 0x4c, 0x71, 0xb3, 0x05, 0x69,
                0x52, 0x63, 0x06, 0xe8, 0xe3, 0xe2, 0xfa, 0x10,
            ],
        ),
        (
            [
                0x87, 0x2f, 0x09, 0x31, 0x90, 0xcf, 0xeb, 0x70, 0x96, 0x9a, 0x67, 0x59, 0xa9, 0x8b,
                0x40, 0xe3, 0xfe, 0xfd, 0xeb, 0x37, 0x53, 0xcf, 0x62, 0xfe, 0x27, 0x13, 0x7b, 0xe6,
                0x08, 0xf0, 0x3d, 0x1c,
            ],
            [
                0x7c, 0x93, 0x91, 0x44, 0x1e, 0x84, 0x07, 0xc2, 0x22, 0xd3, 0x92, 0x3b, 0xb0, 0xfe,
                0x41, 0xe8, 0x8d, 0x1e, 0x1d, 0xd5, 0x56, 0x56, 0x05, 0x31, 0x44, 0xc8, 0xa2, 0x4b,
                0xee, 0xdc, 0x8c, 0x36, 0x3f, 0xc5, 0xe9, 0xfa, 0x57, 0x1e, 0x20, 0x5f, 0xc5, 0x97,
                0xd1, 0xe8, 0x84, 0xaf, 0x74, 0x10, 0xc6, 0x08, 0x6b, 0x3e, 0xea, 0x61, 0x7c, 0x9a,
                0x77, 0x54, 0x31, 0x8b, 0x3b, 0x8b, 0x04, 0xc5,
            ],
        ),
        (
            [
                0xca, 0x8b, 0x60, 0xf1, 0x88, 0x6d, 0xb6, 0xf7, 0x33, 0x4f, 0xcc, 0x39, 0x9c, 0xf4,
                0x82, 0xe7, 0xde, 0x42, 0x37, 0x8d, 0xb9, 0x97, 0xa6, 0x5e, 0x4b, 0x0e, 0xc8, 0xaa,
                0x09, 0xbc, 0xee, 0xac,
            ],
            [
                0x69, 0xc1, 0x7c, 0x3f, 0xaa, 0x06, 0x0a, 0x00, 0x0d, 0xdf, 0x08, 0x1d, 0x38, 0xe3,
                0xa8, 0xc5, 0xe7, 0xa5, 0x80, 0xbd, 0x48, 0x27, 0xdf, 0x20, 0x12, 0x36, 0x9f, 0x4b,
                0xfd, 0x59, 0x2a, 0x92, 0x95, 0x71, 0xa3, 0x0c, 0x58, 0x7d, 0x6a, 0x6d, 0x7b, 0x1f,
                0xc1, 0x43, 0x5a, 0x6d, 0x55, 0x8e, 0xe0, 0xc5, 0x76, 0xc6, 0xf0, 0xdc, 0x92, 0x77,
                0x23, 0x14, 0x49, 0xb6, 0xa6, 0xe5, 0x1d, 0x1c,
            ],
        ),
    ];
}
