#[macro_use]
extern crate fdec;

fdec64! {
    module dec,
    name Decimal,
    length 16,
    scale 100
}

use dec::*;
use std::str::FromStr;

#[test]
fn test_const() {
    assert_eq!(Decimal::zero(), Decimal::from(0));
    assert_eq!(Decimal::ulp(), Decimal::with_scale(1, 100));
    assert_eq!(Decimal::one(), Decimal::from(1));
    assert_eq!(Decimal::max(), Decimal::from_str("17976931348623159077293051907890247336179769789423065727343008115773267580550096313270847732240753602112011387987139335765878976881441662249284743063947412437776789342486548527630221960124609411945308295208500.5768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137215").unwrap());
    assert_eq!(Decimal::min(), Decimal::from_str("-17976931348623159077293051907890247336179769789423065727343008115773267580550096313270847732240753602112011387987139335765878976881441662249284743063947412437776789342486548527630221960124609411945308295208500.5768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137215").unwrap());
}

#[test]
fn test_macro() {
    assert_eq!(dec!(75), Decimal::from(75));
    assert_eq!(dec!(75, 1), Decimal::with_scale(75, 1));
}

#[test]
fn test_from_str() {
    assert_eq!(Decimal::from_str("0").unwrap(), Decimal::zero());
    assert_eq!(Decimal::from_str("1").unwrap(), Decimal::one());
    assert_eq!(Decimal::from_str("0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001").unwrap(), Decimal::ulp());
    assert_eq!(Decimal::from_str("17976931348623159077293051907890247336179769789423065727343008115773267580550096313270847732240753602112011387987139335765878976881441662249284743063947412437776789342486548527630221960124609411945308295208500.5768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137215").unwrap(), Decimal::max());
    assert_eq!(Decimal::from_str("17976931348623159077293051907890247336179769789423065727343008115773267580550096313270847732240753602112011387987139335765878976881441662249284743063947412437776789342486548527630221960124609411945308295208500.5768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137216").unwrap_err(), ParseNumberError::Overflow);
    assert_eq!(Decimal::from_str("999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999").unwrap_err(), ParseNumberError::Overflow);
}

#[test]
fn test_to_string() {
    assert_eq!(Decimal::zero().to_string(), "0");
    assert_eq!(Decimal::ulp().to_string(), "0.0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001");
    assert_eq!(Decimal::one().to_string(), "1");
    assert_eq!(Decimal::max().to_string(), "17976931348623159077293051907890247336179769789423065727343008115773267580550096313270847732240753602112011387987139335765878976881441662249284743063947412437776789342486548527630221960124609411945308295208500.5768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137215");
    assert_eq!(Decimal::min().to_string(), "-17976931348623159077293051907890247336179769789423065727343008115773267580550096313270847732240753602112011387987139335765878976881441662249284743063947412437776789342486548527630221960124609411945308295208500.5768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137215");
    assert_eq!(
        Decimal::from(u64::max_value()).to_string(),
        "18446744073709551615"
    );
}

#[test]
fn test_constants() {
    assert_eq!(dec::consts::E.to_string(), "2.7182818284590452353602874713526624977572470936999595749669676277240766303535475945713821785251664274");
    assert_eq!(dec::consts::PI.to_string(), "3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068");
}
