use std::collections::HashSet;
use std::ops::Deref;

use fastobo::ast as obo;
use fastobo::semantics::Identified;
use horned_owl::model as owl;

use super::into_owl::Context;
use super::into_owl::IntoOwlCtx;

// ---------------------------------------------------------------------------

pub trait ImportProvider {
    fn import(&mut self, import: &obo::Import) -> Result<ImportData, String>;
}

// ---------------------------------------------------------------------------

// WIP: Retrieve imports from the OBO Foundry.
//
// #[derive(Debug, Default)]
// pub struct FoundryProvider {}
//
// impl ImportProvider for FoundryProvider {
//     fn import(&mut self, import: &obo::Import) -> Result<ImportData, String> {
//         // use URL or use default OBO Foundry URL
//         let url = match import {
//             obo::Import::Url(url) => *url.clone(),
//             obo::Import::Abbreviated(id) => {
//                 let s = format!("http://purl.obolibrary.org/obo/{}.obo", id);
//                 obo::Url::parse(&s).expect("invalid import")
//             }
//         };
//
//         // get the imported document
//         let res = ureq::get(url.as_str()).redirects(10).call();
//         let mut buf = BufReader::new(res.into_reader());
//
//         // parse the OBO file if it is a correct OBO file.
//         let resource = url.as_str().rsplit('/').next().unwrap();
//         let mut data = match Path::new(resource).extension() {
//             Some(x) if x == "obo" => {
//                 let mut doc = fastobo::from_reader(&mut buf).expect("could not parse OBO document");
//                 doc.treat_xrefs();
//                 ImportData::from(doc)
//             }
//             Some(x) if x == "owl" => {
//                 return Err(String::from("cannot import OWL now"));
//             }
//             other => {
//                 return Err(format!("unknown import extension: {:?}", other));
//             }
//         };
//
//         // process all imports
//         let mut imports = data.imports.clone();
//         for i in imports.drain() {
//             // import the import in the document and add them to the `ImportData`.
//             let import_data = self.import(&i)?;
//             data.imports.extend(import_data.imports);
//             data.annotation_properties
//                 .extend(import_data.annotation_properties);
//         }
//
//         Ok(data)
//     }
// }

// ---------------------------------------------------------------------------

#[derive(Debug, Default, PartialEq, Eq)]
pub struct ImportData {
    // Needed to check no class-level relationship is used in an
    // `intersection_of` clause (in theory...)
    // class_level_rel: HashSet<obo::RelationIdent>,
    /// The set of annotation properties declared in the document.
    ///
    /// Needed for for `rel(.., .., ..)` translation.
    pub annotation_properties: HashSet<owl::IRI>,

    /// The imports declared in the document.
    pub imports: HashSet<obo::Import>,
}

// ---------------------------------------------------------------------------

impl From<obo::OboDoc> for ImportData {
    fn from(doc: obo::OboDoc) -> Self {
        let mut annotation_properties = HashSet::new();
        let mut imports = HashSet::new();

        // create context to extract IRI
        let mut context = Context::from(&doc);

        // extract imports
        for clause in doc.header().iter() {
            if let obo::HeaderClause::Import(import) = clause {
                imports.insert(import.deref().clone());
            }
        }

        // extract annotation properties
        for frame in doc.entities() {
            if let obo::EntityFrame::Typedef(typedef) = frame {
                for clause in typedef.clauses() {
                    if let obo::TypedefClause::IsMetadataTag(true) = clause.as_inner() {
                        let iri = frame.as_id().clone().into_owl(&mut context);
                        annotation_properties.insert(iri);
                    }
                }
            }
        }

        ImportData {
            annotation_properties,
            imports,
        }
    }
}
