//! Dictionaries for Compute@Edge.

pub(crate) mod handle;

use handle::DictionaryHandle;

/// Maximum Edge Dictionary value size.
///
/// This constant is determined by consulting Fastly's [Edge Dictionary documentation][1]:
///
/// > Dictionary containers, item keys, and their values have specific limits. Dictionary
/// > containers are limited to 1000 items. Dictionary item keys are limited to 256 characters and
/// > their values are limited to 8000 characters.
///
/// This constant is used as the default buffer size for dictionary values when using the
/// high-level dictionary API.
///
/// [1]: https://docs.fastly.com/en/guides/about-edge-dictionaries#limitations-and-considerations
const MAX_LEN: usize = 8000;

/// A Compute@Edge Dictionary.
pub struct Dictionary {
    handle: DictionaryHandle,
}

impl Dictionary {
    /// Open a dictionary, given its name.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// # use fastly::Dictionary;
    /// let merriam = Dictionary::open("merriam webster");
    /// let oed = Dictionary::open("oxford english dictionary");
    /// ```
    pub fn open(name: &str) -> Self {
        let handle = match DictionaryHandle::open(name) {
            Ok(h) if h.is_valid() => h,
            Ok(_) => panic!("could not open dictionary `{}`", name),
            Err(e) => panic!("could not open dictionary `{}`: {}", name, e),
        };

        Self { handle }
    }

    /// Lookup a value in this dictionary.
    ///
    /// If successful, this function returns `Some(_)` if an entry was found, or `None` if no entry
    /// with the given key was found.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// # use fastly::Dictionary;
    /// # let dictionary = Dictionary::open("test dictionary");
    /// #
    /// assert_eq!(
    ///      dictionary.get("bread"),
    ///      Some(String::from("a usually baked and leavened food")),
    /// );
    /// assert_eq!(
    ///     dictionary.get("freedom"),
    ///     Some(String::from("the absence of necessity, coercion, or constraint")),
    /// );
    ///
    /// // Otherwise, `get` will return nothing.
    /// assert!(dictionary.get("zzzzz").is_none());
    /// ```
    pub fn get(&self, key: &str) -> Option<String> {
        self.handle
            .get(key, MAX_LEN)
            .unwrap_or_else(|e| panic!("lookup for key `{}` failed: {}", key, e))
    }

    /// Return true if the dictionary contains an entry with the given key.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// # use fastly::Dictionary;
    /// # let dictionary = Dictionary::open("test dictionary");
    /// #
    /// assert!(dictionary.contains("key"));
    /// ```
    pub fn contains(&self, key: &str) -> bool {
        self.handle
            .contains(key)
            .unwrap_or_else(|e| panic!("lookup for key `{}` failed: {}", key, e))
    }
}
