// TODO ACF 2020-12-01: remove once this is fixed: https://github.com/rust-lang/rust/issues/79581
#![allow(clashing_extern_declarations)]

//! FFI bindings to the Fastly Compute@Edge ABI.
//!
//! This is a low-level package; the [`fastly`](https://docs.rs/fastly) crate wraps these functions
//! in a much friendlier, Rust-like interface. You should not have to depend on this crate
//! explicitly in your `Cargo.toml`.
//!
//! # Versioning and compatibility
//!
//! The Cargo version of this package was previously set according to compatibility with the
//! Compute@Edge platform. Since the [`v0.25.0` release of the Fastly
//! CLI](https://github.com/fastly/cli/releases/tag/v0.25.0), the CLI is configured with the range
//! of `fastly-sys` versions that are currently compatible with the Compute@Edge platform. The Cargo
//! version of this package since `0.4.0` instead follows the [Cargo SemVer compatibility
//! guidelines](https://doc.rust-lang.org/cargo/reference/semver.html).
use fastly_shared::FastlyStatus;

mod legacy;
pub use legacy::*;

/// An alias for `u32` to denote which ABI arguments represent `BodyHandle`s.
pub type BodyHandle = u32;
/// An alias for `u32` to denote which ABI arguments represent `PendingRequestHandle`s.
pub type PendingRequestHandle = u32;
/// An alias for `u32` to denote which ABI arguments represent `RequestHandle`s.
pub type RequestHandle = u32;
/// An alias for `u32` to denote which ABI arguments represent `ResponseHandle`s.
pub type ResponseHandle = u32;
/// An alias for `u32` to denote which ABI arguments represent `DictionaryHandle`s.
pub type DictionaryHandle = u32;

bitflags::bitflags! {
    /// `Content-Encoding` codings.
    ///
    /// This type must match the definition in `typename.witx`, and will be replaced once we
    /// generate Wasm-side bindings from witx.
    #[derive(Default)]
    #[repr(transparent)]
    pub struct ContentEncodings: u32 {
        const GZIP = 1 << 0;
    }
}

pub mod fastly_abi {
    use super::*;

    #[link(wasm_import_module = "fastly_abi")]
    extern "C" {
        #[link_name = "init"]
        /// Tell the runtime what ABI version this program is using (FASTLY_ABI_VERSION)
        pub fn init(abi_version: u64) -> FastlyStatus;
    }
}

pub mod fastly_uap {
    use super::*;

    #[link(wasm_import_module = "fastly_uap")]
    extern "C" {
        #[link_name = "parse"]
        pub fn parse(
            user_agent: *const u8,
            user_agent_max_len: usize,
            family: *mut u8,
            family_max_len: usize,
            family_written: *mut usize,
            major: *mut u8,
            major_max_len: usize,
            major_written: *mut usize,
            minor: *mut u8,
            minor_max_len: usize,
            minor_written: *mut usize,
            patch: *mut u8,
            patch_max_len: usize,
            patch_written: *mut usize,
        ) -> FastlyStatus;
    }
}

pub mod fastly_http_body {
    use super::*;

    #[link(wasm_import_module = "fastly_http_body")]
    extern "C" {
        #[link_name = "append"]
        pub fn append(dst_handle: BodyHandle, src_handle: BodyHandle) -> FastlyStatus;

        #[link_name = "new"]
        pub fn new(handle_out: *mut BodyHandle) -> FastlyStatus;

        #[link_name = "read"]
        pub fn read(
            body_handle: BodyHandle,
            buf: *mut u8,
            buf_len: usize,
            nread_out: *mut usize,
        ) -> FastlyStatus;

        // overeager warning for extern declarations is a rustc bug: https://github.com/rust-lang/rust/issues/79581
        #[allow(clashing_extern_declarations)]
        #[link_name = "write"]
        pub fn write(
            body_handle: BodyHandle,
            buf: *const u8,
            buf_len: usize,
            end: fastly_shared::BodyWriteEnd,
            nwritten_out: *mut usize,
        ) -> FastlyStatus;

        /// Close a body, freeing its resources and causing any sends to finish.
        #[link_name = "close"]
        pub fn close(body_handle: BodyHandle) -> FastlyStatus;
    }
}

pub mod fastly_log {
    use super::*;

    #[link(wasm_import_module = "fastly_log")]
    extern "C" {
        #[link_name = "endpoint_get"]
        pub fn endpoint_get(
            name: *const u8,
            name_len: usize,
            endpoint_handle_out: *mut u32,
        ) -> FastlyStatus;

        // overeager warning for extern declarations is a rustc bug: https://github.com/rust-lang/rust/issues/79581
        #[allow(clashing_extern_declarations)]
        #[link_name = "write"]
        pub fn write(
            endpoint_handle: u32,
            msg: *const u8,
            msg_len: usize,
            nwritten_out: *mut usize,
        ) -> FastlyStatus;

    }
}

pub mod fastly_http_req {
    use super::*;

    #[link(wasm_import_module = "fastly_http_req")]
    extern "C" {
        #[link_name = "body_downstream_get"]
        pub fn body_downstream_get(
            req_handle_out: *mut RequestHandle,
            body_handle_out: *mut BodyHandle,
        ) -> FastlyStatus;

        #[link_name = "cache_override_set"]
        pub fn cache_override_set(
            req_handle: RequestHandle,
            tag: u32,
            ttl: u32,
            swr: u32,
        ) -> FastlyStatus;

        #[link_name = "cache_override_v2_set"]
        pub fn cache_override_v2_set(
            req_handle: RequestHandle,
            tag: u32,
            ttl: u32,
            swr: u32,
            sk: *const u8,
            sk_len: usize,
        ) -> FastlyStatus;

        #[link_name = "downstream_client_ip_addr"]
        pub fn downstream_client_ip_addr(
            addr_octets_out: *mut u8,
            nwritten_out: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "downstream_tls_cipher_openssl_name"]
        pub fn downstream_tls_cipher_openssl_name(
            cipher_out: *mut u8,
            cipher_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "downstream_tls_protocol"]
        pub fn downstream_tls_protocol(
            protocol_out: *mut u8,
            protocol_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "downstream_tls_client_hello"]
        pub fn downstream_tls_client_hello(
            client_hello_out: *mut u8,
            client_hello_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "header_append"]
        pub fn header_append(
            req_handle: RequestHandle,
            name: *const u8,
            name_len: usize,
            value: *const u8,
            value_len: usize,
        ) -> FastlyStatus;

        #[link_name = "header_insert"]
        pub fn header_insert(
            req_handle: RequestHandle,
            name: *const u8,
            name_len: usize,
            value: *const u8,
            value_len: usize,
        ) -> FastlyStatus;

        #[link_name = "original_header_names_get"]
        pub fn original_header_names_get(
            buf: *mut u8,
            buf_len: usize,
            cursor: u32,
            ending_cursor: *mut i64,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "original_header_count"]
        pub fn original_header_count(count_out: *mut u32) -> FastlyStatus;

        #[link_name = "header_names_get"]
        pub fn header_names_get(
            req_handle: RequestHandle,
            buf: *mut u8,
            buf_len: usize,
            cursor: u32,
            ending_cursor: *mut i64,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "header_values_get"]
        pub fn header_values_get(
            req_handle: RequestHandle,
            name: *const u8,
            name_len: usize,
            buf: *mut u8,
            buf_len: usize,
            cursor: u32,
            ending_cursor: *mut i64,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "header_values_set"]
        pub fn header_values_set(
            req_handle: RequestHandle,
            name: *const u8,
            name_len: usize,
            values: *const u8,
            values_len: usize,
        ) -> FastlyStatus;

        #[link_name = "header_value_get"]
        pub fn header_value_get(
            req_handle: RequestHandle,
            name: *const u8,
            name_len: usize,
            value: *mut u8,
            value_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "header_remove"]
        pub fn header_remove(
            req_handle: RequestHandle,
            name: *const u8,
            name_len: usize,
        ) -> FastlyStatus;

        #[link_name = "method_get"]
        pub fn method_get(
            req_handle: RequestHandle,
            method: *mut u8,
            method_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "method_set"]
        pub fn method_set(
            req_handle: RequestHandle,
            method: *const u8,
            method_len: usize,
        ) -> FastlyStatus;

        #[link_name = "new"]
        pub fn new(req_handle_out: *mut RequestHandle) -> FastlyStatus;

        #[link_name = "send"]
        pub fn send(
            req_handle: RequestHandle,
            body_handle: BodyHandle,
            backend: *const u8,
            backend_len: usize,
            resp_handle_out: *mut ResponseHandle,
            resp_body_handle_out: *mut BodyHandle,
        ) -> FastlyStatus;

        #[link_name = "send_async"]
        pub fn send_async(
            req_handle: RequestHandle,
            body_handle: BodyHandle,
            backend: *const u8,
            backend_len: usize,
            pending_req_handle_out: *mut PendingRequestHandle,
        ) -> FastlyStatus;

        #[link_name = "send_async_streaming"]
        pub fn send_async_streaming(
            req_handle: RequestHandle,
            body_handle: BodyHandle,
            backend: *const u8,
            backend_len: usize,
            pending_req_handle_out: *mut PendingRequestHandle,
        ) -> FastlyStatus;

        #[link_name = "uri_get"]
        pub fn uri_get(
            req_handle: RequestHandle,
            uri: *mut u8,
            uri_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "uri_set"]
        pub fn uri_set(req_handle: RequestHandle, uri: *const u8, uri_len: usize) -> FastlyStatus;

        #[link_name = "version_get"]
        pub fn version_get(req_handle: RequestHandle, version: *mut u32) -> FastlyStatus;

        #[link_name = "version_set"]
        pub fn version_set(req_handle: RequestHandle, version: u32) -> FastlyStatus;

        #[link_name = "pending_req_poll"]
        pub fn pending_req_poll(
            pending_req_handle: PendingRequestHandle,
            is_done_out: *mut i32,
            resp_handle_out: *mut ResponseHandle,
            resp_body_handle_out: *mut BodyHandle,
        ) -> FastlyStatus;

        #[link_name = "pending_req_select"]
        pub fn pending_req_select(
            pending_req_handles: *const PendingRequestHandle,
            pending_req_handles_len: usize,
            done_index_out: *mut i32,
            resp_handle_out: *mut ResponseHandle,
            resp_body_handle_out: *mut BodyHandle,
        ) -> FastlyStatus;

        #[link_name = "pending_req_wait"]
        pub fn pending_req_wait(
            pending_req_handle: PendingRequestHandle,
            resp_handle_out: *mut ResponseHandle,
            resp_body_handle_out: *mut BodyHandle,
        ) -> FastlyStatus;

        #[link_name = "fastly_key_is_valid"]
        pub fn fastly_key_is_valid(is_valid_out: *mut u32) -> FastlyStatus;

        #[link_name = "close"]
        pub fn close(req_handle: RequestHandle) -> FastlyStatus;

        #[link_name = "auto_decompress_response_set"]
        pub fn auto_decompress_response_set(
            req_handle: RequestHandle,
            encodings: ContentEncodings,
        ) -> FastlyStatus;
    }
}

pub mod fastly_http_resp {
    use super::*;

    #[link(wasm_import_module = "fastly_http_resp")]
    extern "C" {
        #[link_name = "header_append"]
        pub fn header_append(
            resp_handle: ResponseHandle,
            name: *const u8,
            name_len: usize,
            value: *const u8,
            value_len: usize,
        ) -> FastlyStatus;

        #[link_name = "header_insert"]
        pub fn header_insert(
            resp_handle: ResponseHandle,
            name: *const u8,
            name_len: usize,
            value: *const u8,
            value_len: usize,
        ) -> FastlyStatus;

        #[link_name = "header_names_get"]
        pub fn header_names_get(
            resp_handle: ResponseHandle,
            buf: *mut u8,
            buf_len: usize,
            cursor: u32,
            ending_cursor: *mut i64,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "header_value_get"]
        pub fn header_value_get(
            resp_handle: ResponseHandle,
            name: *const u8,
            name_len: usize,
            value: *mut u8,
            value_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "header_values_get"]
        pub fn header_values_get(
            resp_handle: ResponseHandle,
            name: *const u8,
            name_len: usize,
            buf: *mut u8,
            buf_len: usize,
            cursor: u32,
            ending_cursor: *mut i64,
            nwritten: *mut usize,
        ) -> FastlyStatus;

        #[link_name = "header_values_set"]
        pub fn header_values_set(
            resp_handle: ResponseHandle,
            name: *const u8,
            name_len: usize,
            values: *const u8,
            values_len: usize,
        ) -> FastlyStatus;

        #[link_name = "header_remove"]
        pub fn header_remove(
            resp_handle: ResponseHandle,
            name: *const u8,
            name_len: usize,
        ) -> FastlyStatus;

        #[link_name = "new"]
        pub fn new(resp_handle_out: *mut ResponseHandle) -> FastlyStatus;

        #[link_name = "send_downstream"]
        pub fn send_downstream(
            resp_handle: ResponseHandle,
            body_handle: BodyHandle,
            streaming: u32,
        ) -> FastlyStatus;

        #[link_name = "status_get"]
        pub fn status_get(resp_handle: ResponseHandle, status: *mut u16) -> FastlyStatus;

        #[link_name = "status_set"]
        pub fn status_set(resp_handle: ResponseHandle, status: u16) -> FastlyStatus;

        #[link_name = "version_get"]
        pub fn version_get(resp_handle: ResponseHandle, version: *mut u32) -> FastlyStatus;

        #[link_name = "version_set"]
        pub fn version_set(resp_handle: ResponseHandle, version: u32) -> FastlyStatus;

        #[link_name = "close"]
        pub fn close(resp_handle: ResponseHandle) -> FastlyStatus;
    }
}

pub mod fastly_dictionary {
    use super::*;

    #[link(wasm_import_module = "fastly_dictionary")]
    extern "C" {
        #[link_name = "open"]
        pub fn open(
            name: *const u8,
            name_len: usize,
            dict_handle_out: *mut DictionaryHandle,
        ) -> FastlyStatus;

        #[link_name = "get"]
        pub fn get(
            dict_handle: DictionaryHandle,
            key: *const u8,
            key_len: usize,
            value: *mut u8,
            value_max_len: usize,
            nwritten: *mut usize,
        ) -> FastlyStatus;
    }
}

pub mod fastly_geo {
    use super::*;

    #[link(wasm_import_module = "fastly_geo")]
    extern "C" {
        #[link_name = "lookup"]
        pub fn lookup(
            addr_octets: *const u8,
            addr_len: usize,
            buf: *mut u8,
            buf_len: usize,
            nwritten_out: *mut usize,
        ) -> FastlyStatus;
    }
}
