use anyhow::Result;
use console::style;
use crossbeam_channel::{select, Receiver, Sender};
use indicatif::{MultiProgress, ProgressBar, ProgressDrawTarget};
use std::{
    collections::HashSet,
    path::{Path, PathBuf},
    thread::{self, JoinHandle},
    time::Instant,
};

use crate::{
    env::{resolve_env, ResolvedEnv},
    files::GlobFilesReader,
    hashes, lock,
    manifest::{store_and_clean_results, Manifest},
    outputs::{clean_latest_outputs, restore_outputs},
    tasks::{Task, TaskContext, TaskGraph},
    util::{self, DONE_STYLE, PROGRESS_STYLE, STATUS_STYLE},
    Config, FasterError,
};

const FASTER_ROOT_VAR: &str = "FASTER_ROOT";
const FASTER_TASK_VAR: &str = "FASTER_TASK";

struct TaskMessage(String, Task);

#[derive(Debug)]
pub enum CommandResult {
    Success(String),
    Failed(String),
    CachedFailed(String),
    Cached,
    Empty,
    Cancelled,
    Skipped,
}

pub fn run<P>(
    workdir: P,
    config: &Config,
    task_graph: &mut TaskGraph,
    draw_target: ProgressDrawTarget,
    job_count: usize,
    ctrlc_rx: Receiver<()>,
    disable_cache_global: bool,
) -> Result<()>
where
    P: AsRef<Path>,
{
    let started = Instant::now();

    let progress = MultiProgress::new();
    progress.set_draw_target(draw_target);

    let global_env = resolve_env(&config.environment)?;

    let job_count = std::cmp::min(task_graph.task_count(), job_count);

    let (job_tx, job_rx) = crossbeam_channel::unbounded::<TaskMessage>();
    let (done_tx, done_rx) = crossbeam_channel::unbounded::<(String, Result<CommandResult>)>();
    let (abort_tx, abort_rx) = crossbeam_channel::bounded::<()>(job_count);

    let disable_cache_global = disable_cache_global || config.disable_cache;

    let worker_handles = (0..job_count)
        .map(|_| {
            let job_rx = job_rx.clone();
            let done_tx = done_tx.clone();
            let abort_rx = abort_rx.clone();
            let progress = progress.clone();
            let workdir = PathBuf::from(workdir.as_ref());
            let global_env = global_env.clone();

            thread::spawn(move || {
                run_worker(
                    workdir,
                    &global_env,
                    &progress,
                    job_rx.clone(),
                    done_tx.clone(),
                    abort_rx.clone(),
                    disable_cache_global,
                )
            })
        })
        .collect::<Vec<JoinHandle<_>>>();

    let mut running_tasks = HashSet::<String>::new();
    let mut task_results = Vec::<(String, Result<CommandResult>)>::new();

    while task_graph.has_tasks() {
        let next_targets = task_graph
            .runnable_tasks()
            .filter(|name| !running_tasks.contains(name.as_str()))
            .collect::<Vec<String>>();

        for task_name in next_targets {
            let task = config
                .tasks
                .get(task_name.as_str())
                .ok_or_else(|| FasterError::TargetNotFound(task_name.to_owned()))?;

            running_tasks.insert(task_name.clone());
            job_tx.send(TaskMessage(task_name, task.clone()))?;
        }

        let should_abort;

        // Wait for any thread to finish its task, or an abort signal to arrive
        select! {
            recv(done_rx) -> msg => {
                let (task_name, task_result) = msg.expect("Failed to receive task message");
                should_abort = matches!(task_result, Err(_) | Ok(CommandResult::Failed(_)));

                running_tasks.remove(task_name.as_str());
                task_graph.complete_task(task_name.as_str());
                task_results.push((task_name, task_result));
            }
            recv(ctrlc_rx) -> _ => {
                should_abort = true;
            }
        }

        if should_abort {
            // Print the remaining tasks and mark them as skipped
            let remaining_tasks = task_graph
                .all_tasks()
                .filter(|name| !running_tasks.contains(name.as_str()))
                .collect::<Vec<String>>();

            for task_name in &remaining_tasks {
                let bar = progress.add(ProgressBar::new(1));

                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").black().dim()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("skipped").black().dim(),
                    task_name
                ));

                task_results.push((task_name.to_string(), Ok(CommandResult::Skipped)));
            }

            (0..job_count).for_each(|_| abort_tx.send(()).expect("failed to send abort message"));

            break;
        }
    }

    // Close the channels so the workers quit after
    // we're done
    drop(job_tx);

    for handle in worker_handles {
        handle.join().expect("failed to join handle");
    }

    // Insert a newline after the progress bars
    print!("\n\n");

    for result in &task_results {
        match result {
            (task_name, Ok(CommandResult::Failed(output))) => {
                println!("Script for task `{}` failed:", style(task_name).bold());
                println!("---\n{}\n---\n", output);
            }
            (task_name, Err(err)) => {
                println!("Failed to run task `{}`", style(task_name).bold());
                println!("---\n{}\n---\n", err);
            }
            _ => {}
        }
    }

    // TODO: Print detailed report
    let elapsed = util::format_millis(started.elapsed().as_millis() as u64);
    println!("Took {}\n", elapsed);

    Ok(())
}

fn run_worker<P>(
    workdir: P,
    global_env: &ResolvedEnv,
    progress: &MultiProgress,
    rx: Receiver<TaskMessage>,
    done_tx: Sender<(String, Result<CommandResult>)>,
    abort_rx: Receiver<()>,
    disable_cache_global: bool,
) where
    P: AsRef<Path>,
{
    let mut should_abort = false;

    while let Ok(TaskMessage(task_name, task)) = rx.recv() {
        let bar = progress.add(ProgressBar::new(1));
        let disable_cache = disable_cache_global || task.disable_cache;

        let started = Instant::now();
        let result = run_task(
            &workdir,
            &bar,
            &task_name,
            &task,
            global_env,
            disable_cache,
            &abort_rx,
        );

        match &result {
            Ok(CommandResult::Cancelled) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").yellow()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("cancelled").yellow().dim(),
                    task_name
                ));
            }
            Ok(CommandResult::Success(_output)) => {
                let elapsed = started.elapsed().as_millis() as u64;

                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✔").green()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style(util::format_millis(elapsed)).green().dim(),
                    task_name
                ));
            }
            Ok(CommandResult::Failed(_output)) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").red()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("failed").red().bold().dim(),
                    task_name
                ));

                should_abort = true;
            }
            Ok(CommandResult::CachedFailed(_output)) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").red()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("cached").red().bold().dim(),
                    task_name
                ));

                should_abort = true;
            }
            Ok(CommandResult::Cached) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✔").green()));

                bar.finish_with_message(format!("{} {}", style("cached").black().dim(), task_name));
            }
            Ok(CommandResult::Empty) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✔").green()));

                bar.finish_with_message(format!("{} {}", style("empty").black().dim(), task_name));
            }
            Err(_) => {
                bar.set_style(DONE_STYLE.clone());
                bar.set_prefix(format!("{}", style("✗").red()));
                bar.finish_with_message(format!(
                    "{} {}",
                    style("failed").red().bold().dim(),
                    task_name
                ));

                should_abort = true;
            }
            // This only ever happens when the task is not even run
            Ok(CommandResult::Skipped) => unreachable!(),
        }

        done_tx
            .send((task_name.to_owned(), result))
            .expect("failed to send done message");

        if should_abort {
            break;
        }
    }
}

pub fn run_task<P>(
    workdir: P,
    bar: &ProgressBar,
    task_name: &str,
    task: &Task,
    global_env: &ResolvedEnv,
    disable_cache: bool,
    abort_rx: &Receiver<()>,
) -> Result<CommandResult>
where
    P: AsRef<Path>,
{
    bar.set_style(STATUS_STYLE.clone());
    bar.set_prefix("⌛");
    bar.set_message(format!("{} {}", style("waiting").white().dim(), task_name));

    if let Ok(()) = abort_rx.try_recv() {
        return Ok(CommandResult::Cancelled);
    }

    let task_path = util::ensure_task_file_path(&workdir, task_name)?;
    let _lock = lock::accquire_task_lock(&task_path);

    let task_env = resolve_env(&task.environment)?;

    // TODO: Show an ellipsized command here? Split into lines?
    bar.set_style(PROGRESS_STYLE.clone());
    bar.set_message(format!("{} {}", style("running").blue().dim(), task_name));

    let ctx = TaskContext {
        workdir: PathBuf::from(workdir.as_ref()),
        task_name: task_name.to_owned(),
        task: task.clone(),
        task_path,
        task_env,
    };

    let mut manifest = Manifest::load(&ctx)?;

    // We're about to either restore old outputs or new create new inputs
    // by running the defined script. This means that we should first look
    // if we had previous outputs and remove them.
    clean_latest_outputs(&ctx, &manifest);

    let current_hash = hashes::calc_hash::<GlobFilesReader>(global_env, &ctx)?;

    if !disable_cache {
        if let Some(stored_run) = manifest.runs.iter().find(|run| run.sum == current_hash) {
            if stored_run.success {
                restore_outputs(&ctx, stored_run.sum)?;
                return Ok(CommandResult::Cached);
            }

            // The last run was found, and failed. Retrieve the logs and
            // return them again.
            return Ok(CommandResult::CachedFailed(String::new()));
        }
    }

    if task.script.is_none() {
        return Ok(CommandResult::Empty);
    }

    // Script is guaranteed to be Some here
    let script = task.script.as_ref().unwrap();

    // Run the actual task
    // TODO: Make the run command configurable
    let args = vec!["-exo", "pipefail", "-c", script];
    let mut command = duct::cmd("bash", &args)
        .stderr_to_stdout()
        .stdout_capture()
        .unchecked();

    command = command
        .dir(workdir.as_ref())
        .env(FASTER_ROOT_VAR, workdir.as_ref().as_os_str())
        .env(FASTER_TASK_VAR, &ctx.task_name);

    for (key, value) in global_env {
        command = command.env(key, value);
    }

    for (key, value) in &ctx.task_env {
        command = command.env(key, value);
    }

    let mut last_instant = Instant::now();
    let handle = command.start()?;

    loop {
        if let Ok(()) = abort_rx.try_recv() {
            // We don't really care about the child here,
            // even if it doesn't stop we don't want to block
            let _ = handle.kill();
            return Ok(CommandResult::Cancelled);
        }

        match handle.try_wait() {
            Ok(Some(result)) => {
                // TODO: Can we do this without cloning stdout? Might use up a lot
                // of memory when there's a lot of output from the command
                // NOTE: stdout and sterr are merged into stdout, so we're
                // getting both in the correct order by reading stdout here.
                let logs = String::from_utf8(result.stdout.clone())?;

                store_and_clean_results(
                    &ctx,
                    &mut manifest,
                    result.status.success(),
                    result.status.code(),
                    current_hash,
                    &logs,
                )?;

                return if result.status.success() {
                    Ok(CommandResult::Success(logs))
                } else {
                    Ok(CommandResult::Failed(logs))
                };
            }
            Err(err) => {
                return Err(err.into());
            }

            Ok(None) => {
                // TODO: Add a configurable command timeout?
                if last_instant.elapsed().as_millis() < 100 {
                    continue;
                }

                bar.inc(1);
                last_instant = Instant::now();
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use crossbeam_channel::bounded;

    use super::*;
    use crate::util::tests::{run_in_temp_dir, touch_file};

    #[test]
    fn clean() {
        let workdir = run_in_temp_dir();
        let (_, abort_rx) = bounded::<()>(1);
        let bar = ProgressBar::hidden();

        let env = ResolvedEnv::default();
        let task_name = "test";
        let task = Task {
            script: Some(
                r#"
                # check if the files exist and error out if they do
                [[ -f foo ]] && exit 1
                [[ -d bar ]] && exit 1
                [[ -f bar/baz ]] && exit 1

                touch foo
                mkdir bar
                touch bar/baz
            "#
                .to_owned(),
            ),
            environment: env.clone(),
            outputs: vec!["foo".to_owned(), "bar".to_owned()],
            ..Task::default()
        };

        // Task should fail if the files exist
        touch_file(workdir.as_ref().join("foo"));
        touch_file(workdir.as_ref().join("bar/baz"));

        let result = run_task(&workdir, &bar, task_name, &task, &env, true, &abort_rx);
        assert!(matches!(result, Ok(CommandResult::Failed(..))));

        // Use a new dir here because we can't rely on the files getting deleted immediately
        let workdir = run_in_temp_dir();

        // Task should succeed in empty dir
        let result = run_task(&workdir, &bar, task_name, &task, &env, true, &abort_rx);
        assert!(matches!(result, Ok(CommandResult::Success(..))));
        assert!(workdir.as_ref().join("foo").is_file());
        assert!(workdir.as_ref().join("bar/baz").is_file());

        // Another run should succeed as well, clearing the directory before running the script
        // and then outputting the files again
        let result = run_task(&workdir, &bar, task_name, &task, &env, true, &abort_rx);
        assert!(matches!(result, Ok(CommandResult::Success(..))));
        assert!(workdir.as_ref().join("foo").is_file());
        assert!(workdir.as_ref().join("bar/baz").is_file());
    }

    #[test]
    fn run_single() {
        let workdir = run_in_temp_dir();

        let config: crate::Config = serde_yaml::from_str(
            r#"
            tasks:
              build:
                script: "echo 'single build'"
            "#,
        )
        .unwrap();

        let mut task_graph =
            TaskGraph::single(&config, "build").expect("failed to build task graph");

        let (_, rx) = crossbeam_channel::unbounded::<()>();
        let result = run(
            &workdir,
            &config,
            &mut task_graph,
            indicatif::ProgressDrawTarget::hidden(),
            1,
            rx,
            false,
        );

        assert!(matches!(result, Ok(())));
    }

    #[test]
    fn run_empty() {
        let workdir = run_in_temp_dir();

        let config: crate::Config = serde_yaml::from_str(
            r#"
            tasks:
              build: {}
            "#,
        )
        .unwrap();

        let mut task_graph =
            TaskGraph::single(&config, "build").expect("failed to build task graph");

        let (_, rx) = crossbeam_channel::unbounded::<()>();
        let result = run(
            &workdir,
            &config,
            &mut task_graph,
            indicatif::ProgressDrawTarget::hidden(),
            1,
            rx,
            false,
        );

        assert!(matches!(result, Ok(())));
    }

    #[test]
    fn run_multi() {
        let workdir = run_in_temp_dir();

        let config: crate::Config = serde_yaml::from_str(
            r#"
            tasks:
              generate:
                script: "echo 'multi generate'"

              lint:
                script: "echo 'multi lint'"

              build:
                script: "echo 'multi build'"
                dependencies:
                - generate
                - lint
            "#,
        )
        .unwrap();

        let mut task_graph =
            TaskGraph::build(&config, "build").expect("failed to build task graph");

        let (_, rx) = crossbeam_channel::unbounded::<()>();
        let result = run(
            &workdir,
            &config,
            &mut task_graph,
            indicatif::ProgressDrawTarget::hidden(),
            1,
            rx,
            false,
        );

        assert!(matches!(result, Ok(())));
    }

    #[test]
    fn exit_code_failed() {
        let workdir = run_in_temp_dir();

        let config: crate::Config =
            serde_yaml::from_str(r#"tasks: {build: {script: "exit 1"}}"#).unwrap();

        let mut task_graph =
            TaskGraph::build(&config, "build").expect("failed to build task graph");

        let (_, rx) = crossbeam_channel::unbounded::<()>();
        let result = run(
            &workdir,
            &config,
            &mut task_graph,
            indicatif::ProgressDrawTarget::hidden(),
            1,
            rx,
            false,
        );

        assert!(matches!(result, Ok(())));
    }
}
