use anyhow::{Context, Result};
use std::fs::{self, File};

use crate::{files::FilesReader, manifest::Manifest, tasks::TaskContext, util, FasterError};

pub fn archive_outputs<R: FilesReader>(
    ctx: &TaskContext,
    current_hash: u64,
) -> Result<Vec<String>> {
    if ctx.task.outputs.is_empty() {
        return Ok(vec![]);
    }

    let paths = R::get_paths_from_patterns(&ctx.workdir, &ctx.task.outputs)?;

    if paths.files.is_empty() && !ctx.task.ignore_missing_outputs {
        return Err(FasterError::NoOutputsFound {
            task: ctx.task_name.to_owned(),
        }
        .into());
    }

    let archive_path = ctx.workdir.join(ctx.task_path.as_path()).join(format!(
        "{}.{}",
        current_hash,
        util::ARCHIVE_SUFFIX
    ));

    let archive_file = File::create(archive_path).with_context(|| {
        format!(
            "failed to create output archive for task `{}`",
            ctx.task_name
        )
    })?;

    let compressed_writer = snap::write::FrameEncoder::new(archive_file);
    let mut archive = tar::Builder::new(compressed_writer);

    archive.follow_symlinks(false);

    for path in &paths.files {
        // All patterns we receive here are absolute, so we need to trim
        // the prefix to get the path in the current workdir
        let path_in_workdir = path.strip_prefix(&ctx.workdir)?;

        archive
            .append_path_with_name(path, path_in_workdir)
            .with_context(|| format!("failed to append path {:?} to archive", path))?;
    }

    archive.finish()?;

    let results = paths
        .entries
        .iter()
        .map(|it| String::from(it.to_string_lossy()))
        .collect();

    Ok(results)
}

pub fn restore_outputs(ctx: &TaskContext, hash: u64) -> Result<()> {
    // Adding/removing outputs will invalidate the hash,
    // which means we can assume there were no defined outputs
    // if there are no outputs in the task config.
    if ctx.task.outputs.is_empty() {
        return Ok(());
    }

    let archive_path = ctx.workdir.join(ctx.task_path.as_path()).join(format!(
        "{}.{}",
        hash,
        util::ARCHIVE_SUFFIX
    ));

    if !archive_path.is_file() {
        return Ok(());
    }

    let archive_file = File::open(archive_path)?;
    let compressed_reader = snap::read::FrameDecoder::new(archive_file);
    let mut archive = tar::Archive::new(compressed_reader);

    // Since the output patterns are always relative to the current directory,
    // we can safely unpack cached artifacts to the current directory as well
    Ok(archive.unpack(&ctx.workdir)?)
}

/// Remove the outputs defined fo the latest run in the given manifest. Returns
/// true if any outputs were cleaned, false otherwise.
pub fn clean_latest_outputs(ctx: &TaskContext, manifest: &Manifest) -> bool {
    if let Some(previous_outputs) = manifest
        .get_latest_run()
        .and_then(|run| run.outputs.as_ref())
    {
        if previous_outputs.is_empty() {
            return false;
        }

        // These were previously added to the archive, so they are guaranteed to
        // be files. Just try to delete them 1 by 1 and ignore failures.
        for path in previous_outputs {
            let path = ctx.workdir.join(path);

            if path.is_dir() {
                let _ = fs::remove_dir_all(path);
            } else {
                let _ = fs::remove_file(path);
            }
        }

        return true;
    }

    false
}

#[cfg(test)]
mod tests {
    use std::path::PathBuf;

    use super::*;
    use crate::{
        env::ResolvedEnv,
        files::GlobFilesReader,
        tasks::Task,
        util::{
            ensure_task_file_path,
            tests::{run_in_temp_dir, touch_file},
        },
    };

    #[test]
    fn archive_and_restore() {
        let files = vec![
            "output/foo/1.txt",
            "output/foo/2.txt",
            "output/foo/3.txt",
            "output/foo/bar/1.txt",
            "output/foo/bar/2.txt",
            "output/foo/bar/3.txt",
            "output/bar/a",
            "output/bar/b",
            "output/bar/c",
        ];

        let workdir = run_in_temp_dir();
        let task_name = "test";

        let task = Task {
            outputs: vec!["output/foo/**/*.txt".to_owned(), "output/bar".to_owned()],
            ..Default::default()
        };

        for file in &files {
            touch_file(workdir.as_ref().join(file));
        }

        let task_path =
            ensure_task_file_path(&workdir, task_name).expect("failed to ensure task path");

        let ctx = TaskContext {
            workdir: PathBuf::from(workdir.as_ref()),
            task_name: task_name.to_owned(),
            task,
            task_path,
            task_env: ResolvedEnv::new(),
        };

        archive_outputs::<GlobFilesReader>(&ctx, 0u64).expect("failed to save outputs");

        let mut archive_path = PathBuf::from(&ctx.task_path);
        archive_path.push("0.tar.sz");
        assert!(archive_path.is_file());

        let mut output_dir = PathBuf::from(&workdir.as_ref());
        output_dir.push("output");
        std::fs::remove_dir_all(output_dir).expect("failed to delete output dir");

        // Files should now be deleted
        for file in &files {
            assert!(
                !workdir.as_ref().join(file).is_file(),
                "expected file {:?} to not exist",
                file
            );
        }

        restore_outputs(&ctx, 0u64).expect("failed to restore outputs");

        // Files should now be restored
        for file in &files {
            assert!(
                workdir.as_ref().join(file).is_file(),
                "expected file {:?} to exist",
                file
            );
        }
    }
}
