use anyhow::{Context, Result};
use serde::{Deserialize, Serialize};
use std::{
    collections::HashSet,
    fs::File,
    io::Write,
    time::{SystemTime, UNIX_EPOCH},
};

use crate::{files::GlobFilesReader, outputs::archive_outputs, tasks::TaskContext, util};

const MANIFEST_FILE: &str = "manifest.yaml";

#[derive(Debug, Clone, Deserialize, Serialize, Default)]
#[serde(rename_all = "camelCase")]
pub struct Manifest {
    pub latest: Option<u64>,
    pub runs: Vec<Run>,
}

#[derive(Debug, Clone, Deserialize, Serialize, Default)]
#[serde(rename_all = "camelCase")]
pub struct Run {
    pub sum: u64,
    pub timestamp: u64,
    pub success: bool,

    pub exit_code: Option<i32>,
    pub outputs: Option<Vec<String>>,
}

impl Manifest {
    pub fn load(ctx: &TaskContext) -> Result<Manifest> {
        let manifest_path = ctx
            .workdir
            .join(ctx.task_path.as_path())
            .join(MANIFEST_FILE);

        if !manifest_path.is_file() {
            return Ok(Self::default());
        }

        let file = File::open(manifest_path)
            .with_context(|| format!("failed to read manifest for task `{}`", ctx.task_name))?;
        let parsed = serde_yaml::from_reader(file)
            .with_context(|| format!("failed to parse manifest for task `{}`", ctx.task_name))?;

        Ok(parsed)
    }

    pub fn get_latest_run(&self) -> Option<&Run> {
        self.latest
            .and_then(|latest_hash| self.runs.iter().find(|task| task.sum == latest_hash))
    }

    pub fn save(&self, ctx: &TaskContext) -> Result<()> {
        let manifest_path = ctx
            .workdir
            .join(ctx.task_path.as_path())
            .join(MANIFEST_FILE);

        let file = File::create(manifest_path)
            .with_context(|| format!("failed to create manifest for task `{}`", ctx.task_name))?;

        serde_yaml::to_writer(file, self)?;

        Ok(())
    }
}

// Update and save the manifest, store the latest logs and archive the latest outputs,
// as well as remove old logs and archives according to the global and task settings.
pub fn store_and_clean_results(
    ctx: &TaskContext,
    manifest: &mut Manifest,
    success: bool,
    exit_code: Option<i32>,
    latest_hash: u64,
    logs: &str,
) -> Result<()> {
    let saved_outputs = if success && !ctx.task.outputs.is_empty() {
        Some(archive_outputs::<GlobFilesReader>(ctx, latest_hash)?)
    } else {
        None
    };

    let log_path = ctx.workdir.join(ctx.task_path.as_path()).join(format!(
        "{}.{}",
        latest_hash,
        util::LOG_SUFFIX
    ));

    let mut file = File::create(log_path).context("failed to create log file")?;
    file.write_all(logs.as_bytes())?;

    // First, add the latest run to the list
    manifest.latest = Some(latest_hash);
    manifest.runs.push(Run {
        sum: latest_hash,
        outputs: saved_outputs,
        timestamp: SystemTime::now().duration_since(UNIX_EPOCH)?.as_millis() as u64,
        success,
        exit_code,
    });

    // This will sort by oldest to newest, so we need to reverse
    manifest.runs.sort_by_key(|run| run.timestamp);
    manifest.runs.reverse();

    // Now, delete the oldest logs according to KEEP_LAST_RESULTS
    // TODO: Make KEEP_LAST_RESULTS configurable
    let cleaned_runs = manifest
        .runs
        .iter()
        .skip(util::KEEP_LAST_RESULTS)
        .map(|run| {
            let log_path = ctx.workdir.join(ctx.task_path.as_path()).join(format!(
                "{}.{}",
                run.sum,
                util::LOG_SUFFIX
            ));

            let archive_path = ctx.workdir.join(ctx.task_path.as_path()).join(format!(
                "{}.{}",
                run.sum,
                util::ARCHIVE_SUFFIX
            ));

            // TODO: Log the error in verbose mode here, this is
            // not a blocker though
            let _ = std::fs::remove_file(log_path);
            let _ = std::fs::remove_file(archive_path);

            run.sum
        })
        .collect::<HashSet<_>>();

    // Drop the deleted runs from the manifest
    manifest.runs.retain(|run| !cleaned_runs.contains(&run.sum));
    manifest.save(ctx).context("failed to save context")?;

    Ok(())
}

#[cfg(test)]
mod tests {
    use std::{fs, path::PathBuf, thread, time::Duration};

    use crate::{
        env::ResolvedEnv,
        tasks::Task,
        util::{tests::run_in_temp_dir, FASTER_DIR},
    };

    use super::*;

    #[test]
    fn store_results() {
        let workdir = run_in_temp_dir();

        let ctx = TaskContext {
            workdir: PathBuf::from(workdir.as_ref()),
            task_name: "test".to_owned(),
            task: Task::default(),
            task_path: PathBuf::from(FASTER_DIR).join("test"),
            task_env: ResolvedEnv::default(),
        };

        fs::create_dir_all(workdir.as_ref().join(FASTER_DIR).join("test"))
            .expect("failed to create task dir");

        let mut manifest = Manifest::load(&ctx).expect("failed to load manifest");
        let result = store_and_clean_results(&ctx, &mut manifest, true, Some(0), 1, "foo");
        assert!(matches!(result, Ok(())));
        assert!(matches!(
            manifest,
            Manifest {
                latest: Some(1),
                ..
            }
        ));
        assert!(matches!(
            manifest.runs[..],
            [Run {
                sum: 1,
                success: true,
                exit_code: Some(0),
                ..
            }]
        ));

        assert!(workdir
            .as_ref()
            .join(FASTER_DIR)
            .join("test/1.log")
            .is_file());

        // Store more results and make sure the ordering is correct. We have to wait for
        // at least 1ms since the ordering is done by timestamp.
        thread::sleep(Duration::from_millis(1));
        let result = store_and_clean_results(&ctx, &mut manifest, true, Some(0), 2, "foo");
        assert!(matches!(result, Ok(())));
        assert!(matches!(
            manifest,
            Manifest {
                latest: Some(2),
                ..
            }
        ));
        assert!(matches!(
            manifest.runs[..],
            [
                Run {
                    sum: 2,
                    success: true,
                    exit_code: Some(0),
                    ..
                },
                Run {
                    sum: 1,
                    success: true,
                    exit_code: Some(0),
                    ..
                }
            ]
        ));

        thread::sleep(Duration::from_millis(1));
        let result = store_and_clean_results(&ctx, &mut manifest, false, Some(1), 3, "foo");
        assert!(matches!(result, Ok(())));
        assert!(matches!(
            manifest,
            Manifest {
                latest: Some(3),
                ..
            }
        ));
        assert!(matches!(
            manifest.runs[..],
            [
                Run {
                    sum: 3,
                    success: false,
                    exit_code: Some(1),
                    ..
                },
                Run {
                    sum: 2,
                    success: true,
                    exit_code: Some(0),
                    ..
                },
                Run {
                    sum: 1,
                    success: true,
                    exit_code: Some(0),
                    ..
                }
            ]
        ));

        // Check that the correct result is being cleared
        thread::sleep(Duration::from_millis(1));
        let result = store_and_clean_results(&ctx, &mut manifest, true, Some(0), 4, "foo");
        assert!(matches!(result, Ok(())));
        assert!(matches!(
            manifest,
            Manifest {
                latest: Some(4),
                ..
            }
        ));
        assert!(matches!(
            manifest.runs[..],
            [
                Run {
                    sum: 4,
                    success: true,
                    exit_code: Some(0),
                    ..
                },
                Run {
                    sum: 3,
                    success: false,
                    exit_code: Some(1),
                    ..
                },
                Run {
                    sum: 2,
                    success: true,
                    exit_code: Some(0),
                    ..
                },
            ]
        ));
    }
}
