use anyhow::{Context, Result};
use clap::{Parser, Subcommand};
use console::style;
use env::ResolvedEnv;
use files::{FilesReader, GlobFilesReader};
use indicatif::{ProgressBar, ProgressDrawTarget};
use manifest::Manifest;
use outputs::clean_latest_outputs;
use serde::Deserialize;
use std::{collections::HashMap, fs, path::Path};
use tasks::TaskContext;
use thiserror::Error;
use util::{ensure_task_file_path, DONE_STYLE};

pub mod env;
pub mod files;
pub mod hashes;
pub mod lock;
pub mod manifest;
pub mod outputs;
pub mod run;
pub mod tasks;
pub mod util;

#[derive(Debug, Clone, Deserialize, Default)]
#[serde(default, deny_unknown_fields, rename_all = "camelCase")]
pub struct Config {
    pub environment: HashMap<String, String>,
    pub disable_cache: bool,
    pub tasks: HashMap<String, tasks::Task>,
}

#[derive(Error, Debug)]
pub enum FasterError {
    #[error("cycle found during dependency resolution")]
    DependencyCycle,

    #[error("target `{0}` not found")]
    TargetNotFound(String),

    #[error("target `{task:?}` not found, referenced by `{parent:?}`")]
    DependencyTargetNotFound { parent: String, task: String },

    #[error("no outputs were matched for task `{task:?}`")]
    NoOutputsFound { task: String },

    #[error("environment expression `{expr:?}` was invalid ({stderr:?})")]
    EnvExpressionError { expr: String, stderr: String },
}

/// Simple and smart language-agnostic task runner
#[derive(Parser, Debug)]
#[clap(author, version, about)]
struct Args {
    #[clap(subcommand)]
    subcommand: Commands,

    /// Provide a custom config file path
    #[clap(short, long, default_value = "faster.yaml")]
    config: String,
}

#[derive(Subcommand, Debug)]
enum Commands {
    /// List all defined tasks
    Tasks,

    /// List all input files for a specific task
    Inputs {
        /// Task name for which files should be printed
        task_name: String,

        /// Print files for all direct and transitive dependencies of the task
        #[clap(short, long)]
        include_deps: bool,
    },

    /// Run a task and all its dependencies
    Run {
        /// Task name to be run
        task_name: String,

        /// Don't run dependencies of the task
        #[clap(short, long)]
        skip_dependencies: bool,

        /// Number of tasks to run in parallel
        #[clap(short, long, default_value_t = 4)]
        parallel_tasks: usize,

        /// Globally disable the cache for any tasks
        /// Overrides any options set in the config file
        #[clap(short, long)]
        no_cache: bool,
    },

    /// Remove outputs and delete stored runs for a task
    Clean {
        /// Task name to clean, or nothing to clean all tasks
        task_name: Option<String>,
    },

    /// Parse and validate all defined tasks
    Validate,
}

fn main() -> Result<()> {
    let (ctrlc_tx, ctrlc_rx) = crossbeam_channel::unbounded();
    ctrlc::set_handler(move || ctrlc_tx.send(()).expect("Failed to send abort signal"))
        .expect("Failed to set abort signal handler");

    let args = Args::parse();
    let workdir = std::env::current_dir().context("failed to get current dir")?;

    match args.subcommand {
        Commands::Run {
            task_name,
            skip_dependencies,
            parallel_tasks,
            no_cache,
        } => {
            let config = read_config(&args.config)?;

            let mut task_graph = if skip_dependencies {
                let task_graph = tasks::TaskGraph::single(&config, task_name.as_str())?;

                println!(
                    "{}",
                    style(format!(
                        "Running task `{}` without dependencies\n",
                        task_name,
                    ))
                    .dim(),
                );

                task_graph
            } else {
                let task_graph = tasks::TaskGraph::build(&config, task_name.as_str())?;
                let resolved_tasks = task_graph.resolved_task_names();

                println!(
                    "{}",
                    style(format!(
                        "Resolved {} dependencies for task `{}`\n",
                        resolved_tasks.len(),
                        task_name,
                    ))
                    .dim(),
                );

                task_graph
            };

            run::run(
                &workdir,
                &config,
                &mut task_graph,
                ProgressDrawTarget::stderr(),
                parallel_tasks,
                ctrlc_rx,
                no_cache,
            )?;
        }
        Commands::Tasks => {
            let config = read_config(&args.config)?;
            for task_name in config.tasks.keys() {
                println!("{}", task_name);
            }
        }
        Commands::Inputs {
            task_name,
            include_deps,
        } => {
            let config = read_config(&args.config)?;

            if !include_deps {
                let task = config
                    .tasks
                    .get(&task_name)
                    .ok_or(FasterError::TargetNotFound(task_name))?;

                if task.inputs.is_empty() {
                    return Ok(());
                }

                for file in GlobFilesReader::get_paths_from_patterns(&workdir, &task.inputs)?.files
                {
                    println!("{}", file.to_string_lossy());
                }

                return Ok(());
            }

            let task_graph = tasks::TaskGraph::build(&config, task_name.as_str())?;

            let mut all_paths = Vec::<String>::new();
            for task_name in task_graph.all_tasks() {
                let task = config
                    .tasks
                    .get(&task_name)
                    .ok_or(FasterError::TargetNotFound(task_name))?;

                for file in GlobFilesReader::get_paths_from_patterns(&workdir, &task.inputs)?.files
                {
                    all_paths.push(file.to_string_lossy().to_string());
                }
            }

            all_paths.dedup();
            for path in &all_paths {
                println!("{}", path);
            }
        }
        Commands::Clean { task_name } => {
            let config = read_config(&args.config)?;

            let tasks_to_clean = match task_name {
                Some(task_name) => vec![(
                    task_name.to_owned(),
                    config
                        .tasks
                        .get(&task_name)
                        .cloned()
                        .ok_or(FasterError::TargetNotFound(task_name))?,
                )],

                None => config.tasks.into_iter().collect::<Vec<_>>(),
            };

            for (task_name, task) in tasks_to_clean {
                let bar = ProgressBar::new(1)
                    .with_style(DONE_STYLE.clone())
                    .with_prefix("·")
                    .with_message(format!("Cleaning {}", style(&task_name).bold()));

                let task_path = ensure_task_file_path(workdir.clone(), &task_name)?;
                let ctx = TaskContext {
                    workdir: workdir.clone(),
                    task_name: task_name.clone(),
                    task: task.clone(),
                    task_path: task_path.clone(),
                    task_env: ResolvedEnv::default(),
                };

                let manifest = Manifest::load(&ctx)?;
                let cleaned = clean_latest_outputs(&ctx, &manifest);

                fs::remove_dir_all(workdir.join(task_path))?;

                if cleaned {
                    bar.set_prefix(format!("{}", style("✔").green()));
                    bar.finish_with_message(format!("Cleaned {}", style(&task_name).bold()));
                } else {
                    bar.set_prefix(format!("{}", style("✔").black().dim()));
                    bar.finish_with_message(format!("No outputs for {}", style(&task_name).bold()));
                }
            }
        }
        Commands::Validate => {
            let config = read_config(&args.config)?;
            let mut errs = HashMap::<&str, anyhow::Error>::new();

            for task_name in config.tasks.keys() {
                let result = tasks::TaskGraph::build(&config, task_name.as_str());
                if let Err(err) = result {
                    errs.insert(task_name, err);
                }
            }

            if errs.is_empty() {
                println!("No errors found!");
                return Ok(());
            }

            for (task_name, err) in &errs {
                println!("Error in task `{}`: {} ", task_name, err)
            }
        }
    }

    Ok(())
}

fn read_config<P>(path: P) -> Result<Config>
where
    P: AsRef<Path>,
{
    let contents = std::fs::read_to_string(path).context("Failed to open config file")?;
    serde_yaml::from_str(&contents).context("Failed to parse config file")
}
