use anyhow::Result;
use std::hash::Hasher;
use twox_hash::XxHash64;

use crate::{env::ResolvedEnv, files::FilesReader, tasks::TaskContext};

pub fn calc_hash<R: FilesReader>(global_env: &ResolvedEnv, ctx: &TaskContext) -> Result<u64> {
    if ctx.task.inputs.is_empty()
        && ctx.task.outputs.is_empty()
        && ctx.task.script.is_none()
        && global_env.is_empty()
        && ctx.task_env.is_empty()
    {
        return Ok(0);
    }

    let mut hash = XxHash64::with_seed(0);

    hash.write(ctx.task.outputs.join(",").as_bytes());

    if let Some(script) = ctx.task.script.as_ref() {
        hash.write(script.as_bytes());
    }

    let paths = R::get_paths_from_patterns(&ctx.workdir, &ctx.task.inputs)?;

    for path in paths.files {
        let contents = R::read_file(path)?;
        hash.write(&contents);
    }

    for (key, value) in global_env {
        hash.write(key.as_bytes());
        hash.write(value.as_bytes());
    }

    for (key, value) in &ctx.task_env {
        hash.write(key.as_bytes());
        hash.write(value.as_bytes());
    }

    Ok(hash.finish())
}

#[cfg(test)]
mod tests {
    use lazy_static::lazy_static;
    use std::{
        path::{Path, PathBuf},
        str::FromStr,
    };

    use super::*;
    use crate::{files::ResolvedPaths, tasks::Task, util::tests::random_string};

    struct TestReader;

    impl FilesReader for TestReader {
        fn get_paths_from_patterns<P>(_: P, patterns: &[String]) -> Result<ResolvedPaths>
        where
            P: AsRef<Path>,
        {
            let results = patterns
                .iter()
                .map(|pattern| PathBuf::from_str(pattern).expect("failed to create path"))
                .collect::<Vec<_>>();

            Ok(ResolvedPaths {
                files: results.clone(),
                entries: results,
            })
        }

        fn read_file<P>(path: P) -> Result<Vec<u8>>
        where
            P: AsRef<Path>,
        {
            Ok(path
                .as_ref()
                .to_str()
                .expect("invalid path")
                .as_bytes()
                .to_vec())
        }
    }

    lazy_static! {
        static ref EMPTY_ENV: ResolvedEnv = ResolvedEnv::new();
        static ref EMPTY_CONTEXT: TaskContext = TaskContext::default();
    }

    #[test]
    fn stable_empty_hash() {
        let empty = calc_hash::<TestReader>(&EMPTY_ENV, &EMPTY_CONTEXT).expect("failed to hash");

        assert_eq!(empty, 0, "expected empty hash to be 0");

        let empty = calc_hash::<TestReader>(&EMPTY_ENV, &EMPTY_CONTEXT).expect("failed to hash");

        assert_eq!(empty, 0, "expected empty hash to be stable");
    }

    #[test]
    fn script_changes() {
        let task_before = TaskContext {
            task: Task {
                script: Some(String::from("before")),
                ..Task::default()
            },
            ..TaskContext::default()
        };

        let task_after = TaskContext {
            task: Task {
                script: Some(String::from("after")),
                ..Task::default()
            },
            ..TaskContext::default()
        };

        let before = calc_hash::<TestReader>(&EMPTY_ENV, &task_before).expect("failed to hash");
        let again = calc_hash::<TestReader>(&EMPTY_ENV, &task_before).expect("failed to hash");
        let after = calc_hash::<TestReader>(&EMPTY_ENV, &task_after).expect("failed to hash");

        assert_eq!(before, again, "expected script hash to be stable");
        assert_ne!(before, after, "expected script hash to reflect changes");
    }

    #[test]
    fn file_changes() {
        let task_before = TaskContext {
            task: Task {
                inputs: vec!["file1".to_owned(), "file2".to_owned()],
                ..Task::default()
            },
            ..TaskContext::default()
        };

        let task_after = TaskContext {
            task: Task {
                inputs: vec!["file1".to_owned()],
                ..Task::default()
            },
            ..TaskContext::default()
        };

        let before = calc_hash::<TestReader>(&EMPTY_ENV, &task_before).expect("failed to hash");
        let again = calc_hash::<TestReader>(&EMPTY_ENV, &task_before).expect("failed to hash");
        let after = calc_hash::<TestReader>(&EMPTY_ENV, &task_after).expect("failed to hash");

        assert_eq!(before, again, "expected files hash to be stable");
        assert_ne!(before, after, "expected files hash to reflect changes");
    }

    #[test]
    fn global_env_changes() {
        let mut env = ResolvedEnv::new();
        env.insert(random_string(), random_string());

        let task = TaskContext {
            task: Task { ..Task::default() },
            ..TaskContext::default()
        };

        let before = calc_hash::<TestReader>(&env, &task).expect("failed to hash");
        let again = calc_hash::<TestReader>(&env, &task).expect("failed to hash");

        env.insert(random_string(), random_string());
        let after = calc_hash::<TestReader>(&env, &task).expect("failed to hash");

        assert_eq!(before, again, "expected global env hash to be stable");
        assert_ne!(before, after, "expected global env hash to reflect changes");
    }

    #[test]
    fn task_env_changes() {
        let mut env = ResolvedEnv::new();
        env.insert(random_string(), random_string());

        let task_before = TaskContext {
            task: Task { ..Task::default() },
            task_env: env.clone(),
            ..TaskContext::default()
        };

        let before = calc_hash::<TestReader>(&EMPTY_ENV, &task_before).expect("failed to hash");
        let again = calc_hash::<TestReader>(&EMPTY_ENV, &task_before).expect("failed to hash");

        env.insert(random_string(), random_string());
        let task_after = TaskContext {
            task: Task { ..Task::default() },
            task_env: env.clone(),
            ..TaskContext::default()
        };

        let after = calc_hash::<TestReader>(&EMPTY_ENV, &task_after).expect("failed to hash");

        assert_eq!(before, again, "expected task env hash to be stable");
        assert_ne!(before, after, "expected task env hash to reflect changes");
    }
}
