use anyhow::Result;
use std::{
    collections::HashSet,
    path::{Path, PathBuf},
};

/// ResolvedPaths contains a list of paths resolved from a pattern.
/// The difference between files and entries is that files are guaranteed
/// to always point to files and symlinks, but never to directories.
/// Entries doesn't contain any files that were matched as children of
/// a whole directory that was matched, but the directory instead.
pub struct ResolvedPaths {
    pub files: Vec<PathBuf>,
    pub entries: Vec<PathBuf>,
}

pub trait FilesReader {
    fn get_paths_from_patterns<P>(workdir: P, patterns: &[String]) -> Result<ResolvedPaths>
    where
        P: AsRef<Path>;
    fn read_file<P>(path: P) -> Result<Vec<u8>>
    where
        P: AsRef<Path>;
}

pub struct GlobFilesReader;

impl FilesReader for GlobFilesReader {
    fn get_paths_from_patterns<P>(workdir: P, patterns: &[String]) -> Result<ResolvedPaths>
    where
        P: AsRef<Path>,
    {
        let mut result = ResolvedPaths {
            files: vec![],
            entries: vec![],
        };

        let mut directories = HashSet::new();

        for pattern in patterns {
            let absolute_pattern = workdir.as_ref().join(pattern);
            let files = glob::glob(&absolute_pattern.to_string_lossy())?;

            for file in files {
                let file = file?;

                if file.is_file() {
                    result.files.push(file.clone());
                    result.entries.push(file);
                    continue;
                }

                if file.is_dir() {
                    directories.insert(file.clone());
                    result.entries.push(file);
                    continue;
                }
            }
        }

        for dir in directories.iter() {
            let dir_pattern = dir.join("**/*");
            let files = glob::glob(&dir_pattern.to_string_lossy())?;

            for file in files {
                let file = file?;

                if file.is_file() {
                    result.files.push(file);
                    continue;
                }
            }
        }

        result.files.dedup();
        result.entries.dedup();

        Ok(result)
    }

    fn read_file<P>(path: P) -> Result<Vec<u8>>
    where
        P: AsRef<Path>,
    {
        Ok(std::fs::read(path)?)
    }
}
