use anyhow::Result;
use std::path::{Path, PathBuf};

pub const SUM_FILE: &str = "sum";
pub const FASTER_DIR: &str = ".faster";

pub const KEEP_LAST_RESULTS: usize = 3;

pub const LOG_SUFFIX: &str = "log";
pub const ARCHIVE_SUFFIX: &str = "tar.sz";

pub fn format_millis(millis: u64) -> String {
    if millis >= 1_000 {
        let seconds = millis as f32 / 1000f32;
        return format!("{:.3}s", seconds);
    }

    format!("{}ms", millis)
}

pub fn ensure_task_file_path<P>(workdir: P, task_name: &str) -> Result<PathBuf>
where
    P: AsRef<Path>,
{
    let path = workdir.as_ref().join(FASTER_DIR).join(task_name);

    if !path.is_dir() {
        std::fs::create_dir_all(&path)?;
    }

    Ok(path)
}

#[cfg(test)]
pub mod tests {
    use rand::Rng;
    use tempfile::TempDir;

    use super::*;

    pub fn run_in_temp_dir() -> TempDir {
        tempfile::tempdir().expect("failed to create temp dir")
    }

    pub fn random_string() -> String {
        rand::thread_rng()
            .sample_iter(&rand::distributions::Alphanumeric)
            .take(20)
            .map(char::from)
            .collect()
    }

    #[test]
    fn format_millis_works() {
        assert_eq!(
            format_millis(123),
            "123ms",
            "expected formatting to work for values < 1s"
        );

        assert_eq!(
            format_millis(1234),
            "1.234s",
            "expected formatting to work for values > 1s"
        );
    }

    #[test]
    fn ensure_task_file_path_works() {
        let workdir = run_in_temp_dir();
        let task_name = "test";

        let result =
            ensure_task_file_path(&workdir, task_name).expect("failed to ensure task path");
        assert!(result.is_dir(), "expected task directory to be created");
    }
}
