use anyhow::Result;
use petgraph::{stable_graph::StableDiGraph, EdgeDirection};
use serde::Deserialize;
use std::collections::{hash_map, HashMap, VecDeque};

use crate::{Config, FasterError};

type NodeIndex = petgraph::graph::NodeIndex<petgraph::stable_graph::DefaultIx>;

#[derive(Debug, Clone, Deserialize, Default)]
pub struct Task {
    pub script: Option<String>,

    #[serde(default)]
    pub environment: HashMap<String, String>,

    #[serde(default)]
    pub dependencies: Vec<String>,

    #[serde(default)]
    pub files: Vec<String>,

    #[serde(default)]
    pub outputs: Vec<String>,

    #[serde(default, rename = "ignoreMissingOutputs")]
    pub ignore_missing_outputs: bool,
}

#[derive(Debug)]
pub struct TaskGraph<'a> {
    graph: StableDiGraph<&'a str, ()>,
    indices: HashMap<&'a str, NodeIndex>,
}

impl<'a> TaskGraph<'a> {
    pub fn task_count(&self) -> usize {
        self.graph.node_count()
    }

    pub fn has_tasks(&self) -> bool {
        self.graph.node_count() > 0
    }

    pub fn runnable_tasks(&self) -> impl Iterator<Item = String> + '_ {
        self.graph.externals(EdgeDirection::Incoming).map(|idx| {
            let node = self
                .graph
                .node_weight(idx)
                .expect("should have found node weight");

            (*node).to_owned()
        })
    }

    pub fn all_tasks(&self) -> impl Iterator<Item = String> + '_ {
        self.graph.node_weights().map(|weight| (*weight).to_owned())
    }

    pub fn complete_task(&mut self, task_name: &str) {
        let done_idx = self
            .indices
            .remove(task_name)
            .expect("should have found done task in graph");

        self.indices.remove(task_name);
        self.graph.remove_node(done_idx);
    }

    pub fn resolved_task_names(&self) -> Vec<&'a str> {
        self.indices.keys().copied().collect()
    }

    pub fn single(config: &'a Config, task_name: &'a str) -> Result<TaskGraph<'a>> {
        let _ = config
            .tasks
            .get(task_name)
            .ok_or_else(|| FasterError::TargetNotFound(task_name.to_owned()))?;

        let mut graph = StableDiGraph::<&str, ()>::new();
        let mut indices = HashMap::<&str, NodeIndex>::new();

        let index = graph.add_node(task_name);
        indices.insert(task_name, index);

        Ok(Self { graph, indices })
    }

    pub fn build(config: &'a Config, task_name: &'a str) -> Result<TaskGraph<'a>> {
        let task = config
            .tasks
            .get(task_name)
            .ok_or_else(|| FasterError::TargetNotFound(task_name.to_owned()))?;

        let mut graph = StableDiGraph::<&str, ()>::new();
        let mut task_queue = VecDeque::<(&str, &Task, Option<NodeIndex>)>::new();
        let mut indices = HashMap::<&str, NodeIndex>::new();

        task_queue.push_back((task_name, task, None));
        while !task_queue.is_empty() {
            let (task_name, task, parent) = task_queue
                .pop_front()
                .expect("task queue should not be empty");

            let (already_added, indices) =
                if let hash_map::Entry::Vacant(e) = indices.entry(task_name) {
                    let index = graph.add_node(task_name);
                    e.insert(index);

                    (false, index)
                } else {
                    (true, indices[task_name])
                };

            if let Some(parent) = parent {
                graph.add_edge(indices, parent, ());
            }

            if already_added {
                continue;
            }

            for dep_name in &task.dependencies {
                let dep_config =
                    config
                        .tasks
                        .get(dep_name)
                        .ok_or(FasterError::DependencyTargetNotFound {
                            parent: task_name.to_owned(),
                            task: dep_name.to_owned(),
                        })?;

                task_queue.push_back((dep_name, dep_config, Some(indices)));
            }
        }

        if petgraph::algo::is_cyclic_directed(&graph) {
            // TODO: print the cycle using toposort algo
            return Err(FasterError::DependencyCycle.into());
        }

        Ok(Self { graph, indices })
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn cycle_throws_error() {
        let config = serde_yaml::from_str(
            r#"
            tasks:
              generate:
                dependencies:
                - build

              build:
                dependencies:
                - generate
            "#,
        )
        .unwrap();

        match TaskGraph::build(&config, "build")
            .unwrap_err()
            .downcast_ref::<FasterError>()
        {
            Some(FasterError::DependencyCycle) => {}
            Some(err) => panic!("expected dependency cycle error, got: {:?}", err),
            _ => panic!("expected error"),
        };
    }

    #[test]
    fn dependencies_must_exist() {
        let config = serde_yaml::from_str(
            r#"
            tasks:
              build:
                dependencies:
                - does not exist
            "#,
        )
        .unwrap();

        match TaskGraph::build(&config, "build")
            .unwrap_err()
            .downcast_ref::<FasterError>()
        {
            Some(FasterError::DependencyTargetNotFound { .. }) => {}
            Some(err) => panic!("expected dependency not found error, got: {:?}", err),
            _ => panic!("expected error"),
        };
    }

    #[test]
    fn dependency_resolution_works() {
        let config = serde_yaml::from_str(
            r#"
            tasks:
              lint protobuf files:
                script: ""

              generate protobuf:
                script: ""
                dependencies:
                - lint protobuf files

              generate api stubs:
                script: ""

              generate protobuf client stubs:
                script: ""
                files: []
                dependencies:
                - generate protobuf

              generate protobuf server stubs:
                script: ""
                files: []
                dependencies:
                - generate protobuf

              lint server code:
                script: ""

              build server:
                script: ""
                files: []
                dependencies:
                - generate protobuf client stubs
                - generate protobuf server stubs
                - generate api stubs
                - lint server code
            "#,
        )
        .unwrap();

        TaskGraph::build(&config, "build server").expect("failed to build graph");

        // TODO: Assert result structure
    }

    #[test]
    fn single_task_works() {
        let config = serde_yaml::from_str("tasks: {build: {}}").unwrap();
        let graph = TaskGraph::single(&config, "build").expect("failed to build graph");

        assert_eq!(graph.task_count(), 1);
        assert_eq!(graph.all_tasks().collect::<Vec<_>>(), vec!["build"]);
    }

    #[test]
    fn complete_task_works() {
        let config = serde_yaml::from_str("tasks: {build: {}}").unwrap();
        let mut graph = TaskGraph::single(&config, "build").expect("failed to build graph");
        graph.complete_task("build");

        assert_eq!(graph.task_count(), 0);
        assert_eq!(graph.all_tasks().collect::<Vec<_>>(), Vec::<String>::new());
    }

    #[test]
    fn runnable_tasks_works() {
        let config = serde_yaml::from_str(
            r#"
            tasks:
              generate: {}
              lint: {}

              build:
                dependencies:
                - generate
                - lint
            "#,
        )
        .unwrap();

        let graph = TaskGraph::build(&config, "build").expect("failed to build graph");

        assert_eq!(
            graph.runnable_tasks().collect::<Vec<_>>(),
            vec!["generate", "lint"]
        )
    }
}
