use anyhow::{Context, Result};
use std::{fs::File, path::Path};

use crate::{files::FilesReader, tasks::Task, util, FasterError};

pub fn save_outputs<R: FilesReader, P1, P2>(
    workdir: P1,
    task_name: &str,
    task_path: P2,
    current_hash: u64,
    task: &Task,
) -> Result<()>
where
    P1: AsRef<Path>,
    P2: AsRef<Path>,
{
    if task.outputs.is_empty() {
        return Ok(());
    }

    let paths = R::get_paths_from_patterns(&workdir, &task.outputs)?;

    if paths.is_empty() && !task.ignore_missing_outputs {
        return Err(FasterError::NoOutputsFound {
            task: task_name.to_owned(),
        }
        .into());
    }

    let archive_path =
        workdir
            .as_ref()
            .join(task_path)
            .join(format!("{}.{}", current_hash, util::ARCHIVE_SUFFIX));

    let archive_file = File::create(archive_path)
        .with_context(|| format!("failed to create output archive for task `{}`", task_name))?;
    let compressed_writer = snap::write::FrameEncoder::new(archive_file);
    let mut archive = tar::Builder::new(compressed_writer);

    for path in &paths {
        // All patterns we receive here are absolute, so we need to trim
        // the prefix to get the path in the current workdir
        let path_in_workdir = path.strip_prefix(&workdir)?;

        archive
            .append_path_with_name(path, path_in_workdir)
            .with_context(|| format!("failed to append path {:?} to archive", path))?;
    }

    Ok(archive.finish()?)
}

pub fn restore_outputs<P1, P2>(
    workdir: P1,
    task_path: P2,
    current_hash: u64,
    task: &Task,
) -> Result<()>
where
    P1: AsRef<Path>,
    P2: AsRef<Path>,
{
    // Adding/removing outputs will invalidate the hash,
    // which means we can assume there were no defined outputs
    // if there are no outputs in the task config.
    if task.outputs.is_empty() {
        return Ok(());
    }

    let archive_path =
        workdir
            .as_ref()
            .join(task_path)
            .join(format!("{}.{}", current_hash, util::ARCHIVE_SUFFIX));

    if !archive_path.is_file() {
        return Ok(());
    }

    let archive_file = File::open(archive_path)?;
    let compressed_reader = snap::read::FrameDecoder::new(archive_file);
    let mut archive = tar::Archive::new(compressed_reader);

    // Since the output patterns are always relative to the current directory,
    // we can safely unpack cached artifacts to the current directory as well
    Ok(archive.unpack(workdir)?)
}

#[cfg(test)]
mod tests {
    use std::{fs::OpenOptions, path::PathBuf};

    use super::*;
    use crate::{
        files::GlobFilesReader,
        util::{ensure_task_file_path, tests::run_in_temp_dir},
    };

    fn touch_file<P>(path: P)
    where
        P: AsRef<Path>,
    {
        let path = PathBuf::from(path.as_ref());
        std::fs::create_dir_all(path.parent().unwrap()).unwrap();

        OpenOptions::new()
            .create(true)
            .write(true)
            .open(path)
            .unwrap();
    }

    #[test]
    fn archive_and_restore() {
        let files = vec![
            "output/foo/1.txt",
            "output/foo/2.txt",
            "output/foo/3.txt",
            "output/foo/bar/1.txt",
            "output/foo/bar/2.txt",
            "output/foo/bar/3.txt",
            "output/bar/a",
            "output/bar/b",
            "output/bar/c",
        ];

        let workdir = run_in_temp_dir();
        let task_name = "test";

        let task = Task {
            outputs: vec!["output/foo/**/*.txt".to_owned(), "output/bar".to_owned()],
            ..Default::default()
        };

        for file in &files {
            touch_file(workdir.as_ref().join(file));
        }

        let task_path =
            ensure_task_file_path(&workdir, task_name).expect("failed to ensure task path");

        save_outputs::<GlobFilesReader, _, _>(&workdir, task_name, &task_path, 0u64, &task)
            .expect("failed to save outputs");

        let mut archive_path = PathBuf::from(&task_path);
        archive_path.push("0.tar.sz");
        assert!(archive_path.is_file());

        let mut output_dir = PathBuf::from(&workdir.as_ref());
        output_dir.push("output");
        std::fs::remove_dir_all(output_dir).expect("failed to delete output dir");

        // Files should now be deleted
        for file in &files {
            assert!(
                !workdir.as_ref().join(file).is_file(),
                "expected file {:?} to not exist",
                file
            );
        }

        restore_outputs(&workdir.as_ref(), &task_path.as_path(), 0u64, &task)
            .expect("failed to restore outputs");

        // Files should now be restored
        for file in &files {
            assert!(
                workdir.as_ref().join(file).is_file(),
                "expected file {:?} to exist",
                file
            );
        }
    }
}
